import re

from . import docker_resolver, cliutil

import yp.common
from yp import data_model


OBJECT_ID_LEN = 180
IMAGE_TAG_RECOMMENDED_LEN = 25
PREFIX = 'DOCKER'
PREFIX_LEN = len(PREFIX)
HASH_LEN = 6
BANNED_ID_SYMBOLS = re.compile(r'[^a-zA-Z0-9_\.:-]+')
MULTIPLE_HYPHENS = re.compile(r'-+')


def cleanup_id(s):
    return MULTIPLE_HYPHENS.sub('-', BANNED_ID_SYMBOLS.sub('-', s)).strip('-')


def release(client, registry_url, image_name, image_tag, release_type, title=None, description=None):
    try:
        info = docker_resolver.resolve_docker_info(registry_url, image_name, image_tag)
    except Exception as e:
        raise ValueError("Failed to find docker release for {}/{}:{}: {}".format(
            registry_url, image_name, image_tag, e,
        ))

    image_hash = info['hash']
    hash_id = image_hash.rsplit(':', 1)[-1][:6]

    release_type_part = cleanup_id(release_type)
    image_name_part = cleanup_id(image_name)
    image_tag_part = cleanup_id(image_tag)
    if len(image_name_part) + len(image_tag_part) + len(release_type_part) + 4 + HASH_LEN + PREFIX_LEN > OBJECT_ID_LEN:
        # save the end part of image_tag since version is usually at the end
        image_tag_part = image_tag_part[-IMAGE_TAG_RECOMMENDED_LEN:]
        image_name_part = image_name_part[:OBJECT_ID_LEN - HASH_LEN - PREFIX_LEN - 4 - len(image_tag_part) - len(release_type_part)]

    object_id = '{}-{}-{}:{}-{}'.format(PREFIX, release_type_part, image_name_part, image_tag_part, hash_id)

    title = title or '{}:{}'.format(image_name, image_tag)
    description = description or 'Docker image {}:{} ({})'.format(image_name, image_tag, image_hash)

    model = data_model.TRelease()
    model.meta.id = object_id
    model.spec.title = title
    model.spec.description = description
    model.spec.docker.image_name = image_name
    model.spec.docker.image_tag = image_tag
    model.spec.docker.image_hash = image_hash
    model.spec.docker.registry = registry_url
    model.spec.docker.release_type = release_type
    model.spec.docker.release_author = cliutil.get_user()

    try:
        client.create(data_model.OT_RELEASE, model, add_default_user_acl=False)
    except yp.common.YpDuplicateObjectIdError:
        raise ValueError("Docker image for {}/{} ({}) is already released as {!r}".format(image_name, image_tag, image_hash, model.meta.id))

    return model
