import collections
import hashlib
try:
    from urllib.parse import urljoin
except ImportError:
    from urlparse import urljoin


import requests
from requests import adapters

import yp.data_model as data_model


EMPTY_CHECKSUM = 'EMPTY:'

DOCKER_LAYER_URL_PREFIX = 'docker:'
DOCKER_REGISTRY = 'registry.yandex.net'

BASE_DOCKINFO_URL = 'https://dockinfo.yandex-team.ru/api/docker/'
MAX_DOCKINFO_RETRIES = 5
DOCKINFO_TIMEOUT = 30


def is_docker_layer_url(url):
    return url.startswith(DOCKER_LAYER_URL_PREFIX)


def parse_docker_layer_url(url):
    img = url[len(DOCKER_LAYER_URL_PREFIX):]
    img_name, img_tag = img.rsplit(':', 1)
    return img_name, img_tag


def resolve_docker_info(registry_url, image_name, image_tag):
    rel_url = 'resolve?registryUrl={}/{}&tag={}'.format(registry_url, image_name, image_tag)
    with requests.Session() as s:
        s.mount(BASE_DOCKINFO_URL, adapters.HTTPAdapter(max_retries=MAX_DOCKINFO_RETRIES))
        r = s.get(urljoin(BASE_DOCKINFO_URL, rel_url), timeout=DOCKINFO_TIMEOUT)
        r.raise_for_status()
        manifest = r.json()
        return manifest


def resolve_layers_urls(registry_url, image_name, image_tag):
    manifest = resolve_docker_info(registry_url, image_name, image_tag)
    return manifest['layers'][::-1]


def resolve_docker_layers(spec):
    resolved_r = data_model.TResourceGang()
    is_docker_first = False

    layers_lookup = collections.defaultdict(list)
    for n, l in enumerate(spec.resources.layers):
        if not is_docker_layer_url(l.url):
            resolved_l = resolved_r.layers.add()
            resolved_l.CopyFrom(l)
            continue

        if n == 0:
            is_docker_first = True
        elif is_docker_first:
            raise ValueError("Docker layer must be the first one in the layers list")

        img_name, img_tag = parse_docker_layer_url(l.url)
        for url in resolve_layers_urls(DOCKER_REGISTRY, img_name, img_tag):
            resolved_l = resolved_r.layers.add()
            resolved_l.id = '{}-{}'.format(l.id, hashlib.sha1(url).hexdigest())
            resolved_l.url = url
            resolved_l.checksum = EMPTY_CHECKSUM
            layers_lookup[l.id].append(resolved_l.id)
    del spec.resources.layers[:]
    spec.resources.layers.MergeFrom(resolved_r.layers)

    for b in spec.boxes:
        layer_refs = []
        for ref in b.rootfs.layer_refs:
            if ref in layers_lookup:
                resolved_ids = layers_lookup[ref]
                layer_refs.extend(resolved_ids)
            else:
                layer_refs.append(ref)
        del b.rootfs.layer_refs[:]
        b.rootfs.layer_refs.extend(layer_refs)
