import click
import prettytable
import yt_yson_bindings
import yp.common
from yt import yson
from yp import data_model


def get(dr_id, client):
    return client.get(
        object_type=data_model.OT_DYNAMIC_RESOURCE,
        object_id=dr_id
    )


def get_status(cluster, dr_id, client):
    dr = get(dr_id, client)
    spec = prettytable.PrettyTable([
        'Cluster',
        'ResId',
        'SpecRev',
        'PodSet',
    ])

    revs = prettytable.PrettyTable([
        'Revision',
        'Current',
        'Ready',
        'InProgress',
        'Error',
    ])
    current_rev = dr.spec.revision
    for record in sorted(dr.status.revisions, key=lambda item: item.revision):
        revs.add_row([
            record.revision,
            '*' if record.revision == current_rev else '',
            click.style(str(record.ready and record.ready.pod_count or 0), fg='green'),
            click.style(str(record.in_progress and record.in_progress.pod_count or 0), fg='yellow'),
            click.style(str(record.error and record.error.pod_count or 0), fg='red'),
        ])
    spec.add_row([
        cluster,
        dr.meta.id,
        dr.spec.revision,
        dr.meta.pod_set_id,
    ])
    return spec, revs


def remove(dr_id, client):
    return client.remove(
        object_type=data_model.OT_DYNAMIC_RESOURCE,
        object_id=dr_id,
    )


def put(dr, client):
    has_revision = dr.spec and dr.spec.revision

    try:
        a = dr.meta.acl.add()
        a.action = data_model.ACA_ALLOW
        a.permissions.extend([
            data_model.ACP_READ,
            data_model.ACA_WRITE,
        ])
        a.subjects.append('robot-yp-dynresource')

        if not has_revision:
            dr.spec.revision = 1

        return client.create(data_model.OT_DYNAMIC_RESOURCE, dr)
    except yp.common.YpDuplicateObjectIdError:
        pass

    old_dr = client.get(data_model.OT_DYNAMIC_RESOURCE, dr.meta.id)

    if not has_revision and old_dr.spec and old_dr.spec.revision:
        dr.spec.revision = old_dr.spec.revision + 1
    elif not has_revision:
        dr.spec.revision = 1
    client.update(
        data_model.OT_DYNAMIC_RESOURCE,
        dr.meta.id,
        dr
    )
    return dr


def list_objects(clients, user, pod_set, limit):
    rv = prettytable.PrettyTable(['Cluster', 'ID', 'SpecRev', 'PodSet'])
    query = None if pod_set is None else '[/meta/pod_set_id] = "{}"'.format(pod_set)
    for c, client in clients.items():
        drs = client.list(data_model.OT_DYNAMIC_RESOURCE, user=user, query=query, limit=limit)
        for dr in drs:
            rv.add_row([
                c,
                dr.meta.id,
                dr.spec and dr.spec.revision or '-',
                dr.meta.pod_set_id,
            ])
    return rv


def cast_yaml_dict_to_yp_object(d):
    return yt_yson_bindings.loads_proto(yson.dumps(d),
                                        proto_class=data_model.TDynamicResource,
                                        skip_unknown_fields=False)
