import datetime
import errno
import json
import os

import click
import library.python.oauth as lpo
import yp.data_model
from library.python.vault_client import errors as vault_errors
from infra.dctl.src import consts
from infra.dctl.src import version
from infra.dctl.src.lib import docker_resolver
from infra.dctl.src.lib import yputil


def mkdir_p(path):
    """
    :type path: str
    """
    try:
        os.makedirs(path)
    except OSError as e:
        if e.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


def read_file_if_exists(path):
    try:
        with open(path, 'r') as f:
            return f.read(), True
    except IOError as e:
        if e.errno == errno.ENOENT:
            return None, False
        raise


def update_token(token_path):
    """
    :type token_path: str
    :rtype: str
    """
    token_dir = os.path.dirname(token_path)
    mkdir_p(token_dir)

    try:
        token = lpo.get_token(consts.DCTL_CLIENT_ID, consts.DCTL_CLIENT_SECRET, raise_errors=True)
    except Exception as e:
        raise click.ClickException('OAuth token cannot be retrieved from oauth.yandex-team.ru. Error: {}'.format(e))
    if not token:
        raise click.ClickException('OAuth token cannot be retrieved from oauth.yandex-team.ru. '
                                   'OAuth returned empty token.')
    with open(token_path, 'w') as f:
        f.write(token)
    click.echo('OAuth token saved to file {}'.format(token_path), err=True)
    return token


def get_token(token_env, token_path):
    """
    :type token_env: str
    :type token_path: str
    :rtype: str
    """
    token = os.getenv(token_env)
    if token:
        return token
    token, exists = read_file_if_exists(token_path)
    if exists:
        token = token.strip()
        if token:
            return token
    token = update_token(token_path)
    return token


def get_cached_svn_revision(cache_path):
    data, exists = read_file_if_exists(cache_path)
    if not exists:
        return None
    data = json.loads(data)
    version = data['version']
    now_ts = datetime.datetime.utcnow().timestamp()
    if now_ts - data['last_updated_timestamp'] < consts.VERSION_CACHE_TTL:
        return version['svn_revision']


def update_cached_version_data(cache_path, svn_revision):
    now_ts = datetime.datetime.utcnow().timestamp()
    d = {'svn_revision': svn_revision}
    data = {'version': d, 'last_updated_timestamp': now_ts}
    version_dir = os.path.dirname(cache_path)
    mkdir_p(version_dir)
    with open(cache_path, 'w') as f:
        f.write(json.dumps(data))


def get_actual_svn_revision(yp_client):
    cache_path = os.path.expanduser(consts.VERSION_CACHE_PATH)
    try:
        cached_svn_revision = get_cached_svn_revision(cache_path=cache_path)
    except Exception:
        cached_svn_revision = None
    if cached_svn_revision:
        return cached_svn_revision
    r = yp_client.get(object_type=yp.data_model.OT_RELEASE,
                      object_id=consts.VERSION_RELEASE_ID)
    svn_revision = yputil.get_label(r.labels, consts.SVN_REVISION_LABEL_NAME)
    if svn_revision:
        update_cached_version_data(cache_path, svn_revision)
    return svn_revision


def check_version(yp_client):
    current_svn_revision = int(version.SVN_REVISION)
    actual_svn_revision = int(get_actual_svn_revision(yp_client))
    if current_svn_revision < actual_svn_revision:
        click.echo(consts.OUTDATED_VERSION_MESSAGE, err=True)


def is_delegation_token_renewal_needed(vault_client,
                                       yp_tvm_client_id,
                                       secret_uuid,
                                       signature,
                                       token):
    try:
        info = vault_client.get_token_info(token=token).get('token_info')
    except vault_errors.ClientError:
        # If delegation token doesn't look like valid token (e.g. "XXX", or "adsfasdf"),
        # vault respond 'message': 'Invalid delegation token'
        return True
    if info.get('state_name') != 'normal':
        return True
    if str(info.get('tvm_client_id')) != yp_tvm_client_id:
        return True
    token_uuid = info.get('token_uuid')
    if not token_uuid:
        return True
    # Yav doesn't provide signature in get_token_info method, but provide it in
    # list_tokens. We try to find approriate token only in first page if not
    # found just create new one.
    try:
        tokens = vault_client.list_tokens(secret_uuid=secret_uuid, page_size=100)
    except vault_errors.ClientError as e:
        if e.kwargs.get('status') == 'error' and e.kwargs.get('code') == 'access_error':
            raise click.ClickException('cannot renew delegation token of secret "{}": '
                                       'user or provided DCTL_YP_TOKEN has no access to the secret'.format(secret_uuid))
        raise
    for t in tokens:
        if t.get('token_uuid') == token_uuid and t.get('signature') == signature:
            return False
    return True


def create_delegation_token_if_needed(vault_client,
                                      yp_tvm_client_id,
                                      secret_uuid,
                                      signature,
                                      current_token):
    need_renew = is_delegation_token_renewal_needed(vault_client=vault_client,
                                                    yp_tvm_client_id=yp_tvm_client_id,
                                                    secret_uuid=secret_uuid,
                                                    signature=signature,
                                                    token=current_token)
    if not need_renew:
        return current_token, False
    new_token, _ = vault_client.create_token(secret_uuid=secret_uuid,
                                             tvm_client_id=yp_tvm_client_id,
                                             signature=signature)
    return new_token, True


def patch_pod_spec_secrets(vault_client,
                           vault_client_rsa_fallback,
                           pod_spec,
                           ps_id,
                           cluster):
    """
    :type vault_client: library.python.vault_client.VaultClient
    :type vault_client_rsa_fallback: library.python.vault_client.VaultClient
    :type pod_spec: yp.data_model.TPodSpec
    :type ps_id: str
    :type cluster: str
    """
    yp_tvm_client_id = None
    if cluster in consts.PRODUCTION_CLUSTERS or cluster == consts.XDC_PRODUCTION_CLUSTER:
        yp_tvm_client_id = consts.YP_TVM_CLIENT_ID_PRODUCTION
    elif cluster in consts.PRESTABLE_CLUSTERS:
        yp_tvm_client_id = consts.YP_TVM_CLIENT_ID_PRESTABLE
    elif cluster in consts.DEV_CLUSTERS:
        yp_tvm_client_id = consts.YP_TVM_CLIENT_ID_DEV
    else:
        raise click.ClickException("Cannot find YP TVM client id for cluster {}".format(cluster))

    for s in pod_spec.secrets.values():
        try:
            token, _ = create_delegation_token_if_needed(
                vault_client=vault_client,
                yp_tvm_client_id=yp_tvm_client_id,
                secret_uuid=s.secret_id,
                signature=ps_id,
                current_token=s.delegation_token
            )
        except vault_errors.ClientError:
            click.echo("WARNING! Failed to authorize user in Yandex.Vault by "
                       "token. If you use DCTL_YP_TOKEN environment "
                       "variable, please verify this token is valid. If you "
                       "don't use DCTL_YP_TOKEN then try to remove cached "
                       "token: rm ~/.dctl/token. For now fallback to RSA "
                       "authorization in Yandex.Vault.",
                       err=True)
            token, _ = create_delegation_token_if_needed(
                vault_client=vault_client_rsa_fallback,
                yp_tvm_client_id=yp_tvm_client_id,
                secret_uuid=s.secret_id,
                signature=ps_id,
                current_token=s.delegation_token
            )
        s.delegation_token = token


def patch_pod_agent_spec_mutable_workloads(pod_agent_spec):
    if pod_agent_spec.mutable_workloads:
        return
    for w in pod_agent_spec.workloads:
        mw = pod_agent_spec.mutable_workloads.add()
        mw.workload_ref = w.id
        mw.target_state = yp.data_model.EWorkloadTarget_ACTIVE


def patch_stage_spec_before_put(vault_client,
                                vault_client_rsa_fallback,
                                stage_spec,
                                stage_id,
                                cluster,
                                rewrite_delegation_tokens):
    for du_id, du in stage_spec.deploy_units.items():
        if du.WhichOneof('pod_deploy_primitive') == 'replica_set':
            pod_spec = du.replica_set.replica_set_template.pod_template_spec.spec
        else:
            pod_spec = du.multi_cluster_replica_set.replica_set.pod_template_spec.spec
        if rewrite_delegation_tokens:
            ps_id = '{}.{}'.format(stage_id, du_id)
            patch_pod_spec_secrets(vault_client=vault_client,
                                   vault_client_rsa_fallback=vault_client_rsa_fallback,
                                   pod_spec=pod_spec,
                                   ps_id=ps_id,
                                   cluster=cluster)
        patch_pod_agent_spec_mutable_workloads(pod_spec.pod_agent_payload.spec)
        # TODO: do not resolve docker layers in dctl. Now docker layers are
        # resolved by StageCtl, so it is not needed to resolve them here.
        docker_resolver.resolve_docker_layers(pod_spec.pod_agent_payload.spec)


def validate_cluster_or_address(cluster, address):
    """
    either cluster or address must be provided. if cluster and address is not given, use default cluster(xdc)
    :type cluster: Optional[str]
    :type address: Optional[str]
    :rtype: tuple(Optional[str], Optional[str])
    """
    if cluster and address:
        raise click.ClickException('Cannot use --cluster/-c option and --address simultaneously.')
    if not cluster and not address:
        cluster = consts.XDC_PRODUCTION_CLUSTER
    return cluster, address
