import click
import datetime
import prettytable

import yt_yson_bindings
from yt import yson
from yp import data_model

from infra.dctl.src.lib import cliutil


def format_date(timestamp):
    if not timestamp.seconds:
        return None
    return datetime.datetime.fromtimestamp(timestamp.seconds + timestamp.nanos / 10**9).strftime("%Y-%m-%d %H:%M:%S")


def format_interval_belongs(value, l, r):
    belongs_symbol = '∈' if (l <= value <= r) else '∉'
    if isinstance(l, float) or isinstance(r, float):
        return '%s [%.3f, %.3f]' % (belongs_symbol, l, r)
    else:
        return '%s [%s, %s]' % (belongs_symbol, l, r)


def format_replicas(replica_set_spec, replica_set_status):
    replicas_equal = replica_set_status.current_replicas == replica_set_status.desired_replicas
    current_replicas_fg = 'green' if replicas_equal else 'red'
    if replicas_equal:
        message = str(replica_set_status.current_replicas)
    else:
        message = "%s -> %s" % (replica_set_status.current_replicas, replica_set_status.desired_replicas)
    message += ' ' + format_interval_belongs(replica_set_status.current_replicas, replica_set_spec.min_replicas, replica_set_spec.max_replicas)
    return click.style(message, fg=current_replicas_fg)


def format_metric_value(metric_value, lower_bound, upper_bound):
    belongs_to_the_interval = lower_bound <= metric_value <= upper_bound
    metric_value_fg = 'green' if belongs_to_the_interval else 'yellow'
    return click.style("{:.3f} {}".format(
        metric_value,
        format_interval_belongs(metric_value, lower_bound, upper_bound),
    ), fg=metric_value_fg)


def get_status(cluster, hpa_id, client):
    hpa = client.get(
        object_type=data_model.OT_HORIZONTAL_POD_AUTOSCALER,
        object_id=hpa_id
    )

    status = prettytable.PrettyTable([
        'Replicas',
        'MetricValue',
        'LastUpscaleTime',
        'LastDownscaleTime',
        'Status',
    ])

    replica_set_status = hpa.status.replica_set

    if hpa.spec.replica_set.HasField("cpu"):
        metric_value = format_metric_value(replica_set_status.metric_value,
                                           hpa.spec.replica_set.cpu.lower_bound,
                                           hpa.spec.replica_set.cpu.upper_bound)
    elif hpa.spec.replica_set.HasField("cpu_utilization"):
        metric_value = format_metric_value(replica_set_status.metric_value,
                                           hpa.spec.replica_set.cpu_utilization.lower_bound_percent,
                                           hpa.spec.replica_set.cpu_utilization.upper_bound_percent)
    else:
        metric_value = format_metric_value(replica_set_status.metric_value,
                                           hpa.spec.replica_set.custom.lower_bound,
                                           hpa.spec.replica_set.custom.upper_bound)

    status.add_row([
        format_replicas(hpa.spec.replica_set, replica_set_status),
        metric_value,
        format_date(replica_set_status.last_upscale_time) or "No upscale",
        format_date(replica_set_status.last_downscale_time) or "No downscale",
        cliutil.stringify_status_from_conditions(hpa.status),
    ])

    return status


def remove(hpa_id, client):
    return client.remove(
        object_type=data_model.OT_HORIZONTAL_POD_AUTOSCALER,
        object_id=hpa_id,
    )


def put(hpa, client):
    obj_type = data_model.OT_HORIZONTAL_POD_AUTOSCALER
    obj = client.get(object_type=obj_type, object_id=hpa.meta.id, ignore_nonexistent=True)
    if obj is None:
        a = hpa.meta.acl.add()
        a.action = data_model.ACA_ALLOW
        a.permissions.extend([
            data_model.ACP_READ,
            data_model.ACA_WRITE,
        ])
        a.subjects.append('robot-yd-hpa-ctl')

        return client.create(object_type=obj_type, obj=hpa)
    else:
        client.update(object_type=obj_type, object_id=hpa.meta.id, obj=hpa)
    return hpa


def list_objects(clients, user, replica_set, limit):
    rv = prettytable.PrettyTable(['Cluster', 'ID', 'ReplicaSet', 'Replicas', 'Status'])
    query = None if replica_set is None else '[/meta/replica_set_id] = "{}"'.format(replica_set)
    for c, client in clients.items():
        hpas = client.list(data_model.OT_HORIZONTAL_POD_AUTOSCALER, user=user, query=query, limit=limit)
        for hpa in hpas:
            rv.add_row([
                c,
                hpa.meta.id,
                hpa.meta.replica_set_id,
                format_replicas(hpa.spec.replica_set, hpa.status.replica_set),
                cliutil.stringify_status_from_conditions(hpa.status),
            ])
    return rv


def cast_yaml_dict_to_yp_object(d):
    return yt_yson_bindings.loads_proto(yson.dumps(d),
                                        proto_class=data_model.THorizontalPodAutoscaler,
                                        skip_unknown_fields=False)
