import prettytable

import yt.yson as yson
import yt_yson_bindings
import yp.data_model as data_model
from infra.dctl.src.lib import cliutil
from infra.dctl.src.lib import docker_resolver
from infra.dctl.src.lib import helpers
from infra.dctl.src.lib import replica_set as rs


def get_mcrs(mcrs_id, client):
    """
    :type mcrs_id: str
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :rtype: yp.data_model.TMultiClusterReplicaSet
    """
    return client.get(object_type=data_model.OT_MULTI_CLUSTER_REPLICA_SET,
                      object_id=mcrs_id)


def put_mcrs(mcrs, client):
    """
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :type mcrs: yp.data_model.TMultiClusterReplicaSet
    :rtype: str
    """
    assert mcrs.meta.id
    # Initial revision must be equal to 1, it's required by pod agent
    mcrs.spec.pod_template_spec.spec.pod_agent_payload.spec.revision = 1
    mcrs.spec.revision = 1
    agent_spec = mcrs.spec.pod_template_spec.spec.pod_agent_payload.spec
    docker_resolver.resolve_docker_layers(agent_spec)
    helpers.patch_pod_agent_spec_mutable_workloads(agent_spec)
    obj_type = data_model.OT_MULTI_CLUSTER_REPLICA_SET
    obj = client.get(object_type=obj_type, object_id=mcrs.meta.id, ignore_nonexistent=True)
    if obj is None:
        client.create(object_type=obj_type, obj=mcrs)
    else:
        client.update_revision_increment(object_type=obj_type,
                                         object_id=mcrs.meta.id,
                                         obj=mcrs)
    return mcrs.meta.id


def remove_mcrs(mcrs_id, client):
    """
    :type mcrs_id: str
    :type client: YpClient
    """
    client.remove(object_type=data_model.OT_MULTI_CLUSTER_REPLICA_SET,
                  object_id=mcrs_id)


def list_objects(client, cluster, user, limit):
    """
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :type cluster: str
    :type user: str
    :rtype: prettytable.PrettyTable
    """
    rv = prettytable.PrettyTable([
        'Cluster',
        'ID',
        'SpecRev',
        'StatusRev',
        'Status',
        'CurrentProgress',
        'TotalProgress',
        'ReplicaCount'
    ])
    batch = client.list(data_model.OT_MULTI_CLUSTER_REPLICA_SET, user, limit=limit)
    for mcrs in batch:
        s = mcrs.status
        ds = s.multi_cluster_deploy_status
        r = [
            cluster,
            mcrs.meta.id,
            mcrs.spec.revision,
            mcrs.status.revision,
            cliutil.stringify_condition_as_status(s.ready),
            rs.stringify_progress(ds.details.current_revision_progress),
            rs.stringify_progress(ds.details.total_progress),
            sum(c.spec.replica_count for c in mcrs.spec.clusters)
        ]
        rv.add_row(r)
    return rv


def get_status(mcrs_id, client):
    """
    :type mcrs_id: str
    :rtype: (prettytable.PrettyTable, str)
    """
    table = prettytable.PrettyTable(
        ['Cluster', 'Rev', 'RevPods', 'TotalPods', 'CtlStatus']
    )

    mcrs = get_mcrs(mcrs_id, client)
    s = mcrs.status
    clusters = [c.cluster for c in mcrs.spec.clusters]
    for c in clusters:
        d = rs.make_deploy_status_dict(
            deploy_status=s.cluster_deploy_statuses[c],
            current_revision=s.revision
        )
        for n, r in enumerate(d['revisions']):
            if n == 0:
                total_progress = d['total_progress']
                ctl_status = d['ctl_status']
            else:
                total_progress = ''
                ctl_status = ''
            row = [c, r['revision'], r['progress'], total_progress, ctl_status]
            table.add_row(row)

    info = "SpecRev: {}, Status: {}, TotalPods: {}, ReplicaCount: {}".format(
        mcrs.spec.revision,
        cliutil.stringify_condition_as_status(s.ready),
        rs.stringify_progress(s.multi_cluster_deploy_status.details.total_progress),
        sum(c.spec.replica_count for c in mcrs.spec.clusters)
    )
    return table, info


def cast_yaml_dict_to_yp_object(d):
    """
    :type d: dict
    :rtype: yp.data_model.TMultiClusterReplicaSet
    """
    return yt_yson_bindings.loads_proto(yson.dumps(d),
                                        proto_class=data_model.TMultiClusterReplicaSet,
                                        skip_unknown_fields=False)
