import collections
import datetime

import six
import prettytable
import yp.data_model as data_model
from yp_proto.yp.client.api.proto import pod_agent_pb2
import yt.yson as yson
import yt_yson_bindings


def get_pod(pod_id, client):
    """
    :type pod_id: str
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :rtype: yp.data_model.TPod
    """
    return client.get(object_type=data_model.OT_POD,
                      object_id=pod_id)


def remove_pod(pod_id, client):
    """
    :type pod_id: str
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :rtype: yp_proto.yp.client.api.proto.object_service_pb2.TRspRemoveObject
    """
    return client.remove(object_type=data_model.OT_POD,
                         object_id=pod_id)


def put(pod_dto, client):
    """
    :type pod_dto: yp.data_model.TPod
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :rtype: str
    """
    obj_type = data_model.OT_POD
    client.update(object_type=obj_type,
                  object_id=pod_dto.meta.id,
                  obj=pod_dto)

    return pod_dto.meta.id


def get_status_name(s):
    if s.in_progress.status == pod_agent_pb2.EConditionStatus_TRUE:
        return 'InProgress'
    if s.ready.status == pod_agent_pb2.EConditionStatus_TRUE:
        return 'Ready'
    return 'Unknown'


def get_status_last_transition_time(s):
    t1 = s.ready.last_transition_time.seconds
    t2 = s.in_progress.last_transition_time.seconds
    secs = max(t1, t2)
    if not secs:
        return
    return datetime.datetime.fromtimestamp(secs)


def group_statuses_by_revision_id(d):
    rv = {}
    for item in d:
        rv[item.revision] = get_status_name(item)
    return rv


def collect_pod_revisions(pod):
    """
    :type pod: yp_proto.yp.client.api.proto.data_type_pb2.TPod
    :rtype: dict
    """
    revisions = collections.defaultdict(dict)
    s = pod.status.agent.pod_agent_payload.status

    volumes = group_statuses_by_revision_id(s.volumes)
    for rid, status in six.iteritems(volumes):
        r = revisions[rid]
        r['volumes'] = status

    boxes = group_statuses_by_revision_id(s.boxes)
    for rid, status in six.iteritems(boxes):
        r = revisions[rid]
        r['boxes'] = status

    workloads = group_statuses_by_revision_id(s.workloads)
    for rid, status in six.iteritems(workloads):
        r = revisions[rid]
        r['workloads'] = status

    return revisions


def list_objects(rs_id, clients, limit):
    """
    :type rs_id: str
    :type clients: dict{str:YpClient}
    :rtype: prettytable.PrettyTable
    """
    q = '[/meta/pod_set_id] = "{}"'.format(rs_id)
    rows = []
    for c, client in clients.items():
        pods = client.list(data_model.OT_POD, user=None, query=q, limit=limit)
        for p in pods:
            status = p.status.agent.pod_agent_payload.status
            state = data_model.EPodCurrentState.Name(p.status.agent.state)
            if state.startswith('PCS_'):
                state = state[4:]
            spec = p.spec.pod_agent_payload.spec
            revs = collect_pod_revisions(p)
            r = revs.pop(status.revision, {})
            rows.append([c, p.meta.id, p.spec.node_id, spec.revision, status.revision,
                         get_status_name(status), get_status_last_transition_time(status),
                         state, r.get('volumes'), r.get('workloads'), r.get('boxes')])

    rv = prettytable.PrettyTable(['Cluster', 'PodID', 'Node', 'Target', 'Current', 'Pod', 'Time',
                                  'Agent', 'Volumes', 'Workloads', 'Boxes'])
    for r in rows:
        rv.add_row(r)
    return rv


def cast_yaml_dict_to_yp_object(d):
    """
    :type d: dict
    :rtype: yp.data_model.TPod
    """
    return yt_yson_bindings.loads_proto(yson.dumps(d),
                                        proto_class=data_model.TPod,
                                        skip_unknown_fields=False)
