from __future__ import unicode_literals

import click
import collections
import prettytable
import yt.yson as yson
import yt_yson_bindings
import yp.data_model as data_model
from infra.dctl.src import consts


def get(project_id, client):
    """
    :type project_id: str
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :rtype: yp.data_model.TProject
    """
    project = client.get(object_type=data_model.OT_PROJECT,
                         object_id=project_id)
    project.spec.ClearField("account_id")
    return project


def validate(project):
    """
    :type project: yp.data_model.TProject
    """
    if len(project.meta.id) > consts.MAX_PROJECT_ID_LENGTH:
        raise click.ClickException('Project ID "{}" is too long: {} chars, max: {} '
                                   'chars'.format(project.meta.id, len(project.meta.id), consts.MAX_PROJECT_ID_LENGTH))
    if project.spec.account_id != project.meta.account_id and project.spec.account_id != "":
        raise click.ClickException('/meta/account_id and /spec/account_id are not equal, please fill them with same value or clear field /spec/account_id')


def put(project, client):
    """
    :type project: yp.data_model.TProject
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :rtype: str
    """

    obj_type = data_model.OT_PROJECT
    obj = client.get(object_type=obj_type, object_id=project.meta.id, ignore_nonexistent=True)
    if obj is None:
        client.create(object_type=obj_type, obj=project, create_with_acl=False, add_default_user_acl=False)
    else:
        meta_fields = {}
        if project.meta.account_id != obj.meta.account_id:
            meta_fields = {'/meta/account_id': project.meta.account_id}
        client.update(object_type=obj_type,
                      object_id=project.meta.id,
                      obj=project,
                      path_to_value_specific_fields=meta_fields)
    return project.meta.id


def remove(project_id, client):
    """
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :type project_id: str
    """
    return client.remove(object_type=data_model.OT_PROJECT,
                         object_id=project_id)


def cast_yaml_dict_to_yp_object(d):
    """
    :type d: dict
    :rtype: yp.data_model.TProject
    """
    return yt_yson_bindings.loads_proto(yson.dumps(d),
                                        proto_class=data_model.TProject,
                                        skip_unknown_fields=False)


def list_objects(client, cluster, user, limit):
    """
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :type cluster: str
    :type user: str
    :rtype: prettytable.PrettyTable
    """
    rv = prettytable.PrettyTable(['Cluster', 'ID', 'Stages'])
    stages_batch = client.list(object_type=data_model.OT_STAGE, user=user, limit=limit)
    project_to_stages_count = collections.defaultdict(int)
    for stage in stages_batch:
        project_to_stages_count[stage.meta.project_id] += 1

    projects_batch = client.list(object_type=data_model.OT_PROJECT, user=user, limit=limit)
    for project in projects_batch:
        stages_count = project_to_stages_count[project.meta.id]
        r = [cluster, project.meta.id, stages_count]
        rv.add_row(r)
    return rv
