import click
import prettytable
from yp import data_model


def get_progress_str(progress):
    status = click.style('Pending', fg='blue')
    if progress.in_progress.status == data_model.CS_TRUE:
        status = click.style('InProgress', fg='yellow')
    elif progress.closed.status == data_model.CS_TRUE:
        status = click.style('Closed', fg='green')

    return status


def get_status(
    release_id,
    client,
):
    r = client.get(
        object_type=data_model.OT_RELEASE,
        object_id=release_id,
    )

    preamble = '%s\n%s' % (r.spec.title, r.spec.description)

    if r.spec.sandbox:
        spec = prettytable.PrettyTable([
            'TaskId',
            'TaskType',
            'TaskAuthor',
            'ReleaseAuthor',
            'ReleaseType',
        ])
        spec.add_row([
            r.spec.sandbox.task_id,
            r.spec.sandbox.task_type,
            r.spec.sandbox.task_author,
            r.spec.sandbox.release_author,
            r.spec.sandbox.release_type,
        ])
    else:
        spec = prettytable.PrettyTable([
            'ImageName',
            'ImageTag',
            'ImageHash',
            'ReleaseAuthor',
            'ReleaseType',
        ])
        spec.add_row([
            r.spec.docker.image_name,
            r.spec.docker.image_tag,
            r.spec.docker.image_hash,
            r.spec.docker.release_author,
            r.spec.docker.release_type,
        ])

    status = prettytable.PrettyTable([
        'Status',
        'Finished',
    ])
    status.add_row([
        get_progress_str(r.status.progress),
        'FINISHED' if r.status.processing.finished.status == data_model.CS_TRUE else '',
    ])

    return preamble, spec, status


def list_objects(
    client,
    release_type,
    release_author,
    sandbox_task_type,
    docker_image,
    limit,
):
    query_pieces = []
    if sandbox_task_type:
        query_pieces.append('[/spec/sandbox/task_type] = "%s"' % (sandbox_task_type,))
    if docker_image:
        query_pieces.append('[/spec/docker/image_name] = "%s"' % (docker_image,))
    if release_type:
        query_pieces.append('([/spec/docker/release_type] = "%s" OR [/spec/sandbox/release_type] = "%s")' % (release_type, release_type))
    if release_author:
        query_pieces.append('([/spec/docker/release_author] = "%s" OR [/spec/sandbox/release_author] = "%s")' % (release_author, release_author))

    query = ' AND '.join(query_pieces) or None

    table = prettytable.PrettyTable(['ID', 'SandboxTask', 'DockerImage', 'DockerTag', 'Author', 'Status'])

    for r in client.list(
        object_type=data_model.OT_RELEASE,
        limit=limit,
        user=None,
        query=query,
    ):
        table.add_row([
            r.meta.id,
            r.spec.sandbox and r.spec.sandbox.task_type and ('%s (%s)' % (r.spec.sandbox.task_type, r.spec.sandbox.task_id)) or '',
            r.spec.docker and r.spec.docker.image_name or '',
            r.spec.docker and r.spec.docker.image_tag or '',
            r.spec.sandbox and r.spec.sandbox.release_author or r.spec.docker.release_author or '',
            get_progress_str(r.status.progress),
        ])

    return table
