import prettytable
import yt_yson_bindings
from yt import yson
from yp import data_model


def put(rr, client):
    obj_type = data_model.OT_RELEASE_RULE
    obj = client.get(object_type=obj_type, object_id=rr.meta.id, ignore_nonexistent=True)
    if obj is None:
        return client.create(object_type=obj_type, obj=rr)
    else:
        client.update(object_type=obj_type, object_id=rr.meta.id, obj=rr)
        return rr.meta.id


def remove(release_rule_id, client):
    return client.remove(
        object_type=data_model.OT_RELEASE_RULE,
        object_id=release_rule_id,
    )


def cast_yaml_dict_to_yp_object(d):
    return yt_yson_bindings.loads_proto(yson.dumps(d),
                                        proto_class=data_model.TReleaseRule,
                                        skip_unknown_fields=False)


def list_objects(
    client,
    user,
    stage,
    recipe_id,
    sandbox_task_type,
    docker_image,
    limit,
):
    query_pieces = []
    if stage:
        query_pieces.append('[/meta/stage_id] = "%s"' % (stage,))
    if recipe_id:
        query_pieces.append('[/spec/recipe_id] = "%s"' % (recipe_id,))
    if sandbox_task_type:
        query_pieces.append('[/spec/sandbox/task_type] = "%s"' % (sandbox_task_type,))
    if docker_image:
        query_pieces.append('[/spec/docker/image_name] = "%s"' % (docker_image,))

    query = ' AND '.join(query_pieces) or None

    table = prettytable.PrettyTable(['ID', 'Stage', 'SandboxTask', 'Docker', 'RecipeId', 'AutoCommit'])

    for rr in client.list(
        object_type=data_model.OT_RELEASE_RULE,
        user=user,
        limit=limit,
        query=query,
    ):
        try:
            policy = data_model.TAutoCommitPolicy.EType.Name(rr.spec.auto_commit_policy.type)
        except ValueError:
            policy = 'Unknown'

        table.add_row([
            rr.meta.id,
            rr.meta.stage_id,
            rr.spec.sandbox and rr.spec.sandbox.task_type or '',
            rr.spec.docker and rr.spec.docker.image_name or '',
            rr.spec.recipe_id,
            policy,
        ])

    return table
