import click
import prettytable

import yt.yson as yson
import yt_yson_bindings
import yp.data_model as data_model
from infra.dctl.src.lib import docker_resolver
from infra.dctl.src.lib import helpers
from infra.dctl.src.lib import cliutil


def get_rs(rs_id, client):
    """
    :type rs_id: str
    :type client: YpClient
    :rtype: yp.data_model.TReplicaSet
    """
    return client.get(object_type=data_model.OT_REPLICA_SET,
                      object_id=rs_id)


def put_rs(rs, client):
    """
    :type rs: yp.data_model.TReplicaSet
    :type client: YpClient
    :rtype: str
    """
    # Initial revision must be equal to 1, it's required by pod agent
    rs.spec.pod_template_spec.spec.pod_agent_payload.spec.revision = 1
    rs.spec.revision_id = '1'
    agent_spec = rs.spec.pod_template_spec.spec.pod_agent_payload.spec
    docker_resolver.resolve_docker_layers(agent_spec)
    helpers.patch_pod_agent_spec_mutable_workloads(agent_spec)
    obj_type = data_model.OT_REPLICA_SET
    obj = client.get(object_type=obj_type, object_id=rs.meta.id, ignore_nonexistent=True)
    if obj is None:
        rs = client.create(object_type=obj_type, obj=rs)
    else:
        client.update_replica_set(rs.meta.id, rs)
    return rs.meta.id


def remove_rs(rs_id, client):
    """
    :type rs_id: str
    :type client: YpClient
    :rtype: yp_proto.yp.client.api.proto.object_service_pb2.TRspRemoveObject
    """
    return client.remove(object_type=data_model.OT_REPLICA_SET,
                         object_id=rs_id)


def list_objects(clients, user, limit):
    """
    :type clients: dict{str:YpClient}
    :type user: str
    :rtype: prettytable.PrettyTable
    """
    rv = prettytable.PrettyTable([
        'Cluster',
        'ID',
        'SpecRev',
        'StatusRev',
        'Status',
        'CurrentProgress',
        'TotalProgress',
        'ReplicaCount'
    ])
    for c, client in clients.items():
        batch = client.list(data_model.OT_REPLICA_SET, user, limit=limit)
        for rs in batch:
            s = rs.status
            ds = s.deploy_status
            r = [
                c,
                rs.meta.id,
                rs.spec.revision_id,
                rs.status.revision_id,
                cliutil.stringify_condition_as_status(s.ready_condition),
                stringify_progress(ds.details.current_revision_progress),
                stringify_progress(ds.details.total_progress),
                rs.spec.replica_count
            ]
            rv.add_row(r)
    return rv


def stringify_progress(p):
    return '{}/{}/{}'.format(click.style(str(p.pods_ready), fg='green'),
                             click.style(str(p.pods_in_progress), fg='blue'),
                             p.pods_total)


def make_revision_dict(progress, revision, is_current):
    rv = {'is_current': is_current}
    if is_current:
        rv['revision'] = click.style(str(revision), fg='green')
    else:
        rv['revision'] = str(revision)
    rv['progress'] = stringify_progress(progress)
    return rv


def make_deploy_status_dict(deploy_status, current_revision):
    details = deploy_status.details
    rv = {
        'revisions': [],
        'ctl_status': cliutil.stringify_condition_as_succeeded(
            details.controller_status.last_attempt.succeeded
        ),
        'total_progress': stringify_progress(details.total_progress)
    }

    d = make_revision_dict(
        progress=deploy_status.details.current_revision_progress,
        revision=current_revision,
        is_current=True
    )
    rv['revisions'].append(d)

    for r, progress in sorted(deploy_status.details.revisions.items()):
        if r == int(current_revision):
            continue
        d = make_revision_dict(progress=progress,
                               revision=r,
                               is_current=False)
        rv['revisions'].append(d)
    return rv


def get_status(rs_id, client):
    """
    :type rs_id: str
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :rtype: tuple(prettytable.PrettyTable, str)
    """
    table = prettytable.PrettyTable(['Rev', 'RevPods', 'CtlStatus'])
    rs = get_rs(rs_id, client)
    ds = rs.status.deploy_status
    d = make_deploy_status_dict(deploy_status=ds,
                                current_revision=rs.status.revision_id)

    for n, r in enumerate(d['revisions']):
        if n == 0:
            ctl_status = d['ctl_status']
        else:
            ctl_status = ''
        row = [r['revision'], r['progress'], ctl_status]
        table.add_row(row)

    info = "SpecRev: {}, Status: {}, TotalPods: {}, ReplicaCount: {}".format(
        rs.spec.revision_id,
        cliutil.stringify_condition_as_status(rs.status.ready_condition),
        d['total_progress'],
        rs.spec.replica_count
    )
    return table, info


def cast_yaml_dict_to_yp_object(d):
    """
    :type d: dict
    :rtype: yp.data_model.TReplicaSet
    """
    return yt_yson_bindings.loads_proto(yson.dumps(d),
                                        proto_class=data_model.TReplicaSet,
                                        skip_unknown_fields=False)
