import click
import prettytable
import yt_yson_bindings
import yp.common
from yt import yson
from yp import data_model


def get(rc_id, client):
    return client.get(
        object_type=data_model.OT_RESOURCE_CACHE,
        object_id=rc_id
    )


def get_status(cluster, rc_id, client):
    rc = get(rc_id, client)
    spec = prettytable.PrettyTable([
        'Cluster',
        'ResId',
        'SpecRev',
        'PodSet',
    ])

    spec.add_row([
        cluster,
        rc.meta.id,
        rc.spec.revision,
        rc.meta.pod_set_id,
    ])

    statuses = []
    for cached_resource_status in sorted(rc.status.cached_resource_status, key=lambda item: item.id):
        status = prettytable.PrettyTable([
            'Revision',
            'Current',
            'Ready',
            'InProgress',
            'Failed',
        ])
        current_rev = rc.spec.revision
        for record in cached_resource_status.revisions:
            status.add_row([
                record.revision,
                '*' if record.revision == current_rev else '',
                click.style(str(record.ready and record.ready.pod_count or 0), fg='green'),
                click.style(str(record.in_progress and record.in_progress.pod_count or 0), fg='yellow'),
                click.style(str(record.failed and record.failed.pod_count or 0), fg='red'),
            ])
        statuses.append((cached_resource_status.id, status))

    return spec, statuses


def remove(rc_id, client):
    return client.remove(
        object_type=data_model.OT_RESOURCE_CACHE,
        object_id=rc_id,
    )


def put(rc, client):
    has_revision = rc.spec and rc.spec.revision

    try:
        if not has_revision:
            rc.spec.revision = 1

        return client.create(data_model.OT_RESOURCE_CACHE, rc)
    except yp.common.YpDuplicateObjectIdError:
        pass

    old_rc = client.get(data_model.OT_RESOURCE_CACHE, rc.meta.id)

    if not has_revision and old_rc.spec and old_rc.spec.revision:
        rc.spec.revision = old_rc.spec.revision + 1
    elif not has_revision:
        rc.spec.revision = 1
    client.update(
        data_model.OT_RESOURCE_CACHE,
        rc.meta.id,
        rc
    )
    return rc


def list_objects(clients, user, pod_set, limit):
    rv = prettytable.PrettyTable(['Cluster', 'ID', 'SpecRev', 'PodSet'])
    query = None if pod_set is None else '[/meta/pod_set_id] = "{}"'.format(pod_set)
    for c, client in clients.items():
        rcs = client.list(data_model.OT_RESOURCE_CACHE, user=user, query=query, limit=limit)
        for rc in rcs:
            rv.add_row([
                c,
                rc.meta.id,
                rc.spec and rc.spec.revision or '-',
                rc.meta.pod_set_id,
            ])
    return rv


def cast_yaml_dict_to_yp_object(d):
    return yt_yson_bindings.loads_proto(yson.dumps(d),
                                        proto_class=data_model.TResourceCache,
                                        skip_unknown_fields=False)
