from __future__ import unicode_literals
import click
import uuid

import yp.data_model
import yt.yson as yson
import yt_yson_bindings
from infra.dctl.src.lib import cliutil


def make_draft_from_stage(client, stage):
    """
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :type stage: yp.data_model.TStage
    """
    if stage.meta.account_id == '':
        raise click.ClickException('Empty account_id is not yet supported for stage_draft, '
                                   'it will be available after DEPLOY-3899. Please fill /meta/account_id field in yaml file that you published.')

    draft = yp.data_model.TStageDraft()
    draft.meta.id = str(uuid.uuid4())
    draft.meta.stage_id = stage.meta.id
    draft.spec.revision = 1
    draft.spec.stage_spec.CopyFrom(stage.spec)
    draft.spec.stage_spec.account_id = stage.meta.account_id
    draft.spec.stage_spec.ClearField('revision')
    for du in draft.spec.stage_spec.deploy_units.values():
        du.ClearField('revision')
    cur_stage = client.get(object_type=yp.data_model.OT_STAGE,
                           object_id=draft.meta.stage_id)
    draft.spec.stage_revision = cur_stage.spec.revision
    cliutil.clear_not_initializable_fields(draft)
    return draft


def make_deploy_ticket_from_draft(draft, title):
    """
    :type draft: yp.data_model.TStageDraft
    :type title: str
    :rtype: yp.data_model.TDeployTicket
    """
    ticket = yp.data_model.TDeployTicket()
    ticket.meta.id = '{}-{}'.format(draft.meta.id, draft.spec.revision)
    ticket.meta.stage_id = draft.meta.stage_id
    ticket.spec.source_type = yp.data_model.TDeployTicketSpec.STAGE_DRAFT
    ticket.spec.title = title
    ticket.spec.description = draft.spec.stage_spec.revision_info.description
    ticket.spec.stage_draft_id = draft.meta.id
    ticket.spec.stage_draft_revision = draft.spec.revision
    return ticket


def get(client, draft_id):
    """
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :type draft_id: str
    :rtype: yp.data_model.TStageDraft
    """
    return client.get(object_type=yp.data_model.OT_STAGE_DRAFT,
                      object_id=draft_id)


def create_draft(client, stage):
    """
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :type stage: yp.data_model.TStage
    :rtype: str
    """
    draft = make_draft_from_stage(client, stage)
    client.create(object_type=yp.data_model.OT_STAGE_DRAFT, obj=draft)
    return draft.meta.id


def create_deploy_ticket(client, draft_id, title):
    """
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :type draft_id: str
    :type title: str
    :rtype: str
    """
    draft = client.get(object_type=yp.data_model.OT_STAGE_DRAFT,
                       object_id=draft_id)
    ticket = make_deploy_ticket_from_draft(draft, title)
    client.create(object_type=yp.data_model.OT_DEPLOY_TICKET, obj=ticket)
    return draft.meta.id


def create_draft_with_deploy_ticket(client, stage, title):
    """
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :type draft: yp.data_model.TStageDraft
    :type title: str
    :rtype: list[str]
    """
    draft = make_draft_from_stage(client, stage)
    ticket = make_deploy_ticket_from_draft(draft, title)
    objs = [(draft, yp.data_model.OT_STAGE_DRAFT),
            (ticket, yp.data_model.OT_DEPLOY_TICKET)]
    return client.create_objects(object_with_type_pairs=objs)


def put(client, draft):
    """
    :type client: infra.dctl.src.lib.yp_client.YpClient
    :type draft: yp.data_model.TStageDraft
    :rtype: str
    """
    if not draft.meta.id:
        return create_draft(client, draft)

    cur = client.get(object_type=yp.data_model.OT_STAGE_DRAFT,
                     object_id=draft.meta.id,
                     ignore_nonexistent=True)
    if cur is None:
        return create_draft(client, draft)

    draft.spec.revision = cur.spec.revision + 1
    meta_fields = {}
    if draft.meta.account_id != cur.meta.account_id:
        meta_fields = {'/meta/account_id': draft.meta.account_id}
    client.update(object_type=yp.data_model.OT_STAGE_DRAFT,
                  object_id=draft.meta.id,
                  obj=draft,
                  path_to_value_specific_fields=meta_fields)
    return draft.meta.id


def cast_yaml_dict_to_yp_object(d):
    """
    :type d: dict
    :rtype: yp.data_model.TStageDraft
    """
    return yt_yson_bindings.loads_proto(yson.dumps(d),
                                        proto_class=yp.data_model.TStageDraft,
                                        skip_unknown_fields=False)
