import time

import click
import ujson
import urllib3
import requests
import prettytable
try:
    from urllib.parse import urljoin
except ImportError:
    from urlparse import urljoin

import library.python.ssh_sign as lps

from infra.dctl.src import consts


MAX_QNOTIFIER_RETRIES = 5
QNOTIFIER_TIMEOUT = 20
QNOTIFIER = 'https://qnotifier.yandex-team.ru/'


def setting_to_marker(setting):
    if setting is None:
        return '*'
    elif setting == "true":
        return '+'
    else:
        return '-'


def fetch(user, method, url, headers=None, data=None, need_auth=False):
    headers = headers or {}
    with requests.Session() as s:
        retry_adapter = requests.adapters.HTTPAdapter(
            max_retries=urllib3.Retry(total=MAX_QNOTIFIER_RETRIES, status_forcelist={429, 500}),
        )
        s.mount('https://', retry_adapter)
        if need_auth:
            ts = int(time.time())
            sign_string = '%d%s%s' % (ts, consts.QNOTIFIER_CLIENT_ID, user)

            signatures = [
                'SSH {} {} {}'.format(user, ts, signature.sign.decode())
                for signature in lps.sign(sign_string)
            ]
        else:
            signatures = ['']

        for signature in signatures:
            if signature:
                headers['Authorization'] = signature
            else:
                headers.pop('Authorization', None)

            response = s.request(method, url, headers=headers, data=data)
            if response.status_code == 401:
                continue

            if response.status_code == 204:
                return

            if not response.ok:
                raise click.ClickException("Failed to fetch data: {}".format(response.text))

            return response.json()
        else:
            raise click.ClickException("Failed to fetch data: no valid ssh keys found for {!r}".format(user))


def list_subscriptions(user):
    result = fetch(user, "GET", urljoin(QNOTIFIER, "subscriptions/{}".format(user)))
    table = prettytable.PrettyTable([
        'Project',
        'Stage ID',
        # 'Stage UUID',  # FIXME uuids are temporary not reliable and cannot be used until YP-2230
        'Email',
        'Telegram',
    ])

    for subscription in result['subscriptions']:
        tags = subscription['tags']
        if 'ya.deploy' not in tags:
            continue
        project = ''
        stage_id = ''
        # stage_uuid = ''
        for tag in tags:
            if tag.startswith('stage:id:'):
                stage_id = tag[len('stage:id:'):]
            # elif tag.startswith('stage:uuid:'):
            #     stage_uuid = tag[len('stage:uuid:'):]
            elif tag.startswith('project:id:'):
                project = tag[len('project:id:'):]

        email_enabled = setting_to_marker(subscription['options'].get('email', {}).get('enabled'))
        telegram_enabled = setting_to_marker(subscription['options'].get('telegram', {}).get('enabled'))
        table.add_row([
            project,
            stage_id,
            # stage_uuid,
            email_enabled,
            telegram_enabled
        ])

    return table


def subscribe(user, stage_id, enable_email=None, enable_telegram=None):
    headers = {'Content-Type': 'application/json'}
    data = {
        'tags': ['ya.deploy', 'stage:id:{}'.format(stage_id)],
        'options': {},
    }
    if enable_email is not None:
        data['options']['email'] = {'enabled': str(enable_email).lower()}
    if enable_telegram is not None:
        data['options']['telegram'] = {'enabled': str(enable_telegram).lower()}

    fetch(
        user, "POST", urljoin(QNOTIFIER, "subscriptions/{}".format(user)),
        headers=headers,
        data=ujson.dumps(data),
        need_auth=True,
    )


def unsubscribe(user, stage_id):
    headers = {'Content-Type': 'application/json'}
    data = {
        'tags': ['ya.deploy', 'stage:id:{}'.format(stage_id)],
    }

    fetch(
        user, "DELETE", urljoin(QNOTIFIER, "subscriptions/{}".format(user)),
        headers=headers,
        data=ujson.dumps(data),
        need_auth=True,
    )
