import yt.yson as yson
import yp.client
import yp.common
import yp.data_model as data_model
import yt_yson_bindings
from infra.dctl.src import consts
from infra.dctl.src.lib import cliutil
from yp_proto.yp.client.api.proto import object_service_pb2


class YpClient(object):
    def __init__(self, url, token, user=None, enable_ssl=True):
        self.user = user
        if user is None:
            self.user = cliutil.get_user()

        self.client = yp.client.YpClient(
            address=url,
            config={
                'token': token,
                'grpc_channel_options': {
                    'max_receive_message_length': 8000000
                },
                'enable_ssl': enable_ssl,
            },
        )
        self.stub = self.client.create_grpc_object_stub()
        self.default_acl_permissions = {data_model.ACP_READ,
                                        data_model.ACA_WRITE,
                                        data_model.ACA_CREATE,
                                        data_model.ACA_SSH_ACCESS,
                                        data_model.ACA_ROOT_SSH_ACCESS,
                                        data_model.ACP_READ_SECRETS}
        self.default_deploy_robot_login = 'robot-drug-deploy'

    @staticmethod
    def dict_to_protobuf(object_type, object_dict):
        return yp.common.dict_to_protobuf(object_dict, consts.object_classes[object_type])

    def _get(self, object_type, object_id,
             selectors=None, timestamp=None, fetch_timestamps=False,
             payload_format=None, ignore_nonexistent=False):

        req = object_service_pb2.TReqGetObject()
        req.object_type = object_type
        req.object_id = object_id
        selectors = selectors or ['']
        req.selector.paths.extend(selectors)
        if timestamp:
            req.timestamp = timestamp
        if fetch_timestamps:
            req.options.fetch_timestamps = fetch_timestamps
        if payload_format:
            req.format = payload_format
        if ignore_nonexistent:
            req.options.ignore_nonexistent = True
        resp = self.stub.GetObject(req)
        return resp.result

    def _add_default_user_acl(self, obj, permissions):
        try:
            permissions = sorted(permissions)
            for a in obj.meta.acl:
                if sorted(a.permissions) == permissions and a.action == data_model.ACA_ALLOW:
                    if self.user in a.subjects:
                        return
                    else:
                        a.subjects.append(self.user)
                        return
            a = obj.meta.acl.add()
            a.action = data_model.ACA_ALLOW
            a.permissions.extend(permissions)
            a.subjects.append(self.user)
        except Exception:
            pass

    def make_default_specific_fields_to_update(self, obj):
        self._add_default_user_acl(obj, self.default_acl_permissions)

        acl = []
        for entry in obj.meta.acl:
            acl.append(yp.common.protobuf_to_dict(entry))
        return {
            '/meta/acl': acl,
            '/meta/account_id': obj.meta.account_id,
        }

    def _add_deploy_robot_acl(self, obj):
        try:
            a = obj.meta.acl.add()
            a.action = data_model.ACA_ALLOW
            a.permissions.extend(self.default_acl_permissions)
            a.subjects.append(self.default_deploy_robot_login)
        except Exception:
            pass

    def get_dict(self, object_type, object_id, selectors=None):
        r = self._get(object_type=object_type,
                      object_id=object_id,
                      selectors=selectors)
        return yt_yson_bindings.loads(r.values[0])

    def get(self, object_type, object_id, selectors=None, ignore_nonexistent=False):
        r = self._get(object_type=object_type,
                      object_id=object_id,
                      selectors=selectors,
                      ignore_nonexistent=ignore_nonexistent)
        if not r.values:
            return None
        return yt_yson_bindings.loads_proto(r.values[0],
                                            consts.object_classes[object_type],
                                            skip_unknown_fields=True)

    def _select_objects(self, object_type, selector, query, limit, continuation_token):
        """
        :type object_type: yp.data_model.EObjectType
        :type selector: str
        :type query: Optional[str]
        :type limit: int
        :type continuation_token: str
        :rtype: yp_proto.yp.client.api.proto.object_service_pb2.TRspSelectObjects()
        """
        req = object_service_pb2.TReqSelectObjects()
        req.object_type = object_type
        # Limit is important!
        # YP preserves objects order only if limit is given
        req.options.limit = limit
        req.selector.paths.append(selector)
        if query is not None:
            req.filter.query = query
        if continuation_token is not None:
            req.options.continuation_token = continuation_token
        return self.stub.SelectObjects(req)

    def _get_allowed_object_ids(self, object_ids, object_type, user):
        """
        :type object_ids: List[str]
        :type object_type: yp.data_model.EObjectType
        :type user: str
        :rtype List[str]
        """
        req = object_service_pb2.TReqCheckObjectPermissions()
        for object_id in object_ids:
            sub = req.subrequests.add()
            sub.object_id = object_id
            sub.object_type = object_type
            sub.subject_id = user
            sub.permission = data_model.ACA_WRITE
        resp = self.stub.CheckObjectPermissions(req)
        allowed_ids = []
        for i, r in enumerate(resp.subresponses):
            if r.action == data_model.ACA_ALLOW:
                allowed_ids.append(object_ids[i])
        return allowed_ids

    def _filter_user_objects(self, object_type, user, query, limit, page_size=2000):
        """
        :type object_type: yp.data_model.EObjectType
        :type user: str
        :type query: str
        :type limit: int
        :type page_size: int
        :rtype List[str]
        """
        object_ids = []
        continuation_token = None
        while len(object_ids) < limit:
            selected_obj_rsp = self._select_objects(object_type, '/meta/id', query, page_size, continuation_token)
            continuation_token = selected_obj_rsp.continuation_token
            selected_ids = [yt_yson_bindings.loads(r.values[0]) for r in selected_obj_rsp.results]
            allowed_ids = self._get_allowed_object_ids(selected_ids, object_type, user)
            object_ids.extend(allowed_ids)
            if len(selected_obj_rsp.results) < page_size:
                break
        return object_ids

    def _select_all_user_objects(self, object_type, user, limit):
        """
        :type object_type: yp.data_model.EObjectType
        :type user: str
        :type limit: int
        :rtype List[str]
        """
        req = object_service_pb2.TReqGetUserAccessAllowedTo()
        sub = req.subrequests.add()
        sub.object_type = object_type
        sub.permission = data_model.ACA_WRITE
        sub.limit = limit
        sub.user_id = user
        resp = self.stub.GetUserAccessAllowedTo(req)
        object_ids = resp.subresponses[0].object_ids
        return object_ids

    def list(self, object_type, user, query=None, limit=None):
        limit = max(0, min(limit or 100, 10000))
        if not user:
            resp = self._select_objects(object_type, '', query, limit, None)
            objects = []
            for r in resp.results:
                obj = yt_yson_bindings.loads_proto(r.values[0], consts.object_classes[object_type], skip_unknown_fields=True)
                objects.append(obj)
            return objects

        limit = min(limit, 1000)
        # if query use TReqCheckObjectPermissions as GetUserAccessAllowedTo doesn't support all filters
        # if no query use GetUserAccessAllowedTo to avoid multiple requests and pagination
        if query:
            object_ids = self._filter_user_objects(object_type, user, query, limit)
        else:
            object_ids = self._select_all_user_objects(object_type, user, limit)

        objects = []
        if not object_ids:
            return objects
        req = object_service_pb2.TReqGetObjects()
        req.object_type = object_type
        req.selector.paths.append('')
        for object_id in object_ids:
            sub = req.subrequests.add()
            sub.object_id = object_id
        resp = self.stub.GetObjects(req)
        for sub in resp.subresponses:
            obj = yt_yson_bindings.loads_proto(sub.result.values[0],
                                               consts.object_classes[object_type],
                                               skip_unknown_fields=True)
            objects.append(obj)
        return objects

    def list_endpoints(self, es_id, limit=None):
        limit = max(0, min(limit or 100, 10000))
        # We had to create separate method for endpoints because YP cannot fetch
        # endpoint object as a whole: "Attribute /status cannot be fetched", so we fetch /spec only
        req = object_service_pb2.TReqSelectObjects()
        req.object_type = data_model.OT_ENDPOINT
        # Limit is important!
        # YP preserves objects order only if limit is given
        req.limit.value = limit
        req.selector.paths.append('/meta/id')
        req.selector.paths.append('/spec')
        req.selector.paths.append('/status')
        req.filter.query = '[/meta/endpoint_set_id] = "{}"'.format(es_id)
        resp = self.stub.SelectObjects(req)
        objects = []
        for r in resp.results:
            spec = yt_yson_bindings.loads_proto(r.values[1],
                                                data_model.TEndpointSpec,
                                                skip_unknown_fields=True)
            status = yt_yson_bindings.loads_proto(r.values[2],
                                                  data_model.TEndpointStatus,
                                                  skip_unknown_fields=True)
            obj = data_model.TEndpoint()
            obj.meta.id = r.values[0]
            obj.spec.CopyFrom(spec)
            obj.status.CopyFrom(status)
            objects.append(obj)
        return objects

    def create(self, object_type, obj, create_with_acl=True, add_default_user_acl=True):
        cliutil.clear_not_initializable_fields(obj)
        if create_with_acl:
            if add_default_user_acl:
                self._add_default_user_acl(obj, self.default_acl_permissions)
        else:
            obj.meta.ClearField("acl")
            self._add_deploy_robot_acl(obj)

        r = object_service_pb2.TReqCreateObject(
            object_type=object_type,
            attributes=yt_yson_bindings.dumps_proto(obj)
        )
        rsp = self.stub.CreateObject(r)
        obj.meta.id = rsp.object_id
        return obj

    def create_objects(self, object_with_type_pairs):
        req = object_service_pb2.TReqCreateObjects()
        for obj, obj_type in object_with_type_pairs:
            subreq = req.subrequests.add()
            subreq.object_type = obj_type
            subreq.attributes = yt_yson_bindings.dumps_proto(obj)
        rsp = self.stub.CreateObjects(req)
        return [subrsp.object_id for subrsp in rsp.subresponses]

    def update(self, object_type, object_id, obj, path_to_value_specific_fields=None, timestamps=None):
        if path_to_value_specific_fields is None:
            path_to_value_specific_fields = self.make_default_specific_fields_to_update(obj)
        timestamps = timestamps or {}
        req = object_service_pb2.TReqUpdateObject()
        req.object_type = object_type
        req.object_id = object_id

        upd = req.set_updates.add()
        upd.path = '/spec'
        upd.value = yt_yson_bindings.dumps_proto(obj.spec)

        upd = req.set_updates.add()
        upd.path = '/labels'
        upd.value = yt_yson_bindings.dumps_proto(obj.labels)

        upd = req.set_updates.add()
        upd.path = '/annotations'
        upd.value = yt_yson_bindings.dumps_proto(obj.annotations)

        for key, value in path_to_value_specific_fields.items():
            upd = req.set_updates.add()
            upd.path = key
            if isinstance(value, (str, int, dict, list)):
                upd.value = yson.dumps(value)
            else:
                upd.value = yt_yson_bindings.dumps_proto(value)

        for path, ts in timestamps.items():
            prereq = req.attribute_timestamp_prerequisites.add()
            prereq.path = path
            prereq.timestamp = int(ts)

        self.stub.UpdateObject(req)

    def update_revision_increment(self, object_type, object_id, obj,
                                  path_to_value_specific_fields=None,
                                  copy_notifications_state=False,
                                  assert_same_revision=False):
        if path_to_value_specific_fields is None:
            path_to_value_specific_fields = self.make_default_specific_fields_to_update(obj)
        selectors = ['/spec', '/spec/revision']
        if copy_notifications_state:
            selectors.append('/annotations/notifications_last_state')

        r = self._get(object_type=object_type,
                      object_id=object_id,
                      selectors=selectors,
                      fetch_timestamps=True,
                      payload_format=object_service_pb2.PF_YSON)
        spec_ts = r.timestamps[0]
        rev_yson = r.value_payloads[1].yson
        rev = yson.loads(rev_yson)
        if assert_same_revision and rev != obj.spec.revision:
            raise ValueError("Update failed - decayed object revision")
        obj.spec.revision = int(rev) + 1
        timestamps = {'/spec': spec_ts}

        if copy_notifications_state:
            notifications_state = yson.loads(r.value_payloads[2].yson)
            if notifications_state:
                path_to_value_specific_fields['/annotations/notifications_last_state'] = notifications_state
                timestamps['/annotations/notifications_last_state'] = r.timestamps[2]

        self.update(object_type=object_type,
                    object_id=object_id,
                    obj=obj,
                    timestamps=timestamps,
                    path_to_value_specific_fields=path_to_value_specific_fields)

    def update_replica_set(self, rs_id, rs):
        # We need separate method for replica_set because
        # replica_set spec is not compatible with other objects spec
        r = self._get(object_type=data_model.OT_REPLICA_SET,
                      object_id=rs_id,
                      selectors=('/spec', '/spec/revision_id'),
                      fetch_timestamps=True,
                      payload_format=object_service_pb2.PF_YSON)
        spec_ts = r.timestamps[0]
        rev_yson = r.value_payloads[1].yson
        rev = yson.loads(rev_yson)
        rs.spec.revision_id = str(int(rev) + 1)
        self.update(object_type=data_model.OT_REPLICA_SET,
                    object_id=rs_id,
                    obj=rs,
                    timestamps={'/spec': spec_ts})

    def remove(self, object_type, object_id):
        req = object_service_pb2.TReqRemoveObject(object_type=object_type,
                                                  object_id=object_id)
        return self.stub.RemoveObject(req)

    def commit_ticket(self, object_id, message, reason, patches):
        req = object_service_pb2.TReqUpdateObject()
        req.object_type = data_model.OT_DEPLOY_TICKET
        req.object_id = object_id

        update = data_model.TDeployTicketControl.TCommitAction()
        update.options.reason = reason
        update.options.message = message
        if patches:
            update.options.patch_selector.type = data_model.DTPST_PARTIAL
            update.options.patch_selector.patch_ids.extend(patches)
        else:
            update.options.patch_selector.type = data_model.DTPST_FULL

        req_update = req.set_updates.add()
        req_update.path = '/control/commit'
        req_update.value_payload.yson = yt_yson_bindings.dumps_proto(update)

        self.stub.UpdateObject(req)

    def skip_ticket(self, object_id, message, reason, patches):
        req = object_service_pb2.TReqUpdateObject()
        req.object_type = data_model.OT_DEPLOY_TICKET
        req.object_id = object_id

        update = data_model.TDeployTicketControl.TSkipAction()
        update.options.reason = reason
        update.options.message = message
        if patches:
            update.options.patch_selector.type = data_model.DTPST_PARTIAL
            update.options.patch_selector.patch_ids.extend(patches)
        else:
            update.options.patch_selector.type = data_model.DTPST_FULL

        req_update = req.set_updates.add()
        req_update.path = '/control/skip'
        req_update.value_payload.yson = yt_yson_bindings.dumps_proto(update)

        self.stub.UpdateObject(req)

    def close(self):
        self.client.close()

    def control(self, object_type, object_id, control_name, options):
        req = object_service_pb2.TReqUpdateObject()
        req.object_type = object_type
        req.object_id = object_id

        req_update = req.set_updates.add()
        req_update.path = '/control/{}'.format(control_name)
        req_update.value_payload.yson = yt_yson_bindings.dumps_proto(options)

        self.stub.UpdateObject(req)
