#include <infra/decapinger/src/config.h>

#include <library/cpp/config/config.h>

#include <util/system/file.h>
#include <util/stream/file.h>
#include <util/generic/bitops.h>

namespace NDecaPinger {
    class TConfig::TImpl {
    public:
        TImpl()
            : ProbesUri("noc-export.yandex.net", 80, "/rt/decap-monitoring-info.json")
            , ProbesRefreshInterval(TDuration::Minutes(5))
            , ProbesResendInterval(TDuration::Minutes(1))
            , ProbesJitterInterval(TDuration::Seconds(10))
            , ProbesSourceIPv4(172114432, TIpv6Address::Ipv4) // 10.66.66.0
            , ProbesSourceIPv4PrefixLen(24)
            , ProbesDestinationPort(9)
        {
        }

        void Load(const TString& fileName) {
            const auto config(LoadConfig(fileName));
            const auto probesConfig(config["probes"]);
            if (!probesConfig.IsNull()) {
                // ProbesUri
                const auto probesUri(probesConfig["uri"].As<TString>());
                if (!probesUri.empty()) {
                    ProbesUri.Parse(probesUri);
                }

                // ProbesRefreshInterval
                TDuration::TryParse(probesConfig["refresh_interval"].As<TString>(), ProbesRefreshInterval);
                Y_VERIFY(ProbesRefreshInterval.Seconds() > 0);

                // ProbesResendInterval
                TDuration::TryParse(probesConfig["resend_interval"].As<TString>(), ProbesResendInterval);
                Y_VERIFY(ProbesResendInterval.Seconds() > 0);

                // ProbesJitterInterval
                TDuration::TryParse(probesConfig["jitter_interval"].As<TString>(), ProbesJitterInterval);

                // ProbesSourceIPv4
                ParseSourceIPv4(probesConfig["source_ip_v4"].As<TString>());

                // ProbesDestinationPort
                ProbesDestinationPort = probesConfig["destination_port"].As<ui16>(ProbesDestinationPort);
            }
        }

        void Out(IOutputStream& stream) const {
            stream << "ProbesUri=" << ProbesUri
                   << " ProbesRefreshInterval=" << ProbesRefreshInterval
                   << " ProbesResendInterval=" << ProbesResendInterval
                   << " ProbesJitterInterval=" << ProbesJitterInterval
                   << " ProbesSourceIPv4=" << ProbesSourceIPv4
                   << " ProbesSourceIPv4PrefixLen=" << ProbesSourceIPv4PrefixLen
                   << " ProbesDestinationPort=" << ProbesDestinationPort;
        }

        inline const NUri::TUri& GetProbesUri() const noexcept {
            return ProbesUri;
        }

        inline const TDuration& GetProbesRefreshInterval() const noexcept {
            return ProbesRefreshInterval;
        }

        const TDuration& GetProbesResendInterval() const noexcept {
            return ProbesResendInterval;
        }

        const TDuration& GetProbesJitterInterval() const noexcept {
            return ProbesJitterInterval;
        }

        const TIpv6Address& GetProbesSourceIPv4() const noexcept {
            return ProbesSourceIPv4;
        }

        unsigned int GetProbesSourceIPv4PrefixLen() const noexcept {
            return ProbesSourceIPv4PrefixLen;
        }

        inline ui16 GetProbesDestinationPort() const noexcept {
            return ProbesDestinationPort;
        }

    private:
        NConfig::TConfig LoadConfig(const TString& fileName) {
            const TFile file(fileName, EOpenModeFlag::OpenExisting | EOpenModeFlag::RdOnly);
            TFileInput fileInput(file);
            NConfig::TGlobals globals;
            return NConfig::TConfig::FromJson(fileInput, globals);
        }

        void ParseSourceIPv4(const TString& sourceIPv4) {
            unsigned int probesSourceIPv4PrefixLen = 32;

            const auto slashPos(sourceIPv4.rfind('/'));
            if (slashPos != TString::npos) {
                probesSourceIPv4PrefixLen = FromString<unsigned int>(sourceIPv4.substr(slashPos + 1));
                Y_VERIFY(probesSourceIPv4PrefixLen <= 32);
            }

            bool ok;
            const auto probesSourceIPv4(TIpv6Address::FromString(TStringBuf(sourceIPv4, 0, slashPos), ok));
            if (ok) {
                Y_VERIFY(probesSourceIPv4.Type() == TIpv6Address::Ipv4);
                if (probesSourceIPv4PrefixLen < 32) {
                    // clear least significant bits
                    ui128 netMask(GetLow(ui128(probesSourceIPv4)) & InverseMaskLowerBits(32 - probesSourceIPv4PrefixLen));
                    ProbesSourceIPv4 = TIpv6Address(netMask, TIpv6Address::Ipv4);
                } else {
                    ProbesSourceIPv4 = probesSourceIPv4;
                }
                ProbesSourceIPv4PrefixLen = probesSourceIPv4PrefixLen;
            }
        }

        NUri::TUri ProbesUri;
        TDuration ProbesRefreshInterval;
        TDuration ProbesResendInterval;
        TDuration ProbesJitterInterval;
        TIpv6Address ProbesSourceIPv4;
        unsigned int ProbesSourceIPv4PrefixLen;
        ui16 ProbesDestinationPort;
    };

    TConfig::TConfig()
        : Impl(MakeHolder<TImpl>())
    {
    }

    TConfig::~TConfig() = default;

    void TConfig::Load(const TString& fileName) {
        Impl->Load(fileName);
    }

    void TConfig::Out(IOutputStream& stream) const {
        Impl->Out(stream);
    }

    const NUri::TUri& TConfig::GetProbesUri() const noexcept {
        return Impl->GetProbesUri();
    }

    const TDuration& TConfig::GetProbesRefreshInterval() const noexcept {
        return Impl->GetProbesRefreshInterval();
    }

    const TDuration& TConfig::GetProbesResendInterval() const noexcept {
        return Impl->GetProbesResendInterval();
    }

    const TDuration& TConfig::GetProbesJitterInterval() const noexcept {
        return Impl->GetProbesJitterInterval();
    }

    const TIpv6Address& TConfig::GetProbesSourceIPv4() const noexcept {
        return Impl->GetProbesSourceIPv4();
    }

    unsigned int TConfig::GetProbesSourceIPv4PrefixLen() const noexcept {
        return Impl->GetProbesSourceIPv4PrefixLen();
    }

    ui16 TConfig::GetProbesDestinationPort() const noexcept {
        return Impl->GetProbesDestinationPort();
    }
}

template <>
void Out<NDecaPinger::TConfig>(IOutputStream& stream, const NDecaPinger::TConfig& config) {
    config.Out(stream);
}
