#include <infra/decapinger/src/config.h>
#include <infra/decapinger/src/noc_client.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/getopt/last_getopt.h>

#include <util/random/random.h>

using namespace NDecaPinger;

namespace {
    void ParseOptions(int argc, char* argv[]) {
        using namespace NLastGetopt;

        TOpts opts(TOpts::Default());
        opts.AddHelpOption('h');
        opts.AddVersionOption('v');

        TOpt& verboseOpt = opts.AddLongOption("verbose", "Enable verbose mode");
        verboseOpt.Optional().HasArg(NO_ARGUMENT);

        TOpt& configFileOpt = opts.AddLongOption('c', "config", "Path to config file");
        configFileOpt.Optional().RequiredArgument().DefaultValue("");

        TOptsParseResult parsedOpts(&opts, argc, argv);

        int logLevel = TLOG_INFO;
        if (parsedOpts.Has(&verboseOpt)) {
            logLevel = TLOG_DEBUG;
        }
        InitGlobalLog2Console(logLevel);

        if (parsedOpts.Has(&configFileOpt)) {
            try {
                TConfig::Get().Load(parsedOpts.Get(&configFileOpt));
            } catch (...) {
                ERROR_LOG << "Failed to load config: " << CurrentExceptionMessage() << Endl;
            }
        }
    }

    inline ui64 GetJitterValue() {
        auto jitterInterval(TConfig::Get().GetProbesJitterInterval().Seconds());
        if (jitterInterval == 0) {
            return 0;
        }
        return RandomNumber(jitterInterval);
    }

    void MainLoop() {
        INFO_LOG << "Starting DecaPinger with configuration: " << TConfig::Get() << Endl;

        auto jitter(GetJitterValue());
        if (jitter != 0) {
            sleep(jitter);
        }

        while (true) {
            try {
                for (const auto& probe : TNocClient::Get().GetProbes()) {
                    DEBUG_LOG << "Sending probe: " << probe << Endl;
                    probe.Send();
                }
            } catch (...) {
                ERROR_LOG << "Failed to send probe: " << CurrentExceptionMessage() << Endl;
            }

            sleep(TConfig::Get().GetProbesResendInterval().Seconds() + GetJitterValue());
        }
    }
}

int main(int argc, char* argv[]) {
    ParseOptions(argc, argv);

    MainLoop();

    return 0;
}
