#include <infra/decapinger/src/noc_client.h>
#include <infra/decapinger/src/config.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/json/json_reader.h>

namespace NDecaPinger {
    class TNocClient::TImpl {
    public:
        TImpl(const TString& host, ui16 port)
            : HttpClient(host, port, TDuration::Seconds(10) /*socketTimeout*/, TDuration::Seconds(5) /*connectTimeout*/)
        {
        }

        inline const TVector<TProbe>& GetProbes() {
            const auto now(TInstant::Now());
            if (Probes.empty() || now >= ProbesTimestamp + TConfig::Get().GetProbesRefreshInterval()) {
                TStringStream stream;
                try {
                    HttpClient.DoGet(TConfig::Get().GetProbesUri().GetField(NUri::TField::FieldPath), &stream);
                    ParseProbes(stream);
                    ProbesTimestamp = now;
                } catch (const THttpRequestException& e) {
                    if (e.GetStatusCode() != 304 /*Not Modified*/) {
                        ERROR_LOG << "HTTP client error: " << CurrentExceptionMessage() << Endl;
                    }
                } catch (...) {
                    ERROR_LOG << "HTTP client error: " << CurrentExceptionMessage() << Endl;
                }
            }
            return Probes;
        }

    private:
        void ParseProbes(TStringStream& stream) {
            try {
                TVector<TProbe> newProbes;

                const auto json(NJson::ReadJsonTree(&stream, true));

                for (const auto& probeJson : json["probes"].GetMap()) {
                    bool ok;

                    const auto anycastIPv6(TIpv6Address::FromString(probeJson.second["anycast"].GetString(), ok));
                    if (!ok || anycastIPv6.Type() != TIpv6Address::Ipv6) {
                        ERROR_LOG << "Failed to parse anycast=\"" << probeJson.second["anycast"] << "\" to IPv6 address" << Endl;
                        continue;
                    }

                    TProbe probe(probeJson.first, anycastIPv6);

                    for (const auto& dest : probeJson.second["destinations"].GetArray()) {
                        const auto innerIPv4(TIpv6Address::FromString(dest["inner_v4"].GetString(), ok));
                        if (!ok || innerIPv4.Type() != TIpv6Address::Ipv4) {
                            ERROR_LOG << "Failed to parse inner_v4=\"" << dest["inner_v4"] << "\" to IPv4 address" << Endl;
                            continue;
                        }

                        const auto nativeIPv6(TIpv6Address::FromString(dest["native_v6"].GetString(), ok));
                        if (!ok || nativeIPv6.Type() != TIpv6Address::Ipv6) {
                            ERROR_LOG << "Failed to parse native_v6=\"" << dest["native_v6"] << "\" to IPv6 address" << Endl;
                            continue;
                        }

                        probe.AddDestination(innerIPv4, nativeIPv6);
                    }

                    newProbes.emplace_back(std::move(probe));
                }

                Probes = std::move(newProbes);
            } catch (...) {
                ERROR_LOG << "Failed to parse probes from JSON: " << CurrentExceptionMessage() << Endl;
            }
        }

        TSimpleHttpClient HttpClient;

        TVector<TProbe> Probes;
        TInstant ProbesTimestamp;
    };

    TNocClient::TNocClient()
        : Impl(MakeHolder<TImpl>(TString(TConfig::Get().GetProbesUri().GetHost()), TConfig::Get().GetProbesUri().GetPort()))
    {
    }

    TNocClient::~TNocClient() = default;

    const TVector<TProbe>& TNocClient::GetProbes() const {
        return Impl->GetProbes();
    }
}
