#include <infra/decapinger/src/packet.h>
#include <infra/decapinger/src/config.h>

#include <library/cpp/logger/global/global.h>

#include <util/random/random.h>

#include <netinet/ip.h>
#define __FAVOR_BSD
#include <netinet/udp.h>

namespace {
    static const std::size_t IPv4_HEADER_SIZE = sizeof(struct ip);
    static const std::size_t UDP_HEADER_SIZE = sizeof(struct udphdr);

    ui16 IPv4Checksum(const ui8 buffer[], std::size_t size) {
        unsigned int sum = 0;

        while (size > 1) {
            sum += *reinterpret_cast<const unsigned short*>(buffer);
            buffer += 2;
            size -= 2;
        }

        if (size > 0) {
            sum += *buffer;
        }

        while (sum >> 16) {
            sum = (sum & 0xffff) + (sum >> 16);
        }

        return ~sum;
    }
}

namespace NDecaPinger {
    ui8 TPacket::Buffer[TPacket::BufferSize];

    TPacket::TPacket(std::size_t size)
        : Size(size)
    {
        Y_VERIFY(Size < BufferSize);
        memset(Buffer, 0, BufferSize);
    }

    void TPacket::Send(const TIpv6Address& destination, ui16 port) {
        TSocketHolder s(CreateSocket());
        if (s == INVALID_SOCKET) {
            ERROR_LOG << "socket() failed: " << LastSystemErrorText() << Endl;
            return;
        }

        struct sockaddr_in sa4;
        struct sockaddr_in6 sa6;
        const sockaddr* sa;
        socklen_t len;
        destination.ToSockaddrAndSocklen(sa4, sa6, sa, len, port);
        if (sendto(s, Buffer, Size, 0, sa, len) < 0) {
            ERROR_LOG << "sendto() failed: " << LastSystemErrorText() << Endl;
        }
    }

    TIPv6Packet::TIPv6Packet(const TString& payload)
        : TPacket(payload.size())
    {
        memcpy(Buffer, payload.data(), payload.size());
    }

    SOCKET TIPv6Packet::CreateSocket() {
        return socket(AF_INET6, SOCK_DGRAM, IPPROTO_UDP);
    }

    TIPv4IPv6Packet::TIPv4IPv6Packet(const TString& payload, const TIpv6Address& sourceIPv4, const TIpv6Address& destinationIPv4)
        : TPacket(IPv4_HEADER_SIZE + UDP_HEADER_SIZE + payload.size())
    {
        struct ip* ip = reinterpret_cast<struct ip*>(Buffer);
        ip->ip_v = 4;
        ip->ip_hl = 5;
        ip->ip_tos = 0;
        ip->ip_len = htons(Size);
        ip->ip_id = 0;
        ip->ip_off = 0;
        ip->ip_ttl = 64;
        ip->ip_p = IPPROTO_UDP;
        ip->ip_sum = 0;
        sourceIPv4.ToInAddr(ip->ip_src);
        destinationIPv4.ToInAddr(ip->ip_dst);

        ip->ip_sum = ::IPv4Checksum(Buffer, IPv4_HEADER_SIZE);

        struct udphdr* udp = reinterpret_cast<struct udphdr*>(Buffer + IPv4_HEADER_SIZE);
        udp->uh_sport = RandomNumber<ui16>();
        udp->uh_dport = htons(TConfig::Get().GetProbesDestinationPort());
        udp->uh_ulen = htons(Size - IPv4_HEADER_SIZE);
        udp->uh_sum = 0;

        memcpy(Buffer + IPv4_HEADER_SIZE + UDP_HEADER_SIZE, payload.data(), payload.size());
    }

    SOCKET TIPv4IPv6Packet::CreateSocket() {
        return socket(AF_INET6, SOCK_RAW, IPPROTO_IPIP);
    }
}
