#include <infra/decapinger/src/probe.h>
#include <infra/decapinger/src/packet.h>
#include <infra/decapinger/src/config.h>

#include <util/system/hostname.h>
#include <util/generic/guid.h>
#include <util/generic/bitops.h>
#include <util/random/random.h>

namespace NDecaPinger {
    TProbe::TDestination::TDestination(const TIpv6Address& innerIPv4, const TIpv6Address& nativeIPv6)
        : InnerIPv4(innerIPv4)
        , NativeIPv6(nativeIPv6)
    {
    }

    TProbe::TProbe(const TString& name, const TIpv6Address& anycastIPv6)
        : Name(name)
        , AnycastIPv6(anycastIPv6)
    {
        Payload["probe"] = Name;
        Payload["fqdn"] = FQDNHostName();
    }

    void TProbe::AddDestination(const TIpv6Address& innerIPv4, const TIpv6Address& nativeIPv6) {
        Destinations.emplace_back(innerIPv4, nativeIPv6);
    }

    TString TProbe::GeneratePayload() const {
        Payload["timestamp"] = TInstant::Now().ToString();
        Payload["id"] = CreateGuidAsString();
        return TStringBuilder() << Payload;
    }

    void TProbe::Send() const {
        const auto payload(GeneratePayload());

        for (const auto& dest : Destinations) {
            TIPv6Packet(payload).Send(dest.NativeIPv6, TConfig::Get().GetProbesDestinationPort());

            auto sourceIPv4PrefixLen(TConfig::Get().GetProbesSourceIPv4PrefixLen());
            if (sourceIPv4PrefixLen < 32) {
                ui128 ip(ui128(TConfig::Get().GetProbesSourceIPv4()) | ui128(RandomNumber(MaskLowerBits(32 - sourceIPv4PrefixLen))));
                TIpv6Address sourceIPv4(ip, TIpv6Address::Ipv4);
                TIPv4IPv6Packet(payload, sourceIPv4, dest.InnerIPv4).Send(AnycastIPv6, 0);
            } else {
                TIPv4IPv6Packet(payload, TConfig::Get().GetProbesSourceIPv4(), dest.InnerIPv4).Send(AnycastIPv6, 0);
            }
        }
    }

    void TProbe::Out(IOutputStream& stream) const {
        stream << "Probe \"" << Name << "\": anycast=" << AnycastIPv6.ToString(false) << " destinations=[";
        for (const auto& dest : Destinations) {
            stream << "(inner_v4=" << dest.InnerIPv4.ToString(false)
                   << " native_v6=" << dest.NativeIPv6.ToString(false) << ")";
        }
        stream << "]";
    }
}

template <>
void Out<NDecaPinger::TProbe>(IOutputStream& stream, const NDecaPinger::TProbe& probe) {
    probe.Out(stream);
}
