#! /usr/bin/env python
from __future__ import unicode_literals
from __future__ import print_function

import argparse
import datetime
import os

import yt.yson as yson
from yp import client
from yp_proto.yp.client.api.proto import object_service_pb2
from yp_proto.yp.client.api.proto import autogen_pb2


def main(date, pod_set_id):
    start = datetime.datetime.strptime(date, '%Y-%m-%d')
    end = start + datetime.timedelta(days=1)
    for cluster in ['iva', 'myt', 'man', 'sas', 'vla']:
        p = client.YpClient('{}.yp.yandex.net:8443'.format(cluster), transport='http',
                            config={'enable_https': True, 'token': os.getenv('YP_TOKEN')})
        stub = p.create_grpc_object_stub()
        req = object_service_pb2.TReqSelectObjects()
        req.filter.query = '[/meta/pod_set_id] = "{}"'.format(pod_set_id)
        req.object_type = autogen_pb2.OT_POD
        req.selector.paths.append('/meta/id')
        req.selector.paths.append('/status/scheduling/last_state_update_time')
        req.format = object_service_pb2.PF_PROTOBUF
        req.options.fetch_root_object = True
        resp = stub.SelectObjects(req)
        for res in resp.results:
            pod = autogen_pb2.TPod()
            pod.MergeFromString(res.value_payloads[0].protobuf)
            req = object_service_pb2.TReqSelectObjectHistory()
            req.object_type = autogen_pb2.OT_POD
            req.object_id = pod.meta.id
            req.selector.paths.append('/status/scheduling')
            req.format = object_service_pb2.PF_YSON
            history = stub.SelectObjectHistory(req)
            begin = (None, None)
            events = []
            nodes = set()

            for e in history.events:
                scheduling = yson.loads(e.results.value_payloads[0].yson)
                if not scheduling or scheduling['state'] != 'assigned':
                    continue
                dt = e.time.ToDatetime()
                node = scheduling['node_id']
                if dt < start and node:
                    begin = (dt, node)
                    continue
                if not events:
                    events.append(begin)
                    nodes.add(begin[1])
                if dt > end:
                    break
                events.append((dt, node))
                nodes.add(node)
            if not events:
                nodes.add(begin[1])
            pod_name = '{}.{}.yp-c.yandex.net'.format(pod.meta.id, cluster)
            if len(nodes) == 1:
                node = list(nodes)[0]
                if node:
                    print(pod_name, 'all day', date, 'was on', node)
                else:
                    print(pod_name, 'was not created on', date)
            else:
                print(pod_name, 'migration history')
                for e in events:
                    print(*e)


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--date', help='Date in format YYYY-MM-DD', required=True)
    parser.add_argument('--pod-set-id', help='E.g. ', required=True)
    args = parser.parse_args()
    main(args.date, args.pod_set_id)
