import argparse
import os

import yp.client
import yp.data_model as data_model
import yt.yson as yson
from yp_proto.yp.client.api.proto import object_service_pb2

import infra.dctl.src.consts as consts

YP_TOKEN_ENV = "YP_TOKEN"
YP_TOKEN_FILE = os.path.expanduser("~/.yp/token")

SELECT_LIMIT = 10000


def get_cluster_list():
    ret = []
    for cluster in consts.CLUSTER_CONFIGS.keys():
        ret.append(cluster)

    return ret


def get_token(token_env, token_path):
    token = os.getenv(token_env)
    if token:
        print("Use yp token from env {}".format(token_env))
        return token

    if os.path.isfile(token_path):
        print("Use yp token from file {}".format(token_path))
        with open(token_path, 'r') as f:
            return f.read().strip()

    raise Exception("No yp token provided")


def get_pod_template_spec(yp_client_stub, object_id, object_type):
    req = object_service_pb2.TReqGetObject()
    req.object_type = object_type
    req.object_id = object_id
    req.selector.paths.append("/spec/pod_template_spec")

    resp = yp_client_stub.GetObject(req)

    return yson.loads(resp.result.values[0])


def entity_has_bug(yp_client_stub, object_id, object_type):
    pod_template_spec = get_pod_template_spec(yp_client_stub, object_id, object_type)

    if not isinstance(pod_template_spec, dict):
        return False

    disk_volume_requests = pod_template_spec.get("spec", dict()).get("disk_volume_requests", [])
    for request in disk_volume_requests:
        if request.get("id", "") == 'pod_agent':
            if request.get("quota_policy", dict()).get("capacity", 0) > 1073741824:
                return True

    return False


def filter_entities_with_bug(yp_client_stub, entities, object_type):
    entites_with_bug = []
    for entity in entities:
        if entity_has_bug(yp_client_stub, entity, object_type):
            entites_with_bug.append(entity)

    return entites_with_bug


def list_objects(yp_client_stub, objectType):
    req = object_service_pb2.TReqSelectObjects()
    req.object_type = objectType
    req.limit.value = SELECT_LIMIT

    req.selector.paths.append("/meta/id")
    resp = yp_client_stub.SelectObjects(req)

    objects = []
    for r in resp.results:
        try:
            object_id = yson.loads(r.values[0])
        except Exception:
            print("Error while parsing object id in select '{}'".format(str(r)))
            continue

        objects.append(object_id)

    return objects


def list_entites_with_juggler_disk_quota_bug(
    cluster,
    token
):
    cluster_config = consts.CLUSTER_CONFIGS[cluster]

    print("-" * 50)

    yp_token = get_token(YP_TOKEN_ENV, YP_TOKEN_FILE) if not token else token

    yp_client = yp.client.YpClient(
        address=cluster_config.address,
        config={
            'token': yp_token,
        }
    )
    yp_client_stub = yp_client.create_grpc_object_stub()

    multi_cluster_replica_sets = list_objects(yp_client_stub, data_model.OT_MULTI_CLUSTER_REPLICA_SET)
    filtered_multi_cluster_replica_sets = filter_entities_with_bug(yp_client_stub, multi_cluster_replica_sets, data_model.OT_MULTI_CLUSTER_REPLICA_SET)

    replica_sets = list_objects(yp_client_stub, data_model.OT_REPLICA_SET)
    filtered_replica_sets = filter_entities_with_bug(yp_client_stub, replica_sets, data_model.OT_REPLICA_SET)

    stages = set()

    print("-" * 50)
    print("NUM MCRS {}".format(len(filtered_multi_cluster_replica_sets)))
    print("MCRS LIST:")
    for mcrs in filtered_multi_cluster_replica_sets:
        stages.add(mcrs.split('.')[0])
        print("\'{}\',".format(mcrs))

    print("-" * 50)
    print("NUM RS {}".format(len(filtered_replica_sets)))
    print("RS LIST:")
    for rs in filtered_replica_sets:
        stages.add(rs.split('.')[0])
        print("\'{}\',".format(rs))

    print("-" * 50)
    print("NUM STAGES {}".format(len(stages)))
    print("STAGES LIST:")
    for stage in stages:
        print("\'{}\',".format(stage))

    raw_input("Press enter to continue")


def main(arguments):
    for cluster in arguments.clusters:
        try:
            list_entites_with_juggler_disk_quota_bug(
                cluster,
                arguments.token
            )
        except Exception as e:
            print("Error in cluster {}: '{}'".format(cluster, str(e)))


def parse_arguments():
    parser = argparse.ArgumentParser(description="Get stages and pod sets with juggler disk quota bug DEPLOY-3588.")
    parser.add_argument(
        "clusters",
        metavar="cluster",
        type=str,
        nargs="+",
        choices=get_cluster_list(),
        help="clusters to get info."
    )
    parser.add_argument(
        "--token",
        dest="token",
        default=None,
        help="use specified YP token."
    )

    return parser.parse_args()


if __name__ == "__main__":
    main(parse_arguments())
