import os

import yp.client
import yp.data_model as data_model
import yt.yson as yson
from yp_proto.yp.client.api.proto import object_service_pb2

import infra.dctl.src.consts as consts
import infra.deploy.dri.lib.io_utils as io_utils
from infra.deploy.dri.lib.yp_object_filter import YpObjectFilter, AcceptAnySpecFilter


class StagesGetter:
    __slots__ = ['yp_token', 'yp_client_stub', 'cluster_configs', 'object_filter']

    YP_TOKEN_ENV = "YP_TOKEN"
    YP_TOKEN_FILE = os.path.expanduser("~/.yp/token")

    SELECT_LIMIT = 1000

    META_ID_SELECTOR = '/meta/id'

    @staticmethod
    def get_yp_token(args_token, token_env, token_path):
        if args_token:
            print("Use yp token from arguments")
            return args_token

        env_token = os.getenv(token_env)
        if env_token:
            print("Use yp token from env {}".format(token_env))
            return env_token

        if os.path.isfile(token_path):
            print("Use yp token from file {}".format(token_path))
            with open(token_path, 'r') as f:
                return f.read().strip()

        raise Exception("No yp token provided")

    def __init__(self,
                 token: str = '',
                 cluster_configs=consts.CLUSTER_CONFIGS,
                 object_filter: YpObjectFilter = AcceptAnySpecFilter()):
        self.yp_token = StagesGetter.get_yp_token(token, StagesGetter.YP_TOKEN_ENV, StagesGetter.YP_TOKEN_FILE)
        self.cluster_configs = cluster_configs
        self.object_filter = object_filter

    def get_objects(self,
                    yp_client_stub,
                    object_type):
        req = object_service_pb2.TReqSelectObjects()
        req.object_type = object_type
        req.limit.value = StagesGetter.SELECT_LIMIT

        selectors = req.selector.paths

        selectors.append(StagesGetter.META_ID_SELECTOR)
        selectors.extend(self.object_filter.selectors)

        objects = []

        offset = 0

        while True:
            req.offset.value = offset
            resp = yp_client_stub.SelectObjects(req)
            for r in resp.results:
                try:
                    parsed_data = (yson.loads(data) for data in r.values)
                    object_data = dict(zip(selectors, parsed_data))
                    objects.append(object_data)
                except Exception as e:
                    object_id = yson.loads(r.values[0])
                    print("Error while parsing object in select '{}': {}".format(
                        str(object_id), str(e)
                    ))

            if len(resp.results) < StagesGetter.SELECT_LIMIT:
                break
            else:
                offset += StagesGetter.SELECT_LIMIT

        return objects

    def get_filtered_objects(self,
                               yp_client_stub,
                               object_type,
                               **kwargs):
        objects = self.get_objects(yp_client_stub, object_type)

        filtered = []
        for object in objects:
            object_id = object[StagesGetter.META_ID_SELECTOR]

            try:
                filter_result = self.object_filter(yp_client_stub, object, object_type, **kwargs)
                if filter_result:
                    filtered.append(object_id)
            except Exception as e:
                print("Error while filtering object with id '{}': {}".format(object_id, str(e)))
                continue

        return filtered

    def create_yp_client_stub(self,
                              cluster):
        cluster_config = self.cluster_configs[cluster]

        yp_client = yp.client.YpClient(
            address=cluster_config.address,
            config={
                'token': self.yp_token,
            }
        )

        yp_client_stub = yp_client.create_grpc_object_stub()

        return yp_client_stub

    def process_cluster(self,
                        cluster: str,
                        **kwargs):

        io_utils.print_separating_line()

        print("Processing cluster {}".format(cluster))

        yp_client_stub = self.create_yp_client_stub(cluster)

        multi_cluster_replica_sets = self.get_filtered_objects(
            yp_client_stub,
            data_model.OT_MULTI_CLUSTER_REPLICA_SET,
            **kwargs
        )

        replica_sets = self.get_filtered_objects(
            yp_client_stub,
            data_model.OT_REPLICA_SET,
            **kwargs
        )

        stages = set()

        for mcrs in multi_cluster_replica_sets:
            stages.add(mcrs.split('.')[0])

        for rs in replica_sets:
            stages.add(rs.split('.')[0])

        entities = {
            'rs': replica_sets,
            'mcrs': multi_cluster_replica_sets,
            'stages': stages
        }

        io_utils.print_entities(entities)

        return entities

    def process_all_clusters(self,
                             clusters,
                             **kwargs):

        cluster_to_entities = {}

        all_entities = {}

        for cluster in clusters:
            try:
                cluster_entities = cluster_to_entities[cluster] = self.process_cluster(
                    cluster,
                    **kwargs
                )

                for type, entities in cluster_entities.items():
                    all_entities[type] = all_entities.get(type, set()).union(entities)
            except Exception as e:
                print("Error in cluster {}: '{}'".format(cluster, str(e)))

        io_utils.print_separating_line()
        print("Total statistic for all clusters:")
        io_utils.print_only_statistic(all_entities)

        return cluster_to_entities
