from abc import abstractmethod
import yaml

from infra.deploy.dri.logbroker.logbroker_filter import LogbrokerFilter
import infra.deploy.dri.lib.spec_utils as spec_utils


class UnifiedAgentConfigUrlFilter(LogbrokerFilter):
    def check_pod_template_spec_with_logs(self,
                                          pod_template_spec):
        static_resources = spec_utils.get_statis_resources(pod_template_spec)
        unified_agent_config_resource = spec_utils.find_by_id(static_resources, 'unified_agent_config')

        unified_agent_config_url = unified_agent_config_resource.get('url')
        if unified_agent_config_url:
            return self.check_unified_agent_config_url(unified_agent_config_url)
        else:
            return False

    @abstractmethod
    def check_unified_agent_config_url(self, unified_agent_config_url):
        pass


class UnifiedAgentConfigYamlFilter(UnifiedAgentConfigUrlFilter):
    def check_unified_agent_config_url(self, unified_agent_config_url):
        unified_agent_config_only_yaml = unified_agent_config_url[len('raw:---'):]
        unified_agent_config = yaml.safe_load(unified_agent_config_only_yaml)
        return self.check_unified_agent_config_yaml(unified_agent_config)

    @staticmethod
    def get_route(unified_agent_config):
        return unified_agent_config.get('routes', [dict()])[0]

    @staticmethod
    def get_channel(unified_agent_config):
        return UnifiedAgentConfigYamlFilter.get_route(unified_agent_config) \
            .get('channel', dict())

    @staticmethod
    def get_pipe(unified_agent_config):
        return UnifiedAgentConfigYamlFilter.get_channel(unified_agent_config) \
            .get('pipe', list())

    @staticmethod
    def get_filter(unified_agent_config, plugin_id):
        pipe = UnifiedAgentConfigYamlFilter.get_pipe(unified_agent_config)
        pipe_filter_elements = map(lambda el: el.get('filter', dict()), pipe)
        return spec_utils.find_by_id(pipe_filter_elements, plugin_id, id_field='plugin')

    @abstractmethod
    def check_unified_agent_config_yaml(self, unified_agent_config):
        pass
