import argparse
import os

import yp.client
import yp.data_model as data_model
import yt.yson as yson
from yp_proto.yp.client.api.proto import object_service_pb2

import infra.dctl.src.consts as consts

YP_TOKEN_ENV = "YP_TOKEN"
YP_TOKEN_FILE = os.path.expanduser("~/.yp/token")

SELECT_LIMIT = 10000


def get_cluster_list():
    ret = []
    for cluster in consts.CLUSTER_CONFIGS.keys():
        ret.append(cluster)

    return ret


def get_token(token_env, token_path):
    token = os.getenv(token_env)
    if token:
        print("Use yp token from env {}".format(token_env))
        return token

    if os.path.isfile(token_path):
        print("Use yp token from file {}".format(token_path))
        with open(token_path, 'r') as f:
            return f.read().strip()

    raise Exception("No yp token provided")


def get_pod_template_spec(yp_client_stub, object_id, object_type):
    req = object_service_pb2.TReqGetObject()
    req.object_type = object_type
    req.object_id = object_id
    req.selector.paths.append("/spec/pod_template_spec")

    resp = yp_client_stub.GetObject(req)

    return yson.loads(resp.result.values[0])


def get_entity_layers(yp_client_stub, object_id, object_type):
    req = object_service_pb2.TReqGetObject()
    req.object_type = object_type
    req.object_id = object_id

    resp = yp_client_stub.GetObject(req)

    return yson.loads(resp.result.values[0])


def entity_has_logbroker_layer(yp_client_stub, object_id, object_type, logbroker_tools_layer):
    pod_template_spec = get_pod_template_spec(yp_client_stub, object_id, object_type)

    if not isinstance(pod_template_spec, dict):
        return False

    logs_enabled = pod_template_spec.get("labels", dict()).get("deploy", dict()).get("logs", dict()).get("delivery_enabled", False)

    if logs_enabled:
        layers = pod_template_spec.get("spec", dict()).get("pod_agent_payload", dict()).get("spec", dict()).get("resources", dict()).get("layers", [])
        for layer in layers:
            if layer.get("url", "") == logbroker_tools_layer:
                return True

    return False


def filter_entities_with_concrete_logbroker_layer(yp_client_stub, entities, object_type, logbroker_tools_layer):
    entites_with_layer = []
    for entity in entities:
        if entity_has_logbroker_layer(yp_client_stub, entity, object_type, logbroker_tools_layer):
            entites_with_layer.append(entity)

    return entites_with_layer


def list_objects(yp_client_stub, objectType):
    req = object_service_pb2.TReqSelectObjects()
    req.object_type = objectType
    req.limit.value = SELECT_LIMIT

    req.selector.paths.append("/meta/id")
    resp = yp_client_stub.SelectObjects(req)

    objects = []
    for r in resp.results:
        try:
            object_id = yson.loads(r.values[0])
        except Exception:
            print("Error while parsing object id in select '{}'".format(str(r)))
            continue

        objects.append(object_id)

    return objects


def list_stages_with_concrete_layer(
    cluster,
    logbroker_tools_layer,
    token
):
    cluster_config = consts.CLUSTER_CONFIGS[cluster]

    print("-" * 50)

    yp_token = get_token(YP_TOKEN_ENV, YP_TOKEN_FILE) if not token else token

    yp_client = yp.client.YpClient(
        address=cluster_config.address,
        config={
            'token': yp_token,
        }
    )
    yp_client_stub = yp_client.create_grpc_object_stub()

    multi_cluster_replica_sets = list_objects(yp_client_stub, data_model.OT_MULTI_CLUSTER_REPLICA_SET)
    filtered_multi_cluster_replica_sets = filter_entities_with_concrete_logbroker_layer(yp_client_stub, multi_cluster_replica_sets, data_model.OT_MULTI_CLUSTER_REPLICA_SET, logbroker_tools_layer)

    replica_sets = list_objects(yp_client_stub, data_model.OT_REPLICA_SET)
    filtered_replica_sets = filter_entities_with_concrete_logbroker_layer(yp_client_stub, replica_sets, data_model.OT_REPLICA_SET, logbroker_tools_layer)

    stages = set()

    print("-" * 50)
    print("NUM MCRS {}".format(len(filtered_multi_cluster_replica_sets)))
    print("MCRS LIST:")
    for mcrs in filtered_multi_cluster_replica_sets:
        stages.add(mcrs.split('.')[0])
        print("{}".format(mcrs))

    print("-" * 50)
    print("NUM RS {}".format(len(filtered_replica_sets)))
    print("RS LIST:")
    for rs in filtered_replica_sets:
        stages.add(rs.split('.')[0])
        print("{}".format(rs))

    print("-" * 50)
    print("NUM STAGES {}".format(len(stages)))
    print("STAGES LIST:")
    for stage in stages:
        print("{}".format(stage))

    raw_input("Press enter to continue")


def main(arguments):
    for cluster in arguments.clusters:
        try:
            list_stages_with_concrete_layer(
                cluster,
                arguments.logbroker_tools_layer,
                arguments.token
            )
        except Exception as e:
            print("Error in cluster {}: '{}'".format(cluster, str(e)))


def parse_arguments():
    parser = argparse.ArgumentParser(description="Get stages and pod sets with concrete logbroker tools release.")
    parser.add_argument(
        "clusters",
        metavar="cluster",
        type=str,
        nargs="+",
        choices=get_cluster_list(),
        help="clusters to get info."
    )
    parser.add_argument(
        "--logbroker-tools-layer",
        dest="logbroker_tools_layer",
        type=str,
        required=True,
        help="Logbroker tools layer url or rbtorrent or etc.."
    )
    parser.add_argument(
        "--token",
        dest="token",
        default=None,
        help="use specified YP token."
    )

    return parser.parse_args()


if __name__ == "__main__":
    main(parse_arguments())
