#include "hpa_manager.h"

#include <infra/libs/sensors/macros.h>
#include <infra/libs/sensors/sensor.h>

#include <library/cpp/protobuf/interop/cast.h>

namespace {

void TryUpdateCondition(
    NYP::NClient::NApi::NProto::TCondition* condition
    , NYP::NClient::NApi::NProto::EConditionStatus status
    , const TString& reason
    , const TString& message
    , const TInstant& nowTime
) {
    if (condition->status() != status) {
        condition->set_status(status);
        *(condition->mutable_last_transition_time()) = NProtoInterop::CastToProto(nowTime);
    }
    condition->set_reason(reason);
    condition->set_message(message);
}

void ChangeStatusToFailed(
    const TString& hpaId
    , const TString& reason
    , const TString& message
    , const TInstant& nowTime
    , bool isDryRunMode
    , NInfra::TLogFramePtr frame
    , NYP::NClient::NApi::NProto::THorizontalPodAutoscalerStatus& hpaStatus
) {

    frame->LogEvent(
        ELogPriority::TLOG_DEBUG
        , NInfra::NLogEvent::THorizontalPodAutoscalerStatus(
            hpaId
            , NInfra::NLogEvent::THorizontalPodAutoscalerStatus::FAILED
            , reason
            , 0            /* CurrentReplicas */
            , 0            /* DesiredReplicas */
            , 0            /* MetricValue     */
            , isDryRunMode /* InDryRunMode    */
        )
    );

    TryUpdateCondition(
        hpaStatus.mutable_ready()
        , NYP::NClient::NApi::NProto::EConditionStatus::CS_FALSE
        , reason
        , message
        , nowTime
    );

    TryUpdateCondition(
        hpaStatus.mutable_in_progress()
        , NYP::NClient::NApi::NProto::EConditionStatus::CS_FALSE
        , reason
        , message
        , nowTime
    );

    TryUpdateCondition(
        hpaStatus.mutable_failed()
        , NYP::NClient::NApi::NProto::EConditionStatus::CS_TRUE
        , reason
        , message
        , nowTime
    );
}

NInfra::NClients::EGolovanPeriod ProtoPeriodToGolovanPeriod(NYP::NClient::NApi::NProto::TCustomMetricSpec_TGolovanSignal_EGolovanPeriod period) {
    switch (period) {
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TGolovanSignal::GP_DEFAULT:
            return NInfra::NClients::EGolovanPeriod::FIVE_MIN;
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TGolovanSignal::GP_FIVE_SEC:
            return NInfra::NClients::EGolovanPeriod::FIVE_SEC;
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TGolovanSignal::GP_FIVE_MIN:
            return NInfra::NClients::EGolovanPeriod::FIVE_MIN;
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TGolovanSignal::GP_ONE_HOUR:
            return NInfra::NClients::EGolovanPeriod::ONE_HOUR;
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TGolovanSignal::GP_THREE_HOURS:
            return NInfra::NClients::EGolovanPeriod::THREE_HOURS;
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TGolovanSignal::GP_SIX_HOURS:
            return NInfra::NClients::EGolovanPeriod::SIX_HOURS;
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TGolovanSignal::GP_TWELVE_HOURS:
            return NInfra::NClients::EGolovanPeriod::TWELVE_HOURS;
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TGolovanSignal::GP_ONE_DAY:
            return NInfra::NClients::EGolovanPeriod::ONE_DAY;
        default:
            ythrow yexception() << "Unknown golovan period '" << NYP::NClient::NApi::NProto::TCustomMetricSpec_TGolovanSignal_EGolovanPeriod_Name(period) << "'";
            break;
    }
}

NInfra::NClients::EGolovanPeriod ProtoDurationToGolovanPeriod(const google::protobuf::Duration& period) {
    if (!period.seconds()) { // default period
        return NInfra::NClients::EGolovanPeriod::FIVE_MIN;
    }

    return static_cast<NInfra::NClients::EGolovanPeriod>(period.seconds());
}

TDuration ProtoDurationToSolomonPeriod(const google::protobuf::Duration& period) {
    if (!period.seconds()) { // default solomon period
        return TDuration::Minutes(5);
    }

    return NProtoInterop::CastFromProto(period);
}

TString ProtoAggregationTypeToString(NYP::NClient::NApi::NProto::TCustomMetricSpec_TSolomonSignal_EAggregationType aggregation) {
    switch (aggregation) {
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TSolomonSignal::AT_AVG:
            return "avg";
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TSolomonSignal::AT_MAX:
            return "max";
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TSolomonSignal::AT_MIN:
            return "min";
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TSolomonSignal::AT_SUM:
            return "sum";
        case NYP::NClient::NApi::NProto::TCustomMetricSpec_TSolomonSignal::AT_LAST:
            return "last";
        default:
            ythrow yexception() << "Unknown solomon aggregation type '" << NYP::NClient::NApi::NProto::TCustomMetricSpec_TSolomonSignal_EAggregationType_Name(aggregation) << "'";
    }
}

} // namespace

namespace NInfra::NHorizontalPodAutoscalerController {

const TDuration DEFAULT_UPSCALE_DELAY = TDuration::Minutes(30);
const TDuration DEFAULT_DOWNSCALE_DELAY = TDuration::Minutes(60);
const TDuration MIN_SCALE_DELAY = TDuration::Minutes(5);

const NClients::EGolovanPeriod DEFAULT_PERIOD = NClients::EGolovanPeriod::FIVE_MIN;

TString THorizontalPodAutoscalerManager::GetObjectId() const {
    return HorizontalPodAutoscaler_.Meta().id();
}

void THorizontalPodAutoscalerManager::SetIntGaugeSensor(const TString& nameSuffix, ui64 value) const {
    NON_STATIC_INFRA_INT_GAUGE_SENSOR(
        HORIZONTAL_POD_AUTOSCALER_SENSOR_GROUP
        , TStringBuilder{} << HorizontalPodAutoscaler_.Meta().id() << "." << nameSuffix
        , value
    );
}

void THorizontalPodAutoscalerManager::SetGaugeSensor(const TString& nameSuffix, double value) const {
    NON_STATIC_INFRA_GAUGE_SENSOR(
        HORIZONTAL_POD_AUTOSCALER_SENSOR_GROUP
        , TStringBuilder{} << HorizontalPodAutoscaler_.Meta().id() << "." << nameSuffix
        , value
    );
}

TVector<NController::ISingleClusterObjectManager::TSelectArgument> THorizontalPodAutoscalerManager::GetDependentObjectsSelectArguments() const {
    return {{
        NYP::NClient::NApi::NProto::OT_REPLICA_SET
        , {
            "/meta/id"
            , "/spec/replica_count"
            , "/spec/pod_template_spec/spec/host_infra/monitoring/labels/stage"
            , "/spec/pod_template_spec/spec/host_infra/monitoring/labels/deploy_unit"
            , "/status/deploy_status/details/current_revision_progress/pods_total"
            , "/status/ready_condition"
            , "/labels/deploy/horizontal_pod_autoscaler_id"
        } /* selector */
        , TStringBuilder() << "[/meta/id] = '" << HorizontalPodAutoscaler_.Meta().replica_set_id() << "'" /* filter */
        , {} /* options */
    }};
}

NController::ISingleClusterObjectManager::TSelectArgument THorizontalPodAutoscalerManagerFactory::GetSelectArgument(const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */, NInfra::TLogFramePtr) const {
    NYP::NClient::TSelectObjectsOptions options;
    options.SetFetchTimestamps(true);
    return {
        NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER
        , {
            "/meta/id"
            , "/meta/replica_set_id"
            , "/spec"
            , "/status"
        }    /* selector */
        , "" /* filter   */
        , options /* options  */
    };
}

bool THorizontalPodAutoscalerManager::HpaValidate(
    const TInstant& nowTime
    , bool isDryRunMode
    , TLogFramePtr frame
    , TDuration& upscaleDelay
    , TDuration& downscaleDelay
    , NYP::NClient::NApi::NProto::THorizontalPodAutoscalerStatus& hpaStatus
) const {
    if (upscaleDelay == TDuration::Zero()) {
        upscaleDelay = DEFAULT_UPSCALE_DELAY;
    }

    if (downscaleDelay == TDuration::Zero()) {
        downscaleDelay = DEFAULT_DOWNSCALE_DELAY;
    }

    if (upscaleDelay < MIN_SCALE_DELAY) {
        ChangeStatusToFailed(
            HorizontalPodAutoscaler_.Meta().id()
            , "INVALID_UPSCALE_DELAY"
            , "upscale delay less 5 min"
            , nowTime
            , isDryRunMode
            , frame
            , hpaStatus
        );
        return false;
    }

    if (downscaleDelay < MIN_SCALE_DELAY) {
        ChangeStatusToFailed(
            HorizontalPodAutoscaler_.Meta().id()
            , "INVALID_DOWNSCALE_DELAY"
            , "downscale delay less 5 min"
            , nowTime
            , isDryRunMode
            , frame
            , hpaStatus
        );
        return false;
    }

    if (HorizontalPodAutoscaler_.Spec().replica_set().has_cpu()) {
        if (HorizontalPodAutoscaler_.Spec().replica_set().cpu().lower_bound() >
                HorizontalPodAutoscaler_.Spec().replica_set().cpu().upper_bound()) {
            ChangeStatusToFailed(
                HorizontalPodAutoscaler_.Meta().id()
                , "INVALID_CPU_METRIC_BOUND"
                , "cpu_lower_bound greater cpu_upper_bound"
                , nowTime
                , isDryRunMode
                , frame
                , hpaStatus
            );
            return false;
        }
    } else if (HorizontalPodAutoscaler_.Spec().replica_set().has_cpu_utilization()) {
        if (HorizontalPodAutoscaler_.Spec().replica_set().cpu_utilization().lower_bound_percent() >
                HorizontalPodAutoscaler_.Spec().replica_set().cpu_utilization().upper_bound_percent()) {
            ChangeStatusToFailed(
                HorizontalPodAutoscaler_.Meta().id()
                , "INVALID_CPU_METRIC_BOUND"
                , "cpu_lower_bound_percent greater cpu_upper_bound_percent"
                , nowTime
                , isDryRunMode
                , frame
                , hpaStatus
            );
            return false;
        } else if (HorizontalPodAutoscaler_.Spec().replica_set().cpu_utilization().upper_bound_percent() > 100) {
            ChangeStatusToFailed(
                HorizontalPodAutoscaler_.Meta().id()
                , "INVALID_CPU_METRIC_BOUND"
                , "cpu_upper_bound_percent greater than 100 is not supported"
                , nowTime
                , isDryRunMode
                , frame
                , hpaStatus
            );
            return false;
        } else if (HorizontalPodAutoscaler_.Spec().replica_set().cpu_utilization().lower_bound_percent() < 0) {
            ChangeStatusToFailed(
                HorizontalPodAutoscaler_.Meta().id()
                , "INVALID_CPU_METRIC_BOUND"
                , "cpu_lower_bound_percent less than 0 is not supported"
                , nowTime
                , isDryRunMode
                , frame
                , hpaStatus
            );
            return false;
        }

        switch (HorizontalPodAutoscaler_.Spec().replica_set().cpu_utilization().period().seconds()) {
            case 0:
            case 5:
            case 5 * 60:
            case 60 * 60:
            case 3 * 60 * 60:
            case 6 * 60 * 60:
            case 12 * 60 * 60:
            case 24 * 60 * 60:
                break;
            default:
                ChangeStatusToFailed(
                    HorizontalPodAutoscaler_.Meta().id()
                    , "INVALID_GOLOVAN_PERIOD"
                    , TStringBuilder() << "Not supported " << HorizontalPodAutoscaler_.Spec().replica_set().cpu_utilization().period().seconds() << " seconds golovan period"
                    , nowTime
                    , isDryRunMode
                    , frame
                    , hpaStatus
                );
                return false;
        }

    } else if (HorizontalPodAutoscaler_.Spec().replica_set().has_custom()) {
        if (HorizontalPodAutoscaler_.Spec().replica_set().custom().lower_bound() >
                HorizontalPodAutoscaler_.Spec().replica_set().custom().upper_bound()) {
            ChangeStatusToFailed(
                HorizontalPodAutoscaler_.Meta().id()
                , "INVALID_CUSTOM_METRIC_BOUND"
                , "custom_lower_bound greater custom_upper_bound"
                , nowTime
                , isDryRunMode
                , frame
                , hpaStatus
            );
            return false;
        }
    } else {
        ChangeStatusToFailed(
        HorizontalPodAutoscaler_.Meta().id()
            , "NOT_FOUND_METRIC_BOUND"
            , "hpa spec.replica_set must be contains field cpu or cpu_utilization or custom"
            , nowTime
            , isDryRunMode
            , frame
            , hpaStatus
        );
        return false;
    }

    if (HorizontalPodAutoscaler_.Spec().replica_set().min_replicas() == 0 &&
            HorizontalPodAutoscaler_.Spec().replica_set().max_replicas() != 0) {
        ChangeStatusToFailed(
            HorizontalPodAutoscaler_.Meta().id()
            , "INVALID_MIN_REPLICAS"
            , "min_replicas zero, but max_replicas not zero"
            , nowTime
            , isDryRunMode
            , frame
            , hpaStatus
        );
        return false;
    }

    if (HorizontalPodAutoscaler_.Spec().replica_set().min_replicas() >
            HorizontalPodAutoscaler_.Spec().replica_set().max_replicas()) {
        ChangeStatusToFailed(
            HorizontalPodAutoscaler_.Meta().id()
            , "INVALID_MIN_MAX_REPLICAS"
            , "min_replicas greater than max_replicas"
            , nowTime
            , isDryRunMode
            , frame
            , hpaStatus
        );
        return false;
    }

    return true;
}

double THorizontalPodAutoscalerManager::GetCurrentMetric(
    const NYP::NClient::TReplicaSet& replicaSet
    , NInfra::TLogFramePtr frame
) const {
     if (HorizontalPodAutoscaler_.Spec().replica_set().has_custom() && HorizontalPodAutoscaler_.Spec().replica_set().custom().has_solomon_signal()) {
        const auto& solomonSignal = HorizontalPodAutoscaler_.Spec().replica_set().custom().solomon_signal();
        auto request = SolomonClient_->GetSingleStat()
            .SetProject(solomonSignal.project())
            .SetService(solomonSignal.service())
            .SetCluster(solomonSignal.cluster())
            .SetHost(solomonSignal.host())
            .SetPeriod(ProtoDurationToSolomonPeriod(solomonSignal.period()))
            .SetAggregationType(ProtoAggregationTypeToString(solomonSignal.aggregation()));

        for (const auto& [key, value] : solomonSignal.advanced_attributes()) {
            request.AddAdvancedAttribute(key, value);
        }

        if (solomonSignal.Hassensor()) {
            request.SetSensor(solomonSignal.sensor());
        } else {
            request.SetPath(solomonSignal.path());
        }

        return request.Execute(frame);

    }

    auto request = GolovanClient_->GetSingleStat();
    if (HorizontalPodAutoscaler_.Spec().replica_set().has_cpu() || HorizontalPodAutoscaler_.Spec().replica_set().has_cpu_utilization()) {
        request
            .SetTag("itype", "deploy")
            .SetTag("stage", replicaSet.Spec().pod_template_spec().spec().host_infra().monitoring().labels().at("stage"))
            .SetTag("deploy_unit", replicaSet.Spec().pod_template_spec().spec().host_infra().monitoring().labels().at("deploy_unit"))
            .SetTag("geo", GeoTag_)
            .SetHost("ASEARCH");

        if (HorizontalPodAutoscaler_.Spec().replica_set().has_cpu()){
            request
                .SetSignal("havg(portoinst-cpu_usage_slot_hgram)")
                .SetPeriod(DEFAULT_PERIOD);
        } else {
            request
                .SetSignal("havg(portoinst-cpu_limit_usage_perc_hgram)")
                .SetPeriod(ProtoDurationToGolovanPeriod(HorizontalPodAutoscaler_.Spec().replica_set().cpu_utilization().period()));
        }

    } else {
        const auto& golovanSignal = HorizontalPodAutoscaler_.Spec().replica_set().custom().golovan_signal();
        request.SetSignal(golovanSignal.signal());
        for (const auto& [key, value] : golovanSignal.tags()) {
            request.SetTag(key, value);
        }
        request.SetPeriod(ProtoPeriodToGolovanPeriod(golovanSignal.period()));
        if (golovanSignal.set_local_geo_tag()) {
            request.SetTag("geo", GeoTag_);
        }
        request.SetHost(golovanSignal.host());
    }

    return request.Execute(frame);
}

std::pair<double, double> THorizontalPodAutoscalerManager::GetBounds() const {
    if (HorizontalPodAutoscaler_.Spec().replica_set().has_cpu()) {
        return {
            HorizontalPodAutoscaler_.Spec().replica_set().cpu().lower_bound()
            , HorizontalPodAutoscaler_.Spec().replica_set().cpu().upper_bound()
        };
    }

    if (HorizontalPodAutoscaler_.Spec().replica_set().has_cpu_utilization()) {
        return {
            HorizontalPodAutoscaler_.Spec().replica_set().cpu_utilization().lower_bound_percent()
            , HorizontalPodAutoscaler_.Spec().replica_set().cpu_utilization().upper_bound_percent()
        };
    }

    return {
        HorizontalPodAutoscaler_.Spec().replica_set().custom().lower_bound()
        , HorizontalPodAutoscaler_.Spec().replica_set().custom().upper_bound()
    };
}

void THorizontalPodAutoscalerManager::GenerateYpUpdates(
    const ISingleClusterObjectManager::TDependentObjects& dependentObjects
    , TVector<ISingleClusterObjectManager::TRequest>& requests
    , TLogFramePtr frame
) const {
    auto replicaSet = FillReplicaSet(dependentObjects.SelectedObjects[0]->Results[0]);
    auto hpaStatus = HorizontalPodAutoscaler_.Status();

    TInstant nowTime = Now();

    bool isDryRunMode = (replicaSet.Labels()["deploy"]["horizontal_pod_autoscaler_id"] != HorizontalPodAutoscaler_.Meta().id());

    TDuration upscaleDelay = NProtoInterop::CastFromProto(HorizontalPodAutoscaler_.Spec().replica_set().upscale_delay());
    TDuration downscaleDelay = NProtoInterop::CastFromProto(HorizontalPodAutoscaler_.Spec().replica_set().downscale_delay());

    if (!HpaValidate(
            nowTime
            , isDryRunMode
            , frame
            , upscaleDelay
            , downscaleDelay
            , hpaStatus
        )
    ) {
        requests.emplace_back(
            NYP::NClient::TUpdateRequest(
                NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER
                , HorizontalPodAutoscaler_.Meta().id()
                , {
                    NYP::NClient::TSetRequest("/status/ready", hpaStatus.ready())
                    , NYP::NClient::TSetRequest("/status/in_progress", hpaStatus.in_progress())
                    , NYP::NClient::TSetRequest("/status/failed", hpaStatus.failed())
                    , NYP::NClient::TSetRequest("/status/spec_timestamp", YpTimestamp_)
                }
                , {}
            )
        );

        STATIC_INFRA_RATE_SENSOR(
            HORIZONTAL_POD_AUTOSCALER_SENSOR_GROUP
            , "validate_failed_hpa"
        );
        return;
    }

    TMaybe<double> currentMetric;
    try {
        currentMetric = GetCurrentMetric(replicaSet, frame);
    } catch (...) {
        TString reason = "METRIC_FETCH_FAILED";
        TString message(TStringBuilder() << "Mertic fetch failed with error: '" << CurrentExceptionMessage() << "'");

        ChangeStatusToFailed(
            HorizontalPodAutoscaler_.Meta().id()
            , reason
            , message
            , nowTime
            , isDryRunMode
            , frame
            , hpaStatus
        );

        STATIC_INFRA_RATE_SENSOR(
            HORIZONTAL_POD_AUTOSCALER_SENSOR_GROUP
            , "metric_fetch_failed_hpa"
        );

        currentMetric = Nothing();
    }

    auto [lowerBound, upperBound] = GetBounds();

    double targetMetric = (lowerBound + upperBound) / 2.0;

    const ui64 currentReplicas =
        isDryRunMode
        ? (hpaStatus.replica_set().current_replicas() ? hpaStatus.replica_set().current_replicas() : replicaSet.Spec().replica_count())
        : replicaSet.Spec().replica_count();
    ui64 targetReplicas = currentReplicas;

    TVector<NYP::NClient::TSetRequest> setRequestsForHPA;
    bool updateReplicaSet = false;

    if (isDryRunMode && currentMetric.Defined()) {
        if (currentReplicas) {
            currentMetric = currentMetric.GetRef() * (double)replicaSet.Spec().replica_count() / currentReplicas;
        } else {
            currentMetric = 0;
        }
    }

    if (currentMetric.Defined() && (currentMetric < lowerBound || currentMetric > upperBound)) {
        double scaleCoefficient = targetMetric ? currentMetric.GetRef() / targetMetric : 0;
        targetReplicas = ceil(currentReplicas * scaleCoefficient);
    }

    if (targetReplicas < HorizontalPodAutoscaler_.Spec().replica_set().min_replicas()) {
        targetReplicas = HorizontalPodAutoscaler_.Spec().replica_set().min_replicas();
    }

    if (targetReplicas > HorizontalPodAutoscaler_.Spec().replica_set().max_replicas()) {
        targetReplicas = HorizontalPodAutoscaler_.Spec().replica_set().max_replicas();
    }

    bool canIgnoreScaleDelay = currentReplicas < HorizontalPodAutoscaler_.Spec().replica_set().min_replicas() ||
                                currentReplicas > HorizontalPodAutoscaler_.Spec().replica_set().max_replicas() ;
    if (targetReplicas != currentReplicas) {
        if (targetReplicas > currentReplicas) {
            if (
                nowTime - NProtoInterop::CastFromProto(HorizontalPodAutoscaler_.Status().replica_set().last_upscale_time()) > upscaleDelay
                || canIgnoreScaleDelay
            ) {
                updateReplicaSet = true;
                setRequestsForHPA.emplace_back(
                    NYP::NClient::TSetRequest("/status/replica_set/last_upscale_time", NProtoInterop::CastToProto(nowTime))
                );
            }
        } else {
            if (
                nowTime - NProtoInterop::CastFromProto(HorizontalPodAutoscaler_.Status().replica_set().last_downscale_time()) > downscaleDelay
                || canIgnoreScaleDelay
            ) {
                updateReplicaSet =  true;
                setRequestsForHPA.emplace_back(
                    NYP::NClient::TSetRequest("/status/replica_set/last_downscale_time", NProtoInterop::CastToProto(nowTime))
                );
            }
        }

        if (updateReplicaSet) {
            frame->LogEvent(
                ELogPriority::TLOG_DEBUG
                , NLogEvent::TUpdateReplicaSet(
                    replicaSet.Meta().id()
                    , HorizontalPodAutoscaler_.Meta().id()
                    , targetReplicas > currentReplicas ? NLogEvent::TUpdateReplicaSet::UPSCALE
                                                       : NLogEvent::TUpdateReplicaSet::DOWNSCALE
                    , targetReplicas
                    , isDryRunMode
                )
            );

            if (!isDryRunMode) {
                requests.emplace_back(
                    NYP::NClient::TUpdateRequest(
                        NYP::NClient::NApi::NProto::OT_REPLICA_SET
                        , replicaSet.Meta().id()
                        , {
                            NYP::NClient::TSetRequest("/spec/replica_count", targetReplicas)
                        }
                        , {}
                    )
                );
            }
        }
    }

    if (updateReplicaSet && isDryRunMode) {
        if (currentMetric.Defined()) {
            if (targetReplicas) {
                // real_value * (replicaSet.Spec().replica_count() / currentReplicas) * (currentReplicas / targetReplicas) =
                // real_value * (replicaSet.Spec().replica_count() / targetReplicas)
                currentMetric = currentMetric.GetRef() * (double)currentReplicas / targetReplicas;
            } else {
                currentMetric = 0;
            }
        }
        SetIntGaugeSensor("current_replicas", targetReplicas);
        setRequestsForHPA.emplace_back(
            NYP::NClient::TSetRequest("/status/replica_set/current_replicas", targetReplicas)
        );
    } else {
        SetIntGaugeSensor("current_replicas", currentReplicas);
        setRequestsForHPA.emplace_back(
            NYP::NClient::TSetRequest("/status/replica_set/current_replicas", currentReplicas)
        );
    }

    SetIntGaugeSensor("desired_replicas", targetReplicas);
    setRequestsForHPA.emplace_back(
        NYP::NClient::TSetRequest("/status/replica_set/desired_replicas", targetReplicas)
    );

    if (currentMetric.Defined()) {
        SetGaugeSensor("metric_value", currentMetric.GetRef());
        setRequestsForHPA.emplace_back(
            NYP::NClient::TSetRequest("/status/replica_set/metric_value", currentMetric.GetRef())
        );

        bool hpaReady;
        TString message;
        TString reason;
        if (updateReplicaSet) {
            if (isDryRunMode) {
                hpaReady = true;
                message = "Dry run mode";
                reason = "DRY_RUN_MODE";
            } else {
                hpaReady = false;
                if (targetReplicas > currentReplicas) {
                    message = "Upscale in progress";
                } else {
                    message = "Downscale in progress";
                }
                reason = "CURRENT_REPLICAS_NOT_EQUALS_DESIRED_REPLICAS";
            }
        } else {
            if (targetReplicas == currentReplicas) {
                if (isDryRunMode) {
                    hpaReady = true;
                    message = "Dry run mode";
                    reason = "DRY_RUN_MODE";
                } else if (
                    replicaSet.Status().ready_condition().status() == NYP::NClient::NApi::NProto::EConditionStatus::CS_TRUE &&
                    replicaSet.Status().deploy_status().details().current_revision_progress().pods_total() == currentReplicas
                ) {
                    hpaReady = true;
                    message = "Successful update of replica_count";
                    reason = "CURRENT_REPLICAS_EQUALS_DESIRED_REPLICAS";
                } else {
                    hpaReady = false;
                    message = "replica_set not_ready";
                    reason = "REPLICA_SET_NOT_READY";
                }
            } else {
                hpaReady = false;
                if (targetReplicas > currentReplicas) {
                    message = "Waiting upscale delay";
                } else {
                    message = "Waiting downscale delay";
                }
                reason = "CURRENT_REPLICAS_NOT_EQUALS_DESIRED_REPLICAS";
            }
        }

        if (hpaReady) {
            if (hpaStatus.ready().status() != NYP::NClient::NApi::NProto::EConditionStatus::CS_TRUE) {
                frame->LogEvent(
                    ELogPriority::TLOG_DEBUG
                    , NLogEvent::THorizontalPodAutoscalerStatus(
                        HorizontalPodAutoscaler_.Meta().id()
                        , NLogEvent::THorizontalPodAutoscalerStatus::READY
                        , reason
                        , currentReplicas
                        , targetReplicas
                        , currentMetric.GetRef()
                        , isDryRunMode
                    )
                );
            }
        } else {
            if (hpaStatus.in_progress().status() != NYP::NClient::NApi::NProto::EConditionStatus::CS_TRUE) {
                frame->LogEvent(
                    ELogPriority::TLOG_DEBUG
                    , NLogEvent::THorizontalPodAutoscalerStatus(
                        HorizontalPodAutoscaler_.Meta().id()
                        , NLogEvent::THorizontalPodAutoscalerStatus::IN_PROGRESS
                        , reason
                        , currentReplicas
                        , targetReplicas
                        , currentMetric.GetRef()
                        , isDryRunMode
                    )
                );
            }
        }

        TryUpdateCondition(
            hpaStatus.mutable_ready()
            , hpaReady ? NYP::NClient::NApi::NProto::EConditionStatus::CS_TRUE
                       : NYP::NClient::NApi::NProto::EConditionStatus::CS_FALSE
            , reason
            , message
            , nowTime
        );

        TryUpdateCondition(
            hpaStatus.mutable_in_progress()
            , hpaReady ? NYP::NClient::NApi::NProto::EConditionStatus::CS_FALSE
                       : NYP::NClient::NApi::NProto::EConditionStatus::CS_TRUE
            , reason
            , message
            , nowTime
        );

        TryUpdateCondition(
            hpaStatus.mutable_failed()
            , NYP::NClient::NApi::NProto::EConditionStatus::CS_FALSE
            , reason
            , message
            , nowTime
        );
    }

    setRequestsForHPA.emplace_back(
        NYP::NClient::TSetRequest("/status/ready", hpaStatus.ready())
    );
    setRequestsForHPA.emplace_back(
        NYP::NClient::TSetRequest("/status/in_progress", hpaStatus.in_progress())
    );
    setRequestsForHPA.emplace_back(
        NYP::NClient::TSetRequest("/status/failed", hpaStatus.failed())
    );

    setRequestsForHPA.emplace_back(
        NYP::NClient::TSetRequest("/status/spec_timestamp", YpTimestamp_)
    );

    requests.emplace_back(
        NYP::NClient::TUpdateRequest(
            NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER
            , HorizontalPodAutoscaler_.Meta().id()
            , setRequestsForHPA
            , {}
        )
    );

    return;
}

NYP::NClient::TReplicaSet THorizontalPodAutoscalerManager::FillReplicaSet(
    const NController::TSelectorResultPtr& selectResult
) const {
    NYP::NClient::TReplicaSet replicaSet;
    ui64 replicaCount;
    ui64 podsTotal;
    TString stage;
    TString deployUnit;
    TString horizontalPodAutoscalerId;
    selectResult->Fill(
        replicaSet.MutableMeta()->mutable_id()
        , &replicaCount
        , &stage
        , &deployUnit
        , &podsTotal
        , replicaSet.MutableStatus()->mutable_ready_condition()
        , &horizontalPodAutoscalerId
    );
    replicaSet.MutableSpec()->set_replica_count(replicaCount);
    replicaSet.MutableSpec()->mutable_pod_template_spec()->mutable_spec()->mutable_host_infra()->mutable_monitoring()->mutable_labels()->insert({"stage", stage});
    replicaSet.MutableSpec()->mutable_pod_template_spec()->mutable_spec()->mutable_host_infra()->mutable_monitoring()->mutable_labels()->insert({"deploy_unit", deployUnit});
    replicaSet.MutableStatus()->mutable_deploy_status()->mutable_details()->mutable_current_revision_progress()->set_pods_total(podsTotal);
    (*replicaSet.MutableLabels())["deploy"]["horizontal_pod_autoscaler_id"] = horizontalPodAutoscalerId;
    return replicaSet;
}

TExpected<NController::TSingleClusterObjectManagerPtr, THorizontalPodAutoscalerManagerFactory::TValidationError> THorizontalPodAutoscalerManagerFactory::GetSingleClusterObjectManager(
    const NController::TSelectorResultPtr& selectorResultPtr
    , TLogFramePtr /* frame */
) const {
    NYP::NClient::THorizontalPodAutoscaler horizontalPodAutoscaler;
    selectorResultPtr->Fill(
        horizontalPodAutoscaler.MutableMeta()->mutable_id()
        , horizontalPodAutoscaler.MutableMeta()->mutable_replica_set_id()
        , horizontalPodAutoscaler.MutableSpec()
        , horizontalPodAutoscaler.MutableStatus()
    );

    return NController::TSingleClusterObjectManagerPtr(
        new THorizontalPodAutoscalerManager(
            horizontalPodAutoscaler
            , GolovanClient_
            , SolomonClient_
            , GeoTag_
            , selectorResultPtr->Values().timestamps(2)
        )
    );
}

} // namespace NInfra::NHorizontalPodAutoscalerController
