#include "hpa_manager.h"

#include <infra/libs/clients/http_executer/http_executer_mock.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NHorizontalPodAutoscalerController {

namespace {


enum EStatusType {
    READY
    , IN_PROGRESS
    , FAILED
};

const ui32 SCALE_TIME_SECONDS = 300;

const ui64 SPEC_TIMESTAMP = 1337;

const TString SOLOMON_RESPONSE = "{\"scalar\":42.0}";

const TString SOLOMON_FAILED_RESPONSE = "{\"scalar\":Nan}";

template<class T>
TString ToYson(const T& s) {
    return NYT::NYson::ConvertToYsonString(s).ToString();
}

google::protobuf::Timestamp ToTimestamp(const TInstant& time) {
    google::protobuf::Timestamp timestamp;
    timestamp.set_seconds(time.Seconds());
    timestamp.set_nanos(time.NanoSecondsOfSecond());
    return timestamp;
}

NYP::NClient::NApi::NProto::THorizontalPodAutoscalerStatus GetDefaultHPAStatus(const TInstant& nowTime) {
    NYP::NClient::NApi::NProto::THorizontalPodAutoscalerStatus hpaStatus;

    hpaStatus.mutable_ready()->set_reason("DEFAULT");
    hpaStatus.mutable_ready()->set_message("Test");
    hpaStatus.mutable_ready()->set_status(NYP::NClient::NApi::NProto::EConditionStatus::CS_TRUE);
    *(hpaStatus.mutable_ready()->mutable_last_transition_time()) = ToTimestamp(nowTime);

    hpaStatus.mutable_in_progress()->set_reason("DEFAULT");
    hpaStatus.mutable_in_progress()->set_message("Test");
    hpaStatus.mutable_in_progress()->set_status(NYP::NClient::NApi::NProto::EConditionStatus::CS_FALSE);
    *(hpaStatus.mutable_in_progress()->mutable_last_transition_time()) = ToTimestamp(nowTime);

    hpaStatus.mutable_failed()->set_reason("DEFAULT");
    hpaStatus.mutable_failed()->set_message("Test");
    hpaStatus.mutable_failed()->set_status(NYP::NClient::NApi::NProto::EConditionStatus::CS_FALSE);
    *(hpaStatus.mutable_failed()->mutable_last_transition_time()) = ToTimestamp(nowTime);

    hpaStatus.mutable_replica_set()->set_current_replicas(0);
    hpaStatus.mutable_replica_set()->set_desired_replicas(0);
    hpaStatus.mutable_replica_set()->set_metric_value(42);
    *(hpaStatus.mutable_replica_set()->mutable_last_upscale_time()) = ToTimestamp(nowTime - TDuration::Seconds(SCALE_TIME_SECONDS));
    *(hpaStatus.mutable_replica_set()->mutable_last_downscale_time()) = ToTimestamp(nowTime - TDuration::Seconds(SCALE_TIME_SECONDS));

    return hpaStatus;
}

NJson::TJsonValue ToJson(const TString& val) {
    TStringStream buf;
    NJson2Yson::ConvertYson2Json(val, &buf);
    return NJson::ReadJsonTree(&buf, true);
}


NYP::NClient::NApi::NProto::TAttributeList GetReplicaSetAtributeList(
    ui32 replicaCount
    , ui32 podsTotal
    , bool readyStatus
    , const TString& hpaId
) {
    NYP::NClient::NApi::NProto::TAttributeList dependentObjectsAtrributeList;
    dependentObjectsAtrributeList.add_value_payloads()->set_yson(ToYson("test_replica_set_id"));
    dependentObjectsAtrributeList.add_value_payloads()->set_yson(ToYson(replicaCount));
    dependentObjectsAtrributeList.add_value_payloads()->set_yson(ToYson("test_stage"));
    dependentObjectsAtrributeList.add_value_payloads()->set_yson(ToYson("DeployUnit1"));
    dependentObjectsAtrributeList.add_value_payloads()->set_yson(ToYson(podsTotal));

    NYP::NClient::NApi::NProto::TCondition replicaSetReady;
    replicaSetReady.set_status(
        readyStatus ? NYP::NClient::NApi::NProto::EConditionStatus::CS_TRUE
                    : NYP::NClient::NApi::NProto::EConditionStatus::CS_FALSE
    );
    dependentObjectsAtrributeList.add_value_payloads()->set_yson(NYT::NYson::ConvertToYsonString(replicaSetReady).ToString());
    dependentObjectsAtrributeList.add_value_payloads()->set_yson(ToYson(hpaId));

    return dependentObjectsAtrributeList;
}

void CheckCondition(
    const TString& condition
    , bool status
    , const TString& expectedReason
) {
    NJson::TJsonValue json = ToJson(condition);
    UNIT_ASSERT_EQUAL_C(json["status"].GetString() == "true", status, json["status"].GetString());
    UNIT_ASSERT_STRINGS_EQUAL(json["reason"].GetString(), expectedReason);
}

void CheckStatus(
    const NYP::NClient::TSetRequest& ready
    , const NYP::NClient::TSetRequest& inProgress
    , const NYP::NClient::TSetRequest& failed
    , const TString& expectedReason
    , EStatusType expectedStatus
) {
    UNIT_ASSERT_STRINGS_EQUAL(ready.GetPath(), "/status/ready");
    CheckCondition(ready.GetValue(), expectedStatus == EStatusType::READY, expectedReason);
    UNIT_ASSERT_STRINGS_EQUAL(inProgress.GetPath(), "/status/in_progress");
    CheckCondition(inProgress.GetValue(), expectedStatus == EStatusType::IN_PROGRESS, expectedReason);
    UNIT_ASSERT_STRINGS_EQUAL(failed.GetPath(), "/status/failed");
    CheckCondition(failed.GetValue(), expectedStatus == EStatusType::FAILED, expectedReason);
}

void CheckIntegerUpdate(
    const NYP::NClient::TSetRequest& update
    , const TString& path
    , ui32 value
) {
    UNIT_ASSERT_STRINGS_EQUAL(update.GetPath(), path);
    NJson::TJsonValue json = ToJson(update.GetValue());
    UNIT_ASSERT_EQUAL_C(json.GetUInteger(), value, json.GetUInteger());
}

void CheckDoubleUpdate(
    const NYP::NClient::TSetRequest& update
    , const TString& path
    , double value
) {
    UNIT_ASSERT_STRINGS_EQUAL(update.GetPath(), path);
    NJson::TJsonValue json = ToJson(update.GetValue());
    UNIT_ASSERT_DOUBLES_EQUAL_C(json.GetDouble(), value, 0.001, json.GetDouble());
}
void CheckScaleTime(
    const NYP::NClient::TSetRequest& update
    , const TString& path
    , int scaleTimeInSeconds
) {
    UNIT_ASSERT_STRINGS_EQUAL(update.GetPath(), path);
    NJson::TJsonValue ready = ToJson(update.GetValue());
    TInstant lastUpscaleTime = TInstant::Seconds(ready["seconds"].GetInteger());
    UNIT_ASSERT(lastUpscaleTime >= TInstant::Now() - TDuration::Seconds(scaleTimeInSeconds / 2));
}

TLogger logger({});

void GenerateYpUpdates(
    const NController::TSingleClusterObjectManagerPtr manager
    , const NController::ISingleClusterObjectManager::TDependentObjects& dependentObjects
    , TVector<NYP::NClient::TCreateObjectRequest>& create
    , TVector<NYP::NClient::TRemoveObjectRequest>& remove
    , TVector<NYP::NClient::TUpdateRequest>& update
) {
    TVector<NController::ISingleClusterObjectManager::TRequest> requests;
    manager->GenerateYpUpdates(dependentObjects, requests, logger.SpawnFrame());

    for (auto& req : requests) {
        if (std::holds_alternative<NYP::NClient::TCreateObjectRequest>(req)) {
            create.emplace_back(std::move(std::get<NYP::NClient::TCreateObjectRequest>(req)));
        } else if (std::holds_alternative<NYP::NClient::TRemoveObjectRequest>(req)) {
            remove.emplace_back(std::move(std::get<NYP::NClient::TRemoveObjectRequest>(req)));
        } else if (std::holds_alternative<NYP::NClient::TUpdateRequest>(req)) {
            update.emplace_back(std::move(std::get<NYP::NClient::TUpdateRequest>(req)));
        }
    }
}

NController::TSelectObjectsResultPtr CreateSelectObjectsResultPtr(const TVector<NController::TSelectorResultPtr>& selectorResults) {
    return MakeAtomicShared<NController::TSelectObjectsResult>(selectorResults);
}

} // namespace

class THorizontalPodAutoscalerBaseTestCase : public NUnitTest::TBaseTestCase {
public:
    void TestFactorySelectArgument() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto select = factory.GetSelectArgument();
        UNIT_ASSERT_EQUAL(NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER, select.ObjectType);
        UNIT_ASSERT_EQUAL_C(4, select.Selector.size(), select.Selector.size());
        UNIT_ASSERT_STRINGS_EQUAL("/meta/id", select.Selector[0]);
        UNIT_ASSERT_STRINGS_EQUAL("/meta/replica_set_id", select.Selector[1]);
        UNIT_ASSERT_STRINGS_EQUAL("/spec", select.Selector[2]);
        UNIT_ASSERT_STRINGS_EQUAL("/status", select.Selector[3]);
    }

    void TestManagerDependentObjects() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                40                  /* lowerBound */
                , 50                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Zero() /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjects = manager->GetDependentObjectsSelectArguments()[0];
        UNIT_ASSERT_EQUAL(NYP::NClient::NApi::NProto::OT_REPLICA_SET, dependentObjects.ObjectType);

        UNIT_ASSERT_EQUAL_C(7, dependentObjects.Selector.size(), dependentObjects.Selector.size());
        UNIT_ASSERT_STRINGS_EQUAL("/meta/id", dependentObjects.Selector[0]);
        UNIT_ASSERT_STRINGS_EQUAL("/spec/replica_count", dependentObjects.Selector[1]);
        UNIT_ASSERT_STRINGS_EQUAL("/spec/pod_template_spec/spec/host_infra/monitoring/labels/stage", dependentObjects.Selector[2]);
        UNIT_ASSERT_STRINGS_EQUAL("/spec/pod_template_spec/spec/host_infra/monitoring/labels/deploy_unit", dependentObjects.Selector[3]);
        UNIT_ASSERT_STRINGS_EQUAL("/status/deploy_status/details/current_revision_progress/pods_total", dependentObjects.Selector[4]);
        UNIT_ASSERT_STRINGS_EQUAL("/status/ready_condition", dependentObjects.Selector[5]);
        UNIT_ASSERT_STRINGS_EQUAL("/labels/deploy/horizontal_pod_autoscaler_id", dependentObjects.Selector[6]);

        UNIT_ASSERT_STRINGS_EQUAL(dependentObjects.Filter, "[/meta/id] = 'test_replica_set_id'");
    }

    void TestCurrentMetricInLimitsAndReplicaSetReady() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                40                  /* lowerBound */
                , 50                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Zero() /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(1, 1, true, "test_hpa_id"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 1, update.size());
        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 7, update[0].GetSetVec().size());

        CheckIntegerUpdate(
            update[0].GetSetVec()[0]
            , "/status/replica_set/current_replicas"
            , 1
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[1]
            , "/status/replica_set/desired_replicas"
            , 1
        );

        CheckDoubleUpdate(
            update[0].GetSetVec()[2]
            , "/status/replica_set/metric_value"
            , 42
        );

        CheckStatus(
            update[0].GetSetVec()[3]
            , update[0].GetSetVec()[4]
            , update[0].GetSetVec()[5]
            , "CURRENT_REPLICAS_EQUALS_DESIRED_REPLICAS"
            , EStatusType::READY
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[6]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

    void TestCurrentMetricInLimitsAndReplicaSetStatusNotReady() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                40                  /* lowerBound */
                , 50                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Zero() /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(1, 1, false, "test_hpa_id"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 1, update.size());
        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 7, update[0].GetSetVec().size());

        CheckIntegerUpdate(
            update[0].GetSetVec()[0]
            , "/status/replica_set/current_replicas"
            , 1
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[1]
            , "/status/replica_set/desired_replicas"
            , 1
        );

        CheckDoubleUpdate(
            update[0].GetSetVec()[2]
            , "/status/replica_set/metric_value"
            , 42
        );

        CheckStatus(
            update[0].GetSetVec()[3]
            , update[0].GetSetVec()[4]
            , update[0].GetSetVec()[5]
            , "REPLICA_SET_NOT_READY"
            , EStatusType::IN_PROGRESS
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[6]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

    void TestCurrentMetricInLimitsAndReplicaSetPodCountNotReady() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                40                  /* lowerBound */
                , 50                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Zero() /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(1, 1, false, "test_hpa_id"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 1, update.size());
        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 7, update[0].GetSetVec().size());

        CheckIntegerUpdate(
            update[0].GetSetVec()[0]
            , "/status/replica_set/current_replicas"
            , 1
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[1]
            , "/status/replica_set/desired_replicas"
            , 1
        );

        CheckDoubleUpdate(
            update[0].GetSetVec()[2]
            , "/status/replica_set/metric_value"
            , 42
        );

        CheckStatus(
            update[0].GetSetVec()[3]
            , update[0].GetSetVec()[4]
            , update[0].GetSetVec()[5]
            , "REPLICA_SET_NOT_READY"
            , EStatusType::IN_PROGRESS
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[6]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

    void TestCurrentMetricNotInLimitsAndReplicaSetUpdateUpscale() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                20                  /* lowerBound */
                , 22                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Seconds(SCALE_TIME_SECONDS) /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(2, 1, true, "test_hpa_id"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 2, update.size());

        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_REPLICA_SET);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 1, update[0].GetSetVec().size());
        CheckIntegerUpdate(
            update[0].GetSetVec()[0]
            , "/spec/replica_count"
            , 3
        );

        UNIT_ASSERT_EQUAL(update[1].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[1].GetSetVec().size(), 8, update[1].GetSetVec().size());

        CheckScaleTime(
            update[1].GetSetVec()[0]
            , "/status/replica_set/last_upscale_time"
            , SCALE_TIME_SECONDS
        );

        CheckIntegerUpdate(
            update[1].GetSetVec()[1]
            , "/status/replica_set/current_replicas"
            , 2
        );

        CheckIntegerUpdate(
            update[1].GetSetVec()[2]
            , "/status/replica_set/desired_replicas"
            , 3
        );

        CheckDoubleUpdate(
            update[1].GetSetVec()[3]
            , "/status/replica_set/metric_value"
            , 42
        );

        CheckStatus(
            update[1].GetSetVec()[4]
            , update[1].GetSetVec()[5]
            , update[1].GetSetVec()[6]
            , "CURRENT_REPLICAS_NOT_EQUALS_DESIRED_REPLICAS"
            , EStatusType::IN_PROGRESS
        );

        CheckIntegerUpdate(
            update[1].GetSetVec()[7]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

    void TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                20                  /* lowerBound */
                , 22                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Seconds(SCALE_TIME_SECONDS) /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(2, 1, true, "test_hpa_id1"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 1, update.size());

        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 8, update[1].GetSetVec().size());

        CheckScaleTime(
            update[0].GetSetVec()[0]
            , "/status/replica_set/last_upscale_time"
            , SCALE_TIME_SECONDS
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[1]
            , "/status/replica_set/current_replicas"
            , 3
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[2]
            , "/status/replica_set/desired_replicas"
            , 3
        );

        CheckDoubleUpdate(
            update[0].GetSetVec()[3]
            , "/status/replica_set/metric_value"
            , 28
        );

        CheckStatus(
            update[0].GetSetVec()[4]
            , update[0].GetSetVec()[5]
            , update[0].GetSetVec()[6]
            , "DRY_RUN_MODE"
            , EStatusType::READY
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[7]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

    void TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                20                  /* lowerBound */
                , 22                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Zero() /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(2, 1, true, "test_hpa_id"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 1, update.size());
        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 7, update[0].GetSetVec().size());

        CheckIntegerUpdate(
            update[0].GetSetVec()[0]
            , "/status/replica_set/current_replicas"
            , 2
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[1]
            , "/status/replica_set/desired_replicas"
            , 3
        );

        CheckDoubleUpdate(
            update[0].GetSetVec()[2]
            , "/status/replica_set/metric_value"
            , 42
        );

        CheckStatus(
            update[0].GetSetVec()[3]
            , update[0].GetSetVec()[4]
            , update[0].GetSetVec()[5]
            , "CURRENT_REPLICAS_NOT_EQUALS_DESIRED_REPLICAS"
            , EStatusType::IN_PROGRESS
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[6]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

    void TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscale() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                80                  /* lowerBound */
                , 88                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Seconds(SCALE_TIME_SECONDS) /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(2, 1, true, "test_hpa_id"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 2, update.size());

        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_REPLICA_SET);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 1, update[0].GetSetVec().size());
        CheckIntegerUpdate(
            update[0].GetSetVec()[0]
            , "/spec/replica_count"
            , 1
        );

        UNIT_ASSERT_EQUAL(update[1].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[1].GetSetVec().size(), 8, update[1].GetSetVec().size());

        CheckScaleTime(
            update[1].GetSetVec()[0]
            , "/status/replica_set/last_downscale_time"
            , SCALE_TIME_SECONDS
        );

        CheckIntegerUpdate(
            update[1].GetSetVec()[1]
            , "/status/replica_set/current_replicas"
            , 2
        );

        CheckIntegerUpdate(
            update[1].GetSetVec()[2]
            , "/status/replica_set/desired_replicas"
            , 1
        );

        CheckDoubleUpdate(
            update[1].GetSetVec()[3]
            , "/status/replica_set/metric_value"
            , 42
        );

        CheckStatus(
            update[1].GetSetVec()[4]
            , update[1].GetSetVec()[5]
            , update[1].GetSetVec()[6]
            , "CURRENT_REPLICAS_NOT_EQUALS_DESIRED_REPLICAS"
            , EStatusType::IN_PROGRESS
        );

        CheckIntegerUpdate(
            update[1].GetSetVec()[7]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

    void TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                80                  /* lowerBound */
                , 88                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Seconds(SCALE_TIME_SECONDS) /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(2, 1, true, "test_hpa_id1"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 1, update.size());

        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 8, update[0].GetSetVec().size());

        CheckScaleTime(
            update[0].GetSetVec()[0]
            , "/status/replica_set/last_downscale_time"
            , SCALE_TIME_SECONDS
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[1]
            , "/status/replica_set/current_replicas"
            , 1
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[2]
            , "/status/replica_set/desired_replicas"
            , 1
        );

        CheckDoubleUpdate(
            update[0].GetSetVec()[3]
            , "/status/replica_set/metric_value"
            , 84
        );

        CheckStatus(
            update[0].GetSetVec()[4]
            , update[0].GetSetVec()[5]
            , update[0].GetSetVec()[6]
            , "DRY_RUN_MODE"
            , EStatusType::READY
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[7]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

    void TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                80                  /* lowerBound */
                , 88                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Zero() /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(2, 1, true, "test_hpa_id"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 1, update.size());
        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 7, update[0].GetSetVec().size());

        CheckIntegerUpdate(
            update[0].GetSetVec()[0]
            , "/status/replica_set/current_replicas"
            , 2
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[1]
            , "/status/replica_set/desired_replicas"
            , 1
        );

        CheckDoubleUpdate(
            update[0].GetSetVec()[2]
            , "/status/replica_set/metric_value"
            , 42
        );

        CheckStatus(
            update[0].GetSetVec()[3]
            , update[0].GetSetVec()[4]
            , update[0].GetSetVec()[5]
            , "CURRENT_REPLICAS_NOT_EQUALS_DESIRED_REPLICAS"
            , EStatusType::IN_PROGRESS
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[6]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

    void TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow() {
        // In this test autoscaler wants to downscale replica_count to 1 but min_replicas is 2
        // So replica_count should not be updated
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponse()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                80                  /* lowerBound */
                , 88                /* upperBound */
                , 2                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Seconds(SCALE_TIME_SECONDS) /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(2, 1, true, "test_hpa_id"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 1, update.size());

        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 7, update[0].GetSetVec().size());

        CheckIntegerUpdate(
            update[0].GetSetVec()[0]
            , "/status/replica_set/current_replicas"
            , 2
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[1]
            , "/status/replica_set/desired_replicas"
            , 2
        );

        CheckDoubleUpdate(
            update[0].GetSetVec()[2]
            , "/status/replica_set/metric_value"
            , 42
        );

        CheckStatus(
            update[0].GetSetVec()[3]
            , update[0].GetSetVec()[4]
            , update[0].GetSetVec()[5]
            , "REPLICA_SET_NOT_READY"
            , EStatusType::IN_PROGRESS
        );

        CheckIntegerUpdate(
            update[0].GetSetVec()[6]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

    void TestMetricRequestFailed() {
        NController::TSharding shardFactory;
        THorizontalPodAutoscalerManagerFactory factory(
            new NClients::TGolovanClient(new NHttpExecuter::THttpExecuterMock(GetGolovanResponseFailed()))
            , new NClients::TSolomonClient(new NHttpExecuter::THttpExecuterMock(SOLOMON_FAILED_RESPONSE))
            , "cluster" /* geoTag */
            , shardFactory.GetShard(0)
        );

        auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
            GetHPAAtributeList(
                40                  /* lowerBound */
                , 50                /* upperBound */
                , 1                 /* minReplicas */
                , 3                 /* maxReplicas */
                , TInstant::Now()   /* nowTime */
                , TDuration::Zero() /* lastScaleShift */
            )
        );

        auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
        UNIT_ASSERT(result);
        auto manager = result.Success();

        auto dependentObjectsResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetReplicaSetAtributeList(228, 1, true, "test_hpa_id"));

        TVector<NYP::NClient::TCreateObjectRequest> create;
        TVector<NYP::NClient::TRemoveObjectRequest> remove;
        TVector<NYP::NClient::TUpdateRequest> update;
        GenerateYpUpdates(manager, {{CreateSelectObjectsResultPtr({dependentObjectsResult})}, {}, {}}, create, remove, update);

        UNIT_ASSERT(create.empty());
        UNIT_ASSERT(remove.empty());

        UNIT_ASSERT_EQUAL_C(update.size(), 2, update.size());

        UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_REPLICA_SET);
        UNIT_ASSERT_EQUAL_C(update[0].GetSetVec().size(), 1, update[0].GetSetVec().size());

        CheckIntegerUpdate(
            update[0].GetSetVec()[0]
            , "/spec/replica_count"
            , 3
        );

        UNIT_ASSERT_EQUAL(update[1].GetObjectType(), NYP::NClient::NApi::NProto::OT_HORIZONTAL_POD_AUTOSCALER);
        UNIT_ASSERT_EQUAL_C(update[1].GetSetVec().size(), 7, update[1].GetSetVec().size());

        CheckScaleTime(
            update[1].GetSetVec()[0]
            , "/status/replica_set/last_downscale_time"
            , SCALE_TIME_SECONDS
        );

        CheckIntegerUpdate(
            update[1].GetSetVec()[1]
            , "/status/replica_set/current_replicas"
            , 228
        );

        CheckIntegerUpdate(
            update[1].GetSetVec()[2]
            , "/status/replica_set/desired_replicas"
            , 3
        );

        CheckStatus(
            update[1].GetSetVec()[3]
            , update[1].GetSetVec()[4]
            , update[1].GetSetVec()[5]
            , "METRIC_FETCH_FAILED"
            , EStatusType::FAILED
        );

        CheckIntegerUpdate(
            update[1].GetSetVec()[6]
            , "/status/spec_timestamp"
            , SPEC_TIMESTAMP
        );
    }

private:
    virtual TStringStream GetGolovanResponse() const = 0;
    virtual TStringStream GetGolovanResponseFailed() const = 0;

    virtual NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec GetDefaultHPASpec(
        double lowerBound
        , double upperBound
        , ui32 minReplicas
        , ui32 maxReplicas
    ) const = 0;

    NYP::NClient::NApi::NProto::TAttributeList GetHPAAtributeList(
        double lowerBound
        , double upperBound
        , ui32 minReplicas
        , ui32 maxReplicas
        , const TInstant& nowTime
        , const TDuration& lastScaleShift
    ) {
        NYP::NClient::NApi::NProto::TAttributeList attributeList;
        attributeList.add_value_payloads()->set_yson(ToYson("test_hpa_id"));
        attributeList.add_timestamps(0);
        attributeList.add_value_payloads()->set_yson(ToYson("test_replica_set_id"));
        attributeList.add_timestamps(0);
        {
            auto hpaSpec = GetDefaultHPASpec(
                lowerBound
                , upperBound
                , minReplicas
                , maxReplicas
            );

            attributeList.add_value_payloads()->set_yson(NYT::NYson::ConvertToYsonString(hpaSpec).ToString());
            attributeList.add_timestamps(SPEC_TIMESTAMP);
        }
        {
            NYP::NClient::NApi::NProto::THorizontalPodAutoscalerStatus hpaStatus = GetDefaultHPAStatus(nowTime);
            *(hpaStatus.mutable_replica_set()->mutable_last_upscale_time()) = ToTimestamp(nowTime - lastScaleShift);
            *(hpaStatus.mutable_replica_set()->mutable_last_downscale_time()) = ToTimestamp(nowTime - lastScaleShift);

            attributeList.add_value_payloads()->set_yson(NYT::NYson::ConvertToYsonString(hpaStatus).ToString());
            attributeList.add_timestamps(0);
        }

        return attributeList;
    }
}; // class THorizontalPodAutoscalerBaseTestCase

class THorizontalPodAutoscalerWithCpuMetricTestCase : public THorizontalPodAutoscalerBaseTestCase {
private:
    TStringStream GetGolovanResponse() const final {
        return TStringStream(
            "{\"status\": \"ok\", \"response\": {\"signal\": {\"content\": {\"values\": {\"itype=deploy;geo=cluster;deploy_unit=DeployUnit1;stage=test_stage:havg(portoinst-cpu_usage_slot_hgram)\": [42]}}}}}"
        );
    }

    TStringStream GetGolovanResponseFailed() const final {
        return TStringStream(
            "{\"status\": \"ok\", \"response\": {\"failed\": {\"content\": {\"values\": {\"itype=deploy;geo=;deploy_unit=DeployUnit1;stage=test_stage:havg(portoinst-cpu_usage_slot_hgram)\": [42]}}}}}"
        );
    }

    NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec GetDefaultHPASpec(
        double lowerBound
        , double upperBound
        , ui32 minReplicas
        , ui32 maxReplicas
    ) const final {
        NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec hpaSpec;

        hpaSpec.mutable_replica_set()->set_min_replicas(minReplicas);
        hpaSpec.mutable_replica_set()->set_max_replicas(maxReplicas);
        hpaSpec.mutable_replica_set()->mutable_cpu()->set_lower_bound(lowerBound);
        hpaSpec.mutable_replica_set()->mutable_cpu()->set_upper_bound(upperBound);
        hpaSpec.mutable_replica_set()->mutable_upscale_delay()->set_seconds(SCALE_TIME_SECONDS);
        hpaSpec.mutable_replica_set()->mutable_downscale_delay()->set_seconds(SCALE_TIME_SECONDS);

        return hpaSpec;
    }
};

class THorizontalPodAutoscalerWithCpuUtilizationMetricTestCase : public THorizontalPodAutoscalerBaseTestCase {
private:
    TStringStream GetGolovanResponse() const final {
        return TStringStream(
            "{\"status\": \"ok\", \"response\": {\"signal\": {\"content\": {\"values\": {\"itype=deploy;geo=cluster;deploy_unit=DeployUnit1;stage=test_stage:havg(portoinst-cpu_limit_usage_perc_hgram)\": [42]}}}}}"
        );
    }

    TStringStream GetGolovanResponseFailed() const final {
        return TStringStream(
            "{\"status\": \"ok\", \"response\": {\"failed\": {\"content\": {\"values\": {\"itype=deploy;geo=;deploy_unit=DeployUnit1;stage=test_stage:havg(portoinst-cpu_limit_usage_perc_hgram)\": [42]}}}}}"
        );
    }

    NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec GetDefaultHPASpec(
        double lowerBound
        , double upperBound
        , ui32 minReplicas
        , ui32 maxReplicas
    ) const final {
        NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec hpaSpec;

        hpaSpec.mutable_replica_set()->set_min_replicas(minReplicas);
        hpaSpec.mutable_replica_set()->set_max_replicas(maxReplicas);
        hpaSpec.mutable_replica_set()->mutable_cpu_utilization()->set_lower_bound_percent(lowerBound);
        hpaSpec.mutable_replica_set()->mutable_cpu_utilization()->set_upper_bound_percent(upperBound);
        hpaSpec.mutable_replica_set()->mutable_upscale_delay()->set_seconds(SCALE_TIME_SECONDS);
        hpaSpec.mutable_replica_set()->mutable_downscale_delay()->set_seconds(SCALE_TIME_SECONDS);

        return hpaSpec;
    }
};

class THorizontalPodAutoscalerWithCustomGolovanMetricTestCase : public THorizontalPodAutoscalerBaseTestCase {
private:
    TStringStream GetGolovanResponse() const final {
        return TStringStream(
            "{\"status\": \"ok\", \"response\": {\"signal\": {\"content\": {\"values\": {\"itype=deploy;geo=cluster;mytag=value;deploy_unit=DeployUnit1;stage=test_stage:havg(portoinst-memory_limit_usage_perc_hgram)\": [42]}}}}}"
        );
    }

    TStringStream GetGolovanResponseFailed() const final {
        return TStringStream(
            "{\"status\": \"ok\", \"response\": {\"failed\": {\"content\": {\"values\": {\"itype=deploy;geo=;deploy_unit=DeployUnit1;stage=test_stage:havg(portoinst-cpu_limit_usage_perc_hgram)\": [42]}}}}}"
        );
    }

    NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec GetDefaultHPASpec(
        double lowerBound
        , double upperBound
        , ui32 minReplicas
        , ui32 maxReplicas
    ) const final {
        NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec hpaSpec;

        hpaSpec.mutable_replica_set()->set_min_replicas(minReplicas);
        hpaSpec.mutable_replica_set()->set_max_replicas(maxReplicas);

        hpaSpec.mutable_replica_set()->mutable_custom()->mutable_golovan_signal()->set_signal("havg(portoinst-memory_limit_usage_perc_hgram)");
        auto* tags = hpaSpec.mutable_replica_set()->mutable_custom()->mutable_golovan_signal()->mutable_tags();
        (*tags)["mytag"] = "value";
        (*tags)["itype"] = "deploy";
        (*tags)["deploy_unit"] = "DeployUnit1";
        (*tags)["stage"] = "test_stage";
        hpaSpec.mutable_replica_set()->mutable_custom()->mutable_golovan_signal()->set_set_local_geo_tag(true);

        hpaSpec.mutable_replica_set()->mutable_custom()->set_lower_bound(lowerBound);
        hpaSpec.mutable_replica_set()->mutable_custom()->set_upper_bound(upperBound);

        hpaSpec.mutable_replica_set()->mutable_upscale_delay()->set_seconds(SCALE_TIME_SECONDS);
        hpaSpec.mutable_replica_set()->mutable_downscale_delay()->set_seconds(SCALE_TIME_SECONDS);

        return hpaSpec;
    }
};

class THorizontalPodAutoscalerWithCustomGolovanMetricWithoutGeoTagTestCase : public THorizontalPodAutoscalerBaseTestCase {
private:
    TStringStream GetGolovanResponse() const final {
        return TStringStream(
            "{\"status\": \"ok\", \"response\": {\"signal\": {\"content\": {\"values\": {\"itype=deploy;mytag=value;deploy_unit=DeployUnit1;stage=test_stage:havg(portoinst-memory_limit_usage_perc_hgram)\": [42]}}}}}"
        );
    }

    TStringStream GetGolovanResponseFailed() const final {
        return TStringStream(
            "{\"status\": \"ok\", \"response\": {\"failed\": {\"content\": {\"values\": {\"itype=deploy;geo=;deploy_unit=DeployUnit1;stage=test_stage:havg(portoinst-cpu_limit_usage_perc_hgram)\": [42]}}}}}"
        );
    }

    NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec GetDefaultHPASpec(
        double lowerBound
        , double upperBound
        , ui32 minReplicas
        , ui32 maxReplicas
    ) const final {
        NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec hpaSpec;

        hpaSpec.mutable_replica_set()->set_min_replicas(minReplicas);
        hpaSpec.mutable_replica_set()->set_max_replicas(maxReplicas);

        hpaSpec.mutable_replica_set()->mutable_custom()->mutable_golovan_signal()->set_signal("havg(portoinst-memory_limit_usage_perc_hgram)");
        auto* tags = hpaSpec.mutable_replica_set()->mutable_custom()->mutable_golovan_signal()->mutable_tags();
        (*tags)["mytag"] = "value";
        (*tags)["itype"] = "deploy";
        (*tags)["deploy_unit"] = "DeployUnit1";
        (*tags)["stage"] = "test_stage";
        hpaSpec.mutable_replica_set()->mutable_custom()->mutable_golovan_signal()->set_set_local_geo_tag(false);

        hpaSpec.mutable_replica_set()->mutable_custom()->set_lower_bound(lowerBound);
        hpaSpec.mutable_replica_set()->mutable_custom()->set_upper_bound(upperBound);

        hpaSpec.mutable_replica_set()->mutable_upscale_delay()->set_seconds(SCALE_TIME_SECONDS);
        hpaSpec.mutable_replica_set()->mutable_downscale_delay()->set_seconds(SCALE_TIME_SECONDS);

        return hpaSpec;
    }
};

class THorizontalPodAutoscalerWithCustomSolomonMetricTestCase : public THorizontalPodAutoscalerBaseTestCase {
private:
    TStringStream GetGolovanResponse() const final {
        return TStringStream(
            "{\"status\": \"ok\", \"response\": {\"signal\": {\"content\": {\"values\": {\"itype=deploy;mytag=value;deploy_unit=DeployUnit1;stage=test_stage:havg(portoinst-memory_limit_usage_perc_hgram)\": [42]}}}}}"
        );
    }

    TStringStream GetGolovanResponseFailed() const final {
        return TStringStream(
            "{\"status\": \"ok\", \"response\": {\"failed\": {\"content\": {\"values\": {\"itype=deploy;geo=;deploy_unit=DeployUnit1;stage=test_stage:havg(portoinst-cpu_limit_usage_perc_hgram)\": [42]}}}}}"
        );
    }

    NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec GetDefaultHPASpec(
        double lowerBound
        , double upperBound
        , ui32 minReplicas
        , ui32 maxReplicas
    ) const final {
        NYP::NClient::NApi::NProto::THorizontalPodAutoscalerSpec hpaSpec;

        hpaSpec.mutable_replica_set()->set_min_replicas(minReplicas);
        hpaSpec.mutable_replica_set()->set_max_replicas(maxReplicas);

        auto* solomonSignal = hpaSpec.mutable_replica_set()->mutable_custom()->mutable_solomon_signal();
        solomonSignal->set_project("horizontal_pod_autoscaler_controller");
        solomonSignal->set_service("man_yp");
        solomonSignal->set_cluster("man_yp");
        solomonSignal->set_host("cluster");
        solomonSignal->set_sensor("horizontal_pod_autoscaler_controller.hpa_ctl.yanddmi-test-stage.rs.current_replicas");
        solomonSignal->set_aggregation(NYP::NClient::NApi::NProto::TCustomMetricSpec_TSolomonSignal::AT_AVG);
        solomonSignal->mutable_period()->set_seconds(5 * 60);

        hpaSpec.mutable_replica_set()->mutable_custom()->set_lower_bound(lowerBound);
        hpaSpec.mutable_replica_set()->mutable_custom()->set_upper_bound(upperBound);

        hpaSpec.mutable_replica_set()->mutable_upscale_delay()->set_seconds(SCALE_TIME_SECONDS);
        hpaSpec.mutable_replica_set()->mutable_downscale_delay()->set_seconds(SCALE_TIME_SECONDS);

        return hpaSpec;
    }
};

Y_UNIT_TEST_SUITE_F(HorizontalPodAutoscalerWithCpuMetricTest, THorizontalPodAutoscalerWithCpuMetricTestCase) {

Y_UNIT_TEST(FactorySelectArgument) {
    TestFactorySelectArgument();
}

Y_UNIT_TEST(ManagerDependentObjects) {
    TestManagerDependentObjects();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetReady) {
    TestCurrentMetricInLimitsAndReplicaSetReady();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetStatusNotReady) {
    TestCurrentMetricInLimitsAndReplicaSetStatusNotReady();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetPodCountNotReady) {
    TestCurrentMetricInLimitsAndReplicaSetPodCountNotReady();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetUpdateUpscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateUpscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetUpdateDownscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale();
}

Y_UNIT_TEST(TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow();
}

Y_UNIT_TEST(TestMetricRequestFailed) {
    TestMetricRequestFailed();
}

}

Y_UNIT_TEST_SUITE_F(HorizontalPodAutoscalerWithCpuUtilizationMetricTest, THorizontalPodAutoscalerWithCpuUtilizationMetricTestCase) {

Y_UNIT_TEST(FactorySelectArgument) {
    TestFactorySelectArgument();
}

Y_UNIT_TEST(ManagerDependentObjects) {
    TestManagerDependentObjects();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetReady) {
    TestCurrentMetricInLimitsAndReplicaSetReady();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetStatusNotReady) {
    TestCurrentMetricInLimitsAndReplicaSetStatusNotReady();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetPodCountNotReady) {
    TestCurrentMetricInLimitsAndReplicaSetPodCountNotReady();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetUpdateUpscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateUpscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetUpdateDownscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale();
}

Y_UNIT_TEST(TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow();
}

Y_UNIT_TEST(TestMetricRequestFailed) {
    TestMetricRequestFailed();
}

}

Y_UNIT_TEST_SUITE_F(THorizontalPodAutoscalerWithCustomGolovanMetricTestCase, THorizontalPodAutoscalerWithCustomGolovanMetricTestCase) {

Y_UNIT_TEST(FactorySelectArgument) {
    TestFactorySelectArgument();
}

Y_UNIT_TEST(ManagerDependentObjects) {
    TestManagerDependentObjects();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetReady) {
    TestCurrentMetricInLimitsAndReplicaSetReady();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetStatusNotReady) {
    TestCurrentMetricInLimitsAndReplicaSetStatusNotReady();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetPodCountNotReady) {
    TestCurrentMetricInLimitsAndReplicaSetPodCountNotReady();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetUpdateUpscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateUpscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetUpdateDownscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale();
}

Y_UNIT_TEST(TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow();
}

Y_UNIT_TEST(TestMetricRequestFailed) {
    TestMetricRequestFailed();
}

}

Y_UNIT_TEST_SUITE_F(THorizontalPodAutoscalerWithCustomGolovanMetricWithoutGeoTagTestCase, THorizontalPodAutoscalerWithCustomGolovanMetricWithoutGeoTagTestCase) {

Y_UNIT_TEST(FactorySelectArgument) {
    TestFactorySelectArgument();
}

Y_UNIT_TEST(ManagerDependentObjects) {
    TestManagerDependentObjects();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetReady) {
    TestCurrentMetricInLimitsAndReplicaSetReady();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetStatusNotReady) {
    TestCurrentMetricInLimitsAndReplicaSetStatusNotReady();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetPodCountNotReady) {
    TestCurrentMetricInLimitsAndReplicaSetPodCountNotReady();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetUpdateUpscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateUpscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetUpdateDownscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale();
}

Y_UNIT_TEST(TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow();
}

Y_UNIT_TEST(TestMetricRequestFailed) {
    TestMetricRequestFailed();
}

}

Y_UNIT_TEST_SUITE_F(THorizontalPodAutoscalerWithCustomSolomonMetricTestCase, THorizontalPodAutoscalerWithCustomSolomonMetricTestCase) {

Y_UNIT_TEST(FactorySelectArgument) {
    TestFactorySelectArgument();
}

Y_UNIT_TEST(ManagerDependentObjects) {
    TestManagerDependentObjects();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetReady) {
    TestCurrentMetricInLimitsAndReplicaSetReady();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetStatusNotReady) {
    TestCurrentMetricInLimitsAndReplicaSetStatusNotReady();
}

Y_UNIT_TEST(CurrentMetricInLimitsAndReplicaSetPodCountNotReady) {
    TestCurrentMetricInLimitsAndReplicaSetPodCountNotReady();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetUpdateUpscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateUpscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscaleByTags();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateUpscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetUpdateDownscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscale();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscaleByTags();
}

Y_UNIT_TEST(CurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale) {
    TestCurrentMetricNotInLimitsAndReplicaSetNotUpdateDownscale();
}

Y_UNIT_TEST(TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow) {
    TestCurrentMetricNotInLimitsAndReplicaSetUpdateDownscaleWithDesiredReplicasTooLow();
}

Y_UNIT_TEST(TestMetricRequestFailed) {
    TestMetricRequestFailed();
}

}

} // namespace NInfra::NHorizontalPodAutoscalerController
