from juggler_sdk import (
    Check,
    Child,
    FlapOptions,
    JugglerApi,
    NotificationOptions,
)

import os


def horizontal_pod_autoscaler_controller(cluster, tags, notify_rules):
    tags = tags(cluster)
    tags.append("deploy")
    return Check(host="{}.yp.yandex-team.ru".format(cluster),
                 service="horizontal_pod_autoscaler_controller",
                 ttl=300,
                 refresh_time=90,
                 aggregator="logic_or",
                 tags=tags,
                 namespace="deploy",
                 notifications=notify_rules,
                 flaps_config=FlapOptions(stable=300, critical=900, boost=0),
                 aggregator_kwargs={"unreach_checks": standard_unreach_checks(cluster)},
                 meta=juggler_service_description("https://wiki.yandex-team.ru/deploy/dev/docdrafts/horizontalpodautoscaler/"
                                                  "#diagnostics"),
                 children=[
                     Child(
                         host="(host={}_yp)&service=horizontal_pod_autoscaler_controller.controller.hpa_manager_factory_successful_sync_cycles".format(
                             host_from_cluster(cluster)),
                         service="all",
                         group_type="EVENTS",
                         instance="all"
                     )],
                 description=auto_managed_description()
                 )


YASM_API = "https://yasm.yandex-team.ru/srvambry/alerts"
HOUR = 60 * 60


def downtime_tags(cluster):
    return [
        "deploy-downtime-9b8ee2a2b559b3afff751623e5e79546",
        "deploy-{}-downtime-9b8ee2a2b559b3afff751623e5e79546".format(cluster)
    ]


def prestable_tags(cluster, downtime=True):
    tags = downtime_tags(cluster) if downtime else []
    tags.extend(["deploy-prestable"])
    return tags


def prod_tags(cluster, downtime=True):
    tags = downtime_tags(cluster) if downtime else []
    tags.extend(["deploy-prod"])
    return tags


def testing_tags(cluster, downtime=True):
    tags = downtime_tags(cluster) if downtime else []
    tags.extend(["deploy-test"])
    return tags


def host_from_cluster(cluster):
    if cluster == "man-pre":
        return "man_pre"
    elif cluster == "sas-test":
        return "sas_test"
    else:
        return cluster

SOLOMON_PROJECTS = ["horizontal_pod_autoscaler_controller"]

def standard_unreach_checks(cluster):
    return []


def solomon_alerting_not_working_unreach_check(solomon_project_id):
    if solomon_project_id not in SOLOMON_PROJECTS:
        raise ValueError("Unknown solomon project_id={}, add it to SOLOMON_PROJECTS".format(solomon_project_id))

    return [":deploy.solomon.alerting_check.{}".format(solomon_project_id)]


def juggler_service_description(link):
    return {
        "urls":
            [
                juggler_wiki_service_description(link)
            ]
    }


def juggler_wiki_service_description(link):
    return {
        "url": link,
        "title": "Full description on wiki",
        "type": "wiki"
    }

def deploy_critical_checks(cluster, check_type, tags, services, namespace, notifications):
    tags = tags(cluster)
    children = []

    for service in services:
        children.append(Child(
            host="{}.yp.yandex-team.ru".format(cluster),
            service=service.service,
            group_type="HOST",
            instance=""
        ))

    return Check(host="{}.yp.yandex-team.ru".format(cluster),
                 service="deploy.critical.{}".format(check_type),
                 ttl=900,
                 refresh_time=90,
                 aggregator="logic_or",
                 tags=tags,
                 flaps_config=FlapOptions(stable=600, critical=1200),
                 namespace=namespace,
                 notifications=notifications,
                 aggregator_kwargs={"unreach_checks": standard_unreach_checks(cluster)},
                 meta=juggler_service_description("https://wiki.yandex-team.ru/runtime-cloud/rtc-support/#serviceteams"),
                 children=children,
                 description=auto_managed_description()
                 )


def auto_managed_description():
    return "Automatically managed by infra/deploy/monitoring/juggler/autoconf.py"


status_ok_to_crit = {"from": "OK", "to": "CRIT"}
status_warn_to_crit = {"from": "WARN", "to": "CRIT"}
status_crit_to_ok = {"from": "CRIT", "to": "OK"}

telegram = "telegram"
sms = "sms"
email = "email"
phone = "phone"


def standard_notification(status, logins, notify_methods, business_time_only=False, calendar_id=None):
    template_kwargs = {
        "status": status,
        "login": logins,
        "method": notify_methods,
    }

    if calendar_id is not None:
        template_kwargs["calendar_id"] = calendar_id

    if business_time_only:
        template_kwargs["time_start"] = "11:00"
        template_kwargs["time_end"] = "20:00"

    if status == status_ok_to_crit:
        template_kwargs["repeat"] = 172800

    return NotificationOptions(
        template_name="on_status_change",
        template_kwargs=template_kwargs,
        description=auto_managed_description(),
    )


def get_cluster_type(cluster):
    for cluster_name, cluster_type in CLUSTERS:
        if cluster == cluster_name:
            return cluster_type

    return None


def system_logins_add(cluster, logins_to):
    cluster_type = get_cluster_type(cluster)
    if cluster_type == CLUSTER_TYPE_PROD:
        logins_to.append("yd-monitorings")


def isalambda(v):
    LAMBDA = lambda: 0
    return isinstance(v, type(LAMBDA)) and v.__name__ == LAMBDA.__name__


def standard_notifications(cluster, logins_to, logins_from=None,
                           prod_notifications=None,
                           prestable_notifications=None,
                           testing_notification=None,
                           notifications=None,
                           business_time_only=None,
                           add_system_logins=True,
                           calendar_id=None):

    if isalambda(logins_to):
        logins_to = logins_to(cluster)

    if add_system_logins:
        system_logins_add(cluster, logins_to)

    cluster_type = get_cluster_type(cluster)
    business_time_only_ = False if business_time_only is None else business_time_only

    if cluster_type == CLUSTER_TYPE_PROD:
        return [
            standard_notification(status_ok_to_crit, logins_to,
                                  prod_notifications
                                  or notifications
                                  or [phone, telegram, sms, email],
                                  business_time_only=business_time_only_,
                                  calendar_id=calendar_id),

            standard_notification(status_warn_to_crit, logins_to,
                                  prod_notifications
                                  or notifications
                                  or [phone, telegram, sms, email],
                                  business_time_only=business_time_only_,
                                  calendar_id=calendar_id),

            standard_notification(status_crit_to_ok, logins_from or logins_to, [telegram],
                                  business_time_only=business_time_only_,
                                  calendar_id=calendar_id)
        ]
    elif cluster_type == CLUSTER_TYPE_PRESTABLE:
        return [
            standard_notification(status_ok_to_crit, logins_to,
                                  prestable_notifications
                                  or notifications
                                  or [phone, telegram, sms, email],
                                  business_time_only=True,
                                  calendar_id=calendar_id),

            standard_notification(status_warn_to_crit, logins_to,
                                  prestable_notifications
                                  or notifications
                                  or [phone, telegram, sms, email],
                                  business_time_only=True,
                                  calendar_id=calendar_id),

            standard_notification(status_crit_to_ok, logins_from or logins_to, [telegram],
                                  business_time_only=True,
                                  calendar_id=calendar_id)
        ]
    elif cluster_type == CLUSTER_TYPE_TESTING:
        return [
            standard_notification(status_ok_to_crit, logins_to,
                                  testing_notification
                                  or notifications
                                  or [telegram, sms, email],
                                  business_time_only=True,
                                  calendar_id=calendar_id),

            standard_notification(status_warn_to_crit, logins_to,
                                  testing_notification
                                  or notifications
                                  or [telegram, sms, email],
                                  business_time_only=True,
                                  calendar_id=calendar_id),

            standard_notification(status_crit_to_ok, logins_from or logins_to, [telegram],
                                  business_time_only=True,
                                  calendar_id=calendar_id)
        ]

    assert not "Unknown cluster type"
    return None


def critical_notifications(cluster, logins,
                           calendar_id=None):

    cluster_type = get_cluster_type(cluster)
    assert cluster_type == CLUSTER_TYPE_PROD

    status_to_method = [
        (status_ok_to_crit, [telegram, phone]),
        (status_warn_to_crit, [telegram, phone]),
        (status_crit_to_ok, [telegram])
    ]

    notification_options = list()

    for status, methods in status_to_method:
        for method in methods:
            template_kwargs = {
                "status": status,
                "login": logins,
            }

            if calendar_id is not None:
                template_kwargs["calendar_id"] = calendar_id

            if method == telegram and status != status_crit_to_ok:
                template_kwargs["repeat"] = 5 * 60  # every 5 minutes repeat

            template_kwargs["method"] = method

            notification_options.append(NotificationOptions(
                template_name="on_status_change",
                template_kwargs=template_kwargs,
                description=auto_managed_description(),
            ))

    return notification_options


CLUSTER_TYPE_PRESTABLE = "prestable"
CLUSTER_TYPE_PROD = "prod"
CLUSTER_TYPE_TESTING = "testing"
ALL_CLUSTER_TYPES = [CLUSTER_TYPE_PRESTABLE, CLUSTER_TYPE_PROD, CLUSTER_TYPE_TESTING]

class CriticalCheckType(object):

    @staticmethod
    def deploy():
        return "deploy"

class AutoCheck:
    def __init__(self, check, notifications, cluster_types, critical_type=None):
        self.check = check
        self.notifications = notifications
        self.cluster_types = cluster_types
        self.critical_type = critical_type


CHECKS = [
    AutoCheck(horizontal_pod_autoscaler_controller,
                lambda cluster_name: standard_notifications(cluster_name, ["danibw"]),
                ALL_CLUSTER_TYPES,
                critical_type=CriticalCheckType.deploy()),
]

CLUSTERS = [
    ("sas-test", CLUSTER_TYPE_TESTING),
    ("man-pre", CLUSTER_TYPE_PRESTABLE),
    ("iva", CLUSTER_TYPE_PROD),
    ("man", CLUSTER_TYPE_PROD),
    ("myt", CLUSTER_TYPE_PROD),
    ("sas", CLUSTER_TYPE_PROD),
    ("vla", CLUSTER_TYPE_PROD),
    ("xdc", CLUSTER_TYPE_PROD),
]


def get_cluster_tags(cluster_type):
    if cluster_type == CLUSTER_TYPE_PROD:
        return prod_tags
    elif cluster_type == CLUSTER_TYPE_PRESTABLE:
        return prestable_tags
    elif cluster_type == CLUSTER_TYPE_TESTING:
        return testing_tags
    else:
        assert (not "Unknown cluster type")


def main():
    with JugglerApi("http://juggler-api.search.yandex.net",
                    mark="deploy_checks",
                    oauth_token=os.environ["JUGGLER_TOKEN"]) as api:

        for cluster, cluster_type in CLUSTERS:
            tags = get_cluster_tags(cluster_type)
            critical_checks = dict()
            for check_instance in CHECKS:

                check = check_instance.check
                notifications = check_instance.notifications
                check_cluster_type = check_instance.cluster_types

                if notifications is None:
                    notifications = []
                else:
                    notifications = notifications(cluster)

                if isinstance(check_cluster_type, str):
                    check_cluster_type = [check_cluster_type]

                if cluster_type in check_cluster_type:
                    juggler_check = check(cluster=cluster, tags=tags, notify_rules=notifications)
                    if juggler_check is not None:
                        if isinstance(juggler_check, list):
                            for check in juggler_check:
                                api.upsert_check(check)
                        else:
                            api.upsert_check(juggler_check)

                        if check_instance.critical_type is not None:
                            if check_instance.critical_type not in critical_checks:
                                critical_checks[check_instance.critical_type] = list()

                            critical_checks[check_instance.critical_type].append(juggler_check)

            if cluster_type == CLUSTER_TYPE_PROD:
                for check_type in critical_checks:
                    checks = critical_checks[check_type]

                    checks_namespaces = set(map(lambda check: check.namespace, checks))

                    # all checks of one critical type must be in one juggler namespace
                    assert len(checks_namespaces) == 1
                    critical_check = deploy_critical_checks(cluster,
                                                        check_type,
                                                        tags,
                                                        checks,
                                                        list(checks_namespaces)[0],
                                                        critical_notifications(
                                                            cluster,
                                                            ["deploy-monitoring-marty"]))
                    api.upsert_check(critical_check)


if __name__ == "__main__":
    main()
