import os
import argparse
import sys
import time
import csv
from datetime import datetime

from typing import Optional, List, Tuple
import prettytable

import requests
import traceback


MEGA = 10 ** 6
SECONDS_IN_MONTH = 3600 * 24 * 31  # count as 31 days in a month
YP_API = 'https://xdc.yp.yandex.net:8443/ObjectService/SelectObjects'


class ReleaseStats:
    def __init__(self, project_id: str, stage_id: str, deploy_time: str, release_type: str,
                 approved_logins: List[str], ticket_link: str):
        self.project_id = project_id
        self.stage_id = stage_id
        self.deploy_time = deploy_time
        self.release_type = release_type
        self.approved_logins = approved_logins
        self.ticket_link = ticket_link


def collect_stage_release_statistics(yp_client, project_id: str, stage_id: str, from_time: int,
                                     to_time: int, page_limit: int = 100) -> List[ReleaseStats]:
    release_stats = []
    continuation_token = None
    from_time *= MEGA
    to_time *= MEGA

    message = {
        "object_type": "deploy_ticket",
        "limit": {"value": page_limit},
        "filter": {"query": '[/meta/stage_id] = "{}" AND [/meta/creation_time] >= {} '
                            'AND [/meta/creation_time] < {}'.format(stage_id, from_time, to_time)},
        "selector": {"paths": ["/status", "/meta", "/spec"]}
    }

    while True:
        if continuation_token is not None:
            message["options"] = {"continuation_token": continuation_token}
        resp = send_request(yp_client, message)
        continuation_token = resp["continuation_token"]
        results = resp.get("results", [])
        for dp_ticket in results:
            status = dp_ticket["values"][0]
            meta = dp_ticket["values"][1]
            spec = dp_ticket["values"][2]
            if status.get('action'):  # and status['action'].get('type'):
                is_committed = status['action']['type'] == "commit"
            else:
                is_committed = False
            if not is_committed:
                patches = status.get("patches", [])
                for key in patches:
                    if patches[key].get('action') and patches[key]["action"]["type"] == "commit":
                        is_committed = True
                        break
            if is_committed:
                ticket_id = meta["id"]
                deploy_time = timestamp_to_str(meta["creation_time"])
                approved_logins = []
                if status.get("approval", False) and status["approval"].get("user_approvals"):
                    users = list(status["approval"]["user_approvals"])
                    users.sort(key=lambda u: u["approval_time"]["seconds"], reverse=True)
                    checked_users = set()
                    for user in users:
                        has_approved = user["status"] == "approved"
                        if user["login"] not in checked_users and has_approved:
                            approved_logins.append(user["login"])
                        checked_users.add(user["login"])
                release_type = spec.get("source_type", "release_integration")
                ticket_link = "https://deploy.yandex-team.ru/stages/{}/deploy-tickets/{}".format(stage_id, ticket_id)
                release_stats.append(
                    ReleaseStats(project_id, stage_id, deploy_time, release_type, approved_logins, ticket_link)
                )
        if len(results) < page_limit:
            break
    return release_stats


def list_project_stages(yp_client, project_id: str, page_limit: int = 100) -> List[str]:
    continuation_token = None
    stage_ids = []
    message = {
        "object_type": "stage",
        "limit": {"value": page_limit},
        "filter": {"query": '[/meta/project_id] = "{}"'.format(project_id)},
        "selector": {"paths": ["/meta/id", ]}}

    while True:
        if continuation_token is not None:
            message["options"] = {"continuation_token": continuation_token}
        stages_resp = send_request(yp_client, message)
        continuation_token = stages_resp["continuation_token"]
        results = stages_resp.get("results", [])
        for st in results:
            stage_ids.append(st["values"][0])
        if len(results) < page_limit:
            break
    return stage_ids


def get_project_id_from_stage(yp_client, stage_id: str) -> str:
    message = {
        "object_type": "stage",
        "filter": {"query": '[/meta/id] = "{}"'.format(stage_id)},
        "selector": {"paths": ["/meta/project_id", ]}}

    stages_resp = send_request(yp_client, message)
    results = stages_resp.get("results", [])
    return results[0]['values'][0]


def process(project_ids: Optional[str], stage_ids: Optional[str], from_time: Optional[int],
            to_time: Optional[int], yp_token: str, export_option: str = "csv"):
    s = requests.Session()
    h = {
        "Authorization": "OAuth {}".format(yp_token),
        "Content-Type": "application/json",
        "Accept": "application/json"
    }
    s.headers.update(h)
    release_stats = []
    if project_ids:
        for project_id in project_ids:
            stage_ids = list_project_stages(s, project_id)
            for stage_id in stage_ids:
                release_stats.extend(collect_stage_release_statistics(s, project_id, stage_id, from_time, to_time))
    else:
        for stage_id in stage_ids:
            try:
                project_id = get_project_id_from_stage(s, stage_id)
            except IndexError:
                continue
            release_stats.extend(collect_stage_release_statistics(s, project_id, stage_id, from_time, to_time))
    export(release_stats, export_option)


def send_request(yp_client, message) -> dict:
    try:
        resp = yp_client.post(YP_API, json=message)
        if resp.headers.get('X-YT-Response-Code') != '0':
            print("[API ERROR]: {}".format(resp.headers.get('X-YT-Response-Message')))
            sys.exit(1)
        resp = resp.json()
        return resp
    except Exception as e:
        print("Error while sending request: {}".format(e))
        traceback.print_exc()
        sys.exit(1)


def export(release_stats: List[ReleaseStats], export_option: str) -> None:
    header = ['project_id', 'stage-id', 'creation-time', 'release-type', 'approved-users', 'ticket-link']
    if export_option == "csv" or export_option is None:
        export_csv(release_stats, header)
    elif export_option == "table":
        export_table(release_stats, header)


def export_csv(release_stats: List[ReleaseStats], header: List[str]) -> None:
    writer = csv.writer(sys.stdout)
    writer.writerow(header)
    for r in release_stats:
        writer.writerow([r.project_id, r.stage_id, r.deploy_time, r.release_type, r.approved_logins, r.ticket_link])


def export_table(release_stats: List[ReleaseStats], header: List[str]) -> None:
    table = prettytable.PrettyTable(header)
    for r in release_stats:
        table.add_row([r.project_id, r.stage_id, r.deploy_time, r.release_type, r.approved_logins, r.ticket_link])
    print(table)


def parse_date(s: str) -> int:
    try:
        d = datetime.strptime(s, '%Y-%m-%d')
    except Exception:
        print('Cannot parse date, it must be in format YYYY-MM-DD, got: "{}"'.format(s))
        sys.exit(1)
    return int(d.timestamp())


def timestamp_to_str(t: int) -> str:
    t //= MEGA
    return datetime.utcfromtimestamp(t).strftime('%Y-%m-%d %H:%M:%S')


def last_n_months_to_time(n: int) -> Tuple[int, int]:
    from_time = int(time.time()) - SECONDS_IN_MONTH * n
    to_time = int(time.time())
    return from_time, to_time


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--from-time', '-ft', type=parse_date)
    parser.add_argument('--to-time', '-tt', type=parse_date)
    parser.add_argument('--last-n-months', '-lm', type=int)
    parser.add_argument('--project-id', '-p', nargs='+')
    parser.add_argument('--stage-id', '-s', nargs='+')
    parser.add_argument('--format', type=str, choices=['csv', 'table'])
    args = parser.parse_args()
    if bool(args.stage_id) == bool(args.project_id):
        print('Exactly one parameter stage-id or project-id must be given')
        sys.exit(1)
    if (bool(args.from_time) and bool(args.to_time)) and bool(args.last_n_months):
        print('Exactly one parameter from-time and to-time or last-n-month must be given')
        sys.exit(1)
    if args.last_n_months:
        args.from_time, args.to_time = last_n_months_to_time(args.last_n_months)
    if args.from_time is None:
        args.from_time, args.to_time = last_n_months_to_time(3)

    yp_token = os.getenv('YP_TOKEN')
    process(args.project_id, args.stage_id, args.from_time, args.to_time, yp_token, export_option=args.format)


if __name__ == '__main__':
    main()
