from infra.deploy.tools.yd_migrate.lib.deploy_migrate import dump_yp_object_to_yaml
from infra.deploy.tools.yd_migrate.lib.migrator_nanny import NannyMigrator
from infra.deploy.tools.yd_migrate.lib.migrator_qloud import QloudMigrator
from infra.deploy.tools.yd_migrate.lib.nanny_api import NannyApi
from infra.deploy.tools.yd_migrate.lib.qloud_api import QloudApi
from infra.deploy.tools.yd_migrate.lib.staff_api import StaffApi
from infra.deploy.tools.yd_migrate.lib.qloud_helper import qloud_signals_helper
from infra.deploy.tools.yd_migrate.lib.oauth_helper import get_token

import nanny_rpc_client
import nanny_rpc_client.exceptions
from infra.awacs.proto import api_pb2, api_stub, model_pb2, modules_pb2
from google.protobuf import json_format

import library.python.init_log

import argparse
import json
import logging
import time
import yaml


def dump_nanny_specs(args):
    api = NannyApi(args.nanny_token)
    nanny_spec = api.get_service_spec(args.service)

    nanny_spec_filename = 'nanny_spec.json'
    with open(nanny_spec_filename, 'w') as f:
        json.dump(nanny_spec, f, indent=4)
    logging.info('save nanny debug spec to %s. if you have any problems with migrator, hide secrets and attach this file to the ticket', nanny_spec_filename)

    stage, release_rules = NannyMigrator(
        spec=nanny_spec,
        stage_id=args.stage_id,
        service_id=args.service,
        raise_on_error=False,
        nanny_token=args.nanny_token,
        mock_clients=False,
        yp_token=args.yp_token
    ).migrate()

    filename = 'stage.yaml'
    with open(filename, 'w') as f:
        print >> f, dump_yp_object_to_yaml(stage)
    logging.info('check %s (nanny service -> YD.stage) and run `ya tool dctl put stage %s` when ready', filename, filename)

    for i in range(len(release_rules)):
        filename = 'release_rule_%s.yaml' % i
        with open(filename, 'w') as f:
            print >> f, dump_yp_object_to_yaml(release_rules[i])
        logging.info('check %s (nanny release rule -> YD.release_rule) and run `ya tool dctl put release_rule %s` when ready', filename, filename)


def dump_qloud_specs(args):
    token = get_token()
    api = QloudApi(token, args.contour)
    env_spec = api.get_environment_spec(args.project, args.application, args.environment)
    env_spec["abcId"] = args.abc_id

    awacs_alerting_recipient = StaffApi(token).staff_group_id_by_slug(args.awacs_alerting_recipient)

    qloud_spec_filename = 'qloud_spec.json'
    with open(qloud_spec_filename, 'w') as f:
        json.dump(env_spec, f, indent=4)
    logging.info('save qloud debug spec to %s. if you have any problems with migrator, hide secrets and attach this file to the ticket', qloud_spec_filename)

    deploy_spec, create_namespace_request, awacs_l7_macro, awacs_domains, awacs_upstreams, awacs_backends, awacs_endpoint_sets = QloudMigrator(
        env_spec,
        stage_id=args.stage_id,
        raise_on_error=False,
        yav_secret_alias=args.yav_secret_alias,
        awacs_category=args.awacs_category,
        awacs_namespace_id=args.awacs_namespace_id,
        awacs_alerting_recipient=awacs_alerting_recipient,
        mock_clients=False,
    ).migrate()

    with open('deploy.yaml', 'w') as f:
        print >> f, dump_yp_object_to_yaml(deploy_spec)
    logging.info('Check deploy.yaml (qloud components -> YD.stage) and run `ya tool dctl put stage deploy.yaml` when ready')

    with open('awacs.yaml', 'w') as f:
        print >> f, yaml.safe_dump({
            'create_namespace_request': json_format.MessageToDict(create_namespace_request),
            'awacs_l7_macro': json_format.MessageToDict(awacs_l7_macro),
            'awacs_domains': [json_format.MessageToDict(v) for v in awacs_domains],
            'awacs_upstreams': [json_format.MessageToDict(v) for v in awacs_upstreams],
            'awacs_backends': {k : json_format.MessageToDict(v) for k, v in awacs_backends.iteritems()},
            'awacs_endpoint_sets': {k : json_format.MessageToDict(v) for k, v in awacs_endpoint_sets.iteritems()},
        })
    logging.info('Check awacs.yaml (qloud domains/routes -> awacs namespace) and run `ya tool yd-migrate push_awacs <awacs_token>` when ready')
    logging.warning('Note that awacs balancer does not provide headers like "X-Balancer-Host" "X-Forwarded-For" yet - subscribe at SWAT-6465')

    qloud_signals = qloud_signals_helper('%s.%s.%s' % (args.project, args.application, args.environment))
    if qloud_signals:
        logging.info('Don\'t forget to reconfigure your yasm panels:\n%s', qloud_signals)


def push_awacs(args):
    AWACS_RPC_URL = 'https://awacs.yandex-team.ru/api/'
    awacs_rpc = nanny_rpc_client.RequestsRpcClient(rpc_url=AWACS_RPC_URL, oauth_token=args.awacs_token)

    with open('awacs.yaml', 'r') as f:
        awacs = yaml.safe_load(f.read())
    create_namespace_request = json_format.ParseDict(awacs['create_namespace_request'], api_pb2.CreateNamespaceRequest())
    awacs_l7_macro = json_format.ParseDict(awacs['awacs_l7_macro'], modules_pb2.L7Macro())
    awacs_domains = [json_format.ParseDict(v, model_pb2.DomainOrder.Content()) for v in awacs['awacs_domains']]
    awacs_upstreams = [json_format.ParseDict(v, modules_pb2.L7UpstreamMacro()) for v in awacs['awacs_upstreams']]
    awacs_backends = {k: json_format.ParseDict(v, model_pb2.BackendSelector()) for k, v in awacs['awacs_backends'].iteritems()}
    awacs_endpoint_sets = {k: json_format.ParseDict(v, model_pb2.EndpointSetSpec()) for k, v in awacs['awacs_endpoint_sets'].iteritems()}

    logging.info('check your namespace at https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/%s/show/', create_namespace_request.meta.id)
    namespace_service_stub = api_stub.NamespaceServiceStub(awacs_rpc)
    req_pb = api_pb2.GetNamespaceRequest()
    req_pb.id = create_namespace_request.meta.id
    while True:
        try:
            resp_pb = namespace_service_stub.get_namespace(req_pb)
            logging.info("namespace status: %s", resp_pb.namespace.order.status.status)
            if resp_pb.namespace.order.status.status != 'IN_PROGRESS':
                break
            else:
                logging.info("waiting for namespace creation")
        except nanny_rpc_client.exceptions.NotFoundError:
            logging.info("creating namespace")
            resp_pb = namespace_service_stub.create_namespace(create_namespace_request)
        logging.info("sleeping for 10 sec")
        time.sleep(10)

    logging.info('change balancer resources (cpu/mem/net if needed) manually at awacs UI')

    balancers_stub = api_stub.BalancerServiceStub(awacs_rpc)
    for cl in create_namespace_request.order.yp_lite_allocation_request.locations:
        bal_id = create_namespace_request.meta.id + '_' + cl.lower()
        req_pb = api_pb2.GetBalancerRequest()
        req_pb.id = bal_id
        req_pb.namespace_id = create_namespace_request.meta.id
        try:
            resp_pb = balancers_stub.get_balancer(req_pb)

            req_pb = api_pb2.UpdateBalancerRequest()
            req_pb.meta.id = bal_id
            req_pb.meta.namespace_id = create_namespace_request.meta.id
            req_pb.meta.version = resp_pb.balancer.meta.version
            req_pb.spec.CopyFrom(resp_pb.balancer.spec)
            req_pb.spec.yandex_balancer.ClearField("yaml")
            req_pb.spec.yandex_balancer.mode = model_pb2.YandexBalancerSpec.ConfigMode.EASY_MODE
            req_pb.spec.yandex_balancer.config.l7_macro.CopyFrom(awacs_l7_macro)
            resp_pb = balancers_stub.update_balancer(req_pb)
            logging.info("updated balancer: %s", bal_id)
        except nanny_rpc_client.exceptions.NotFoundError:
            req_pb = api_pb2.CreateBalancerRequest()
            req_pb.meta.id = bal_id
            req_pb.meta.namespace_id = create_namespace_request.meta.id
            req_pb.meta.auth.type = req_pb.meta.auth.STAFF
            req_pb.spec.yandex_balancer.mode = model_pb2.YandexBalancerSpec.ConfigMode.EASY_MODE
            req_pb.spec.yandex_balancer.config.l7_macro.CopyFrom(awacs_l7_macro)
            resp_pb = balancers_stub.create_balancer(req_pb)
            logging.info("added balancer: %s", req_pb.meta.id)

    backends_stub = api_stub.BackendServiceStub(awacs_rpc)
    backend_versions = {}
    for backend_id, backend in awacs_backends.iteritems():
        req_pb = api_pb2.GetBackendRequest()
        req_pb.id = backend_id
        req_pb.namespace_id = create_namespace_request.meta.id
        try:
            resp_pb = backends_stub.get_backend(req_pb)
            backend_versions[backend_id] = resp_pb.backend.meta.version
            logging.info('backend %s already exists, skipping', req_pb.id)
        except nanny_rpc_client.exceptions.NotFoundError:
            req_pb = api_pb2.CreateBackendRequest()
            req_pb.meta.id = backend_id
            req_pb.meta.namespace_id = create_namespace_request.meta.id
            req_pb.meta.auth.type = req_pb.meta.auth.STAFF
            req_pb.spec.selector.CopyFrom(backend)
            resp_pb = backends_stub.create_backend(req_pb)
            backend_versions[backend_id] = resp_pb.backend.meta.version
            logging.info("added backend: %s", req_pb.meta.id)

    endpoint_sets_stub = api_stub.EndpointSetServiceStub(awacs_rpc)
    for endpoint_set_id, endpoint_set in awacs_endpoint_sets.iteritems():
        req_pb = api_pb2.GetEndpointSetRequest()
        req_pb.id = endpoint_set_id
        req_pb.namespace_id = create_namespace_request.meta.id
        try:
            resp_pb = endpoint_sets_stub.get_endpoint_set(req_pb)
            logging.info('endpoint_set %s already exists, skipping', req_pb.id)
        except nanny_rpc_client.exceptions.NotFoundError:
            req_pb = api_pb2.CreateEndpointSetRequest()
            req_pb.backend_version = backend_versions[endpoint_set_id]  # backend_id == endpoint_set_id
            req_pb.meta.id = endpoint_set_id
            req_pb.meta.namespace_id = create_namespace_request.meta.id
            req_pb.meta.auth.type = req_pb.meta.auth.STAFF
            req_pb.spec.CopyFrom(endpoint_set)
            logging.info(req_pb)
            resp_pb = endpoint_sets_stub.create_endpoint_set(req_pb)
            logging.info("added endpoint_set: %s", req_pb.meta.id)

    domains_stub = api_stub.DomainServiceStub(awacs_rpc)
    for domain in awacs_domains:
        req_pb = api_pb2.GetDomainRequest()
        req_pb.id = ':'.join(domain.fqdns)
        req_pb.namespace_id = create_namespace_request.meta.id
        try:
            resp_pb = domains_stub.get_domain(req_pb)
            logging.info('domain %s already exists, skipping', req_pb.id)
        except nanny_rpc_client.exceptions.NotFoundError:
            req_pb = api_pb2.CreateDomainRequest()
            req_pb.meta.id = '-'.join(domain.fqdns)
            req_pb.meta.namespace_id = create_namespace_request.meta.id
            req_pb.order.CopyFrom(domain)
            resp_pb = domains_stub.create_domain(req_pb)
            logging.info("added domain: %s", req_pb.meta.id)

    upstreams_stub = api_stub.UpstreamServiceStub(awacs_rpc)
    for upstream in awacs_upstreams:
        req_pb = api_pb2.GetUpstreamRequest()
        req_pb.id = upstream.id
        req_pb.namespace_id = create_namespace_request.meta.id
        try:
            resp_pb = upstreams_stub.get_upstream(req_pb)
            logging.info('upstream %s already exists, skipping', req_pb.id)
        except nanny_rpc_client.exceptions.NotFoundError:
            req_pb = api_pb2.CreateUpstreamRequest()
            req_pb.meta.id = upstream.id
            req_pb.meta.namespace_id = create_namespace_request.meta.id
            req_pb.meta.auth.type = req_pb.meta.auth.STAFF
            req_pb.spec.labels["order"] = "0"
            req_pb.spec.yandex_balancer.mode = req_pb.spec.yandex_balancer.EASY_MODE2
            req_pb.spec.yandex_balancer.config.l7_upstream_macro.CopyFrom(upstream)
            resp_pb = upstreams_stub.create_upstream(req_pb)
            logging.info("added upstream: %s", req_pb.meta.id)


def main():
    library.python.init_log.init_log(level='INFO', append_ts=True)

    parser = argparse.ArgumentParser(description='Service -> Deploy.stage migration tool.')
    parser.add_argument('--log-file', type=str, default='yd_migrate.log', help='log file name')
    subparsers = parser.add_subparsers()

    parser_dump_nanny = subparsers.add_parser('dump_nanny', help='Outputs result specs to deploy.yaml')
    parser_dump_nanny.add_argument('nanny_token', type=str)
    parser_dump_nanny.add_argument('yp_token', type=str)
    parser_dump_nanny.add_argument('service', type=str, help='Nanny service')
    parser_dump_nanny.add_argument('--stage-id', type=str, help='deploy stage id')
    parser_dump_nanny.set_defaults(func=dump_nanny_specs)

    parse_dump_qloud = subparsers.add_parser('dump_qloud', help='Outputs result specs to deploy.yaml and awacs.yaml.')
    parse_dump_qloud.add_argument('contour', type=str, choices=['qloud', 'qloud-ext', 'platform'], help='qloud contour')
    parse_dump_qloud.add_argument('project', type=str, help='qloud project')
    parse_dump_qloud.add_argument('application', type=str, help='qloud application')
    parse_dump_qloud.add_argument('environment', type=str, help='qloud environment')
    parse_dump_qloud.add_argument('abc_id', type=int, help='service abc id for quota accounting')
    parse_dump_qloud.add_argument('awacs_alerting_recipient',
                                  type=str,
                                  help='servicerole slug for awacs balancer alerts',
                                  metavar='svc_myservice_administration')
    parse_dump_qloud.add_argument('--stage-id', type=str, default='', help='deploy stage id')
    parse_dump_qloud.add_argument('--yav-secret-alias', type=str, default='', help='yav alias for secret migration')
    parse_dump_qloud.add_argument('--awacs-category', type=str, default='', help='awacs category for balancer migration')
    parse_dump_qloud.add_argument('--awacs-namespace-id', type=str, default='', help='awacs namespace id for balancer migration')
    parse_dump_qloud.set_defaults(func=dump_qloud_specs)

    parse_awacs = subparsers.add_parser('push_awacs', help='Push awacs.yaml to Awacs API.')
    parse_awacs.add_argument('awacs_token', type=str)
    parse_awacs.set_defaults(func=push_awacs)

    args = parser.parse_args()
    library.python.init_log.init_handler(level='INFO', append_ts=True, log_file=args.log_file)
    args.func(args)

    logging.info('This logs are also available at %s', args.log_file)

if __name__ == '__main__':
    main()
