from infra.deploy.tools.yd_migrate.lib.feature_coverage import calc_over_dumped, print_covered_enviroments
from infra.deploy.tools.yd_migrate.lib.migrator_qloud import QloudMigrator
from infra.deploy.tools.yd_migrate.lib.qloud_api import QloudApi

import library.python.init_log

import argparse
import json
import logging
import requests
import time


def calc_over_dumped_args(args):
    calc_over_dumped(QloudMigrator, args.dump_file, args.with_prefix_implementation_result)


def dump_qloud_environments(args):
    api = QloudApi(args.qloud_token, args.contour)
    environments = []
    project_tree = api.get_project_tree()
    for project in project_tree['projects']:
        if project['projectName'] in [
            'development',  # skip dev machines that should migrate to QYP
            'qloud',  # skip qloud infra and l7 balancers
            'rr-priemka',  # skip rr-priemka that will migrate to yappy
            'unity',  # skip unity (samogon balancers)
        ]:
            logging.warning('skip project %s', project['projectName'])
            continue
        for app in project['applications']:
            for env in app['environments']:
                if env['name'].startswith('bootstrap'):
                    logging.warning('skip samogon %s', env['objectId'])
                    continue
                digits = sum(c.isdigit() for c in env['name'])
                if digits > 2:
                    logging.warning('skip robot-created %s', env['objectId'])
                    continue
                environments.append(env)
    logging.info('total environments: %s', len(environments))
    dumps = {}
    s_i = max(0, args.start_i)
    e_i = min(len(environments), args.end_i)
    for i in range(s_i, e_i):
        logging.info('progress: %s / %s', i, e_i)
        env = environments[i]
        time.sleep(0.05)  # 20 RPS
        try:
            spec = api.get_environment_spec(env['projectName'], env['applicationName'], env['name'])
            if spec.get("stable") and spec.get("stable").get("targetState") == "DEPLOYED":
                dumps[env['objectId']] = spec
        except (requests.HTTPError, requests.ConnectionError) as e:
            logging.warning(e)
            continue
    logging.info('total environments: %s', len(dumps))
    with open(args.dump_file, 'w') as f:
        f.write(json.dumps(dumps))


def print_covered_enviroments_args(args):
    print_covered_enviroments(QloudMigrator, args.dump_file)

logger = logging.getLogger(__name__)
library.python.init_log.init_log(level='DEBUG')

parser = argparse.ArgumentParser(description='Calc coverage of qloud environment features.')
parser.add_argument('-d', '--dump-file', type=str, default='dump.json')
subparsers = parser.add_subparsers()

parse_dump = subparsers.add_parser('dump', help='Dump qloud environments to dump.json.')
parse_dump.add_argument('qloud_token', type=str)
parse_dump.add_argument('contour', type=str, default='qloud', choices=['qloud', 'platform'], help='qloud contour')
parse_dump.add_argument('start_i', type=int, help='start_i for qloud environment dumps requests')
parse_dump.add_argument('end_i', type=int, help='end_i for qloud environment dumps requests')
parse_dump.set_defaults(func=dump_qloud_environments)

parse_calc = subparsers.add_parser('calc', help='Calc coverage over dump.json.')
parse_calc.add_argument('--with-prefix-implementation-result', type=int, default=50, help='For first N prefixes of top unimplemented features: calc result if prefix would be implemented.')
parse_calc.set_defaults(func=calc_over_dumped_args)

parse_print_covered = subparsers.add_parser('print_covered', help='Print covered environments')
parse_print_covered.set_defaults(func=print_covered_enviroments_args)

args = parser.parse_args()
args.func(args)
