import yp.client

import migrator_base
import template_resolver

import configparser
import logging
import re
import requests

from yp_proto.yp.client.api.proto import access_control_pb2
from yp_proto.yp.client.api.proto import autogen_pb2
from yp_proto.yp.client.api.proto import downloadable_resources_pb2
from yp_proto.yp.client.api.proto import host_infra_pb2
from yp_proto.yp.client.api.proto import pod_agent_pb2
from yp_proto.yp.client.api.proto import release_rule_pb2

import yt.yson as yson

from library.python.vault_client.instances import Production as VaultClient


def cast_command_to_str(cmd):
    """
    :type cmd: list[str | unicode]
    """
    # https://github.com/yandex/porto/blob/master/src/property.cpp#L1205
    rv = []
    for arg in cmd:
        escaped = arg.replace("'", "'\\''")
        r = u"'{}'".format(escaped)
        rv.append(r)
    # RTCSUPPORT-4139: /sbin/init is special case
    if len(rv) == 1 and rv[0] == "'/sbin/init'":
        return "/sbin/init"
    return ' '.join(rv)


class NannyMigrator(migrator_base.MigratorBase):
    def __init__(
        self,
        spec,
        stage_id,
        raise_on_error,
        service_id=None,
        nanny_token=None,
        mock_clients=False,
        yp_token=None
    ):
        super(NannyMigrator, self).__init__(
            migrator_name="nanny",
            spec=spec,
            stage_id=stage_id,
            raise_on_error=raise_on_error,
            implemented_map={
                "_id": self._set_stage_id,
                "unique_id_index": self.EImplemented.IMPLEMENTED,  # skip
                "auth_attrs": {
                    "_id": self.EImplemented.IMPLEMENTED,
                    "change_info": self.EImplemented.IMPLEMENTED,
                    "content": {
                        "conf_managers": self.EImplemented.IMPLEMENTED,  # Deploy does not support stage 'drafts' and conf_managers
                        "observers": self.EImplemented.IMPLEMENTED,  # Deploy does not support observers (CC in tickets)
                        "ops_managers": self.EImplemented.IMPLEMENTED,  # Deploy does not support start/stop without changing stage spec
                        "owners": self._set_stage_meta_acl,
                    },
                },
                "current_state": self.EImplemented.IMPLEMENTED,
                "info_attrs": {
                    "_id": self.EImplemented.IMPLEMENTED,
                    "change_info": self.EImplemented.IMPLEMENTED,
                    "content": {
                        "abc_group": self._set_abc_id,
                        "balancers_integration": {
                            "auto_update_services_balancers": self.EImplemented.IMPLEMENTED,  # all YD balancers are using endpoints for automated service discovery
                        },
                        "category": self._set_stage_project,
                        "cms_settings": {
                            "cms_stub_policy": self._assert_equal_one_of_enum(["SKIP"])
                        },
                        "component_path": self.EImplemented.IMPLEMENTED,  # skip platform path
                        "desc": self.EImplemented.IMPLEMENTED,
                        "disk_quotas": self._set_disk_quotas,
                        "infra_notifications": {
                            "environment_id": self._assert_equal(""),
                            "environment_name": self._assert_equal(""),
                            "service_id": self._assert_equal(""),
                            "service_name": self._assert_equal(""),
                        },
                        "instancectl_settings": self.EImplemented.IMPLEMENTED,
                        "labels": self.EImplemented.IMPLEMENTED,
                        "maintenance_notifications": {
                            self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                            }
                        },
                        "monitoring_settings": {
                            "deploy_monitoring": self._set_deploy_monitoring,
                            "juggler_settings": {
                                "content": {
                                    "active_checks": {
                                        self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                            "checks": {
                                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                                    "juggler_host_name": self._assert_equal(""),
                                                    "juggler_service_name": self._assert_equal(""),
                                                    "module": {},
                                                    "notifications": self._assert_equal([]),
                                                    "options": {
                                                        "args": self._assert_equal([]),
                                                        "env_vars": self._assert_equal([]),
                                                    },
                                                }
                                            },
                                            "flap_detector": {
                                                "is_enabled": self._assert_equal(False),
                                                "settings": {
                                                    "boost_time": self._assert_equal(""),
                                                    "critical_time": self._assert_equal(""),
                                                    "stable_time": self._assert_equal(""),
                                                },
                                            },
                                            "module": {
                                                "type": self._assert_equal_one_of_enum([]),
                                            },
                                            "passive_checks": {
                                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                                    "juggler_host_name": self._set_juggler_settings,
                                                    "juggler_service_name": self._assert_equal(""),
                                                    "notifications": self._assert_equal([]),
                                                    "options": {
                                                        "args": self._assert_equal([]),
                                                        "env_vars": self._assert_equal([]),
                                                    },
                                                }
                                            },
                                            "per_auto_tags": self._assert_equal([]),
                                        }
                                    },
                                    "juggler_hosts": {
                                        self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                        }
                                    },
                                    "juggler_tags": self._assert_equal([]),
                                    "instance_resolve_type": self._assert_equal_one_of_enum("NANNY"),
                                },
                                "is_enabled": self.EImplemented.IMPLEMENTED,
                            },
                            "panels": self.EImplemented.IMPLEMENTED,  # skip links to panels
                        },
                        "queue_id": self.EImplemented.IMPLEMENTED,  # at _service_release_rule
                        "recipes": {
                            "content": {  # DEPLOY-233
                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    "context": {
                                        self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: self._set_recipe_context,
                                    },
                                    "desc": self.EImplemented.IMPLEMENTED,  # field for humans only
                                    "id": self.EImplemented.IMPLEMENTED,  # field for humans only
                                    "labels": self._assert_equal([]),
                                    "name": self._assert_equal_one_of([
                                        "_activate_only_service_configuration.yaml",
                                        "_activate_only_service_configuration_with_wait.yaml",
                                        "_activate_only_web_base_configuration.yaml",
                                        "_activate_service_configuration.yaml",
                                        "_activate_service_configuration_2.yaml",
                                        "unconditional_activate_service_configuration.yaml",
                                    ]),
                                },
                            },
                            "prepare_recipes": {
                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    "context": {
                                        self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: self._set_recipe_context,
                                    },
                                    "desc": self.EImplemented.IMPLEMENTED,  # field for humans only
                                    "id": self.EImplemented.IMPLEMENTED,  # field for humans only
                                    "labels": self._assert_equal([]),
                                    "name": self._assert_equal_one_of([
                                        "_prepare_service_configuration.yaml",
                                        "unconditional_prepare_service_configuration.yaml",
                                    ]),
                                },
                            },
                        },
                        "scheduling_policy": {
                            "type": self._warn_if_not_equal_one_of(
                                ["NONE", "MAINTAIN_ACTIVE_TRUNK"],
                                "scheduling_policy is not implemented at YD, last revision is always the one being activating"
                            ),
                            "based_on_snapshot_priority": self.EImplemented.IMPLEMENTED,  # skip
                            "maintain_active_trunk": self.EImplemented.IMPLEMENTED,  # skip
                            "force_active_trunk": self.EImplemented.IMPLEMENTED,  # skip
                            "sequential_snapshot_activation": self.EImplemented.IMPLEMENTED,  # skip
                        },
                        "sox_compliant": self.EImplemented.IMPLEMENTED,  # skip flag
                        "tickets_integration": {
                            "docker_release_rule": {
                                "auto_commit_settings": self.EImplemented.SKIP,
                                "match_image_name": self._assert_equal(""),
                                "match_release_type": self._assert_equal_one_of_enum(["-"]),
                                "responsibles": {
                                    self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    },
                                },
                                "queue_id": self._assert_equal(""),
                            },
                            "gencfg_release_rule": self.EImplemented.IMPLEMENTED,  # gencfg is not linked to YD, so we don't need to migrate this part
                            "instancectl_release_rule": self.EImplemented.IMPLEMENTED,
                            "service_release_rules": {
                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    "desc": self._set_release_rule_desc,
                                    "sandbox_task_type": self._set_release_rule_sandbox_task_type,
                                    "sandbox_resource_type": self._set_release_rule_sandbox_resource_type,
                                    "filter_params": {
                                        "expression": self._set_release_rule_sandbox_release_types,
                                    },
                                    "ticket_priority": self.EImplemented.IMPLEMENTED,  # skip
                                    "approve_policy": {
                                        "type": self._assert_equal_one_of_enum(["NONE"]),
                                        "multiple_approve": {
                                            "approves_count": self._assert_equal(1),
                                            "approvers": {
                                                "logins": self._assert_equal([]),
                                                "groups": self._assert_equal([]),
                                            },
                                            "mandatory_approvers": {
                                                "logins": self._assert_equal([]),
                                                "groups": self._assert_equal([]),
                                            },
                                        },
                                    },
                                    "sched_activate_recipe": self._assert_equal_one_of_enum(["default"]),
                                    "sched_prepare_recipe": self._assert_equal_one_of_enum(["default"]),
                                    "responsibles": self._assert_equal([]),
                                    "queue_id": self.EImplemented.IMPLEMENTED,  # skip
                                    "auto_commit_settings": self._set_release_rule_auto_commit_policy,
                                },
                            },
                            "service_release_tickets_enabled": self.EImplemented.IMPLEMENTED,  # at _service_release_rule
                        },
                        "type": self.EImplemented.IMPLEMENTED,  # skip
                        "ui_settings": {
                            "set_snapshot_as_current_on_activate": self.EImplemented.IMPLEMENTED,  # skip
                        },
                        "yp_cluster": self.EImplemented.IMPLEMENTED,  # skip flag
                        "yp_settings": self._assert_equal({
                            "mirror_to_yp": "SKIP"
                        }),
                    },
                    "parent_id": self.EImplemented.IMPLEMENTED,
                },
                "runtime_attrs": {
                    "_id": self.EImplemented.IMPLEMENTED,
                    "change_info": self.EImplemented.IMPLEMENTED,
                    "content": {
                        "engines": {
                            "engine_type": self.EImplemented.IMPLEMENTED,
                        },
                        "instance_spec": {
                            "appContainer": {
                                "auxDaemons": {
                                    self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    }
                                },
                                "containers": {
                                    self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    }
                                },
                                "dockerImage": {
                                    "name": self._assert_equal(""),
                                    "registry": self._assert_equal_one_of_enum(["registry.yandex.net"]),
                                },
                                "initContainers": {
                                    self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    }
                                },
                                "instancectl": self.EImplemented.IMPLEMENTED,
                                "notifyAction": {
                                    "handlers": {
                                        self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                            "execAction": {
                                                "command": self._assert_equal([]),
                                            },
                                            "httpGet": {
                                                "host": self._assert_equal(""),
                                                "httpHeaders": self._assert_equal([]),
                                                "path": self._assert_equal(""),
                                                "port": self._assert_equal(""),
                                                "uriScheme": self._assert_equal(""),
                                            },
                                            "tcpSocket": {
                                                "host": self._assert_equal(""),
                                                "port": self._assert_equal(""),
                                            },
                                            "type": self._assert_equal_one_of_enum(["NONE"]),
                                        }
                                    },
                                    "resourceRequest": {
                                        "limit": self._assert_equal([]),
                                        "request": self._assert_equal([]),
                                    }
                                },
                                "rootVolume": {
                                    "quota": self._assert_equal_one_of_enum(["0"]),
                                },
                                "volumes": {
                                    self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    }
                                },
                                "workDir": self._assert_equal_one_of_enum(["/"]),
                            },
                            "auxDaemons": {
                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: self._set_aux_daemon,
                            },
                            "containers": {
                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    "command": self._set_workload_start_command_line,
                                    "coredumpPolicy": {
                                        "coredumpProcessor": {
                                            "aggregator": {
                                                "saas": {
                                                    "gdb": {
                                                        "execPath": self._assert_equal("/usr/bin/gdb"),
                                                        "type": self._assert_equal_one_of_enum(["LITERAL"]),
                                                    },
                                                    "serviceName": self.EImplemented.IMPLEMENTED,  # not needed at YD, cores are tagged with stage_id/workload_id
                                                    "url": self._set_coredump_aggregator_url,
                                                },
                                                "type": self._assert_equal_one_of_enum(["DISABLED", "SAAS_AGGREGATOR"]),
                                            },
                                            "cleanupPolicy": {
                                                "ttl": {
                                                    "seconds": self._set_coredump_processor_option("cleanup_ttl_seconds"),
                                                },
                                                "type": self._assert_equal_one_of_enum(["DISABLED", "TTL"]),
                                            },
                                            "countLimit": self._set_coredump_processor_option("count_limit"),
                                            "path": self._assert_equal("/cores"),
                                            "probability": self._set_coredump_processor_option("probability"),
                                            "totalSizeLimit": self._set_coredump_processor_option("total_size_limit_megabytes"),
                                        },
                                        "customProcessor": {
                                            "command": self._assert_equal(""),
                                        },
                                        "type": self._assert_equal_one_of_enum(["NONE", "COREDUMP"]),
                                    },
                                    "env": self._set_env,
                                    "hostDevices": self._assert_equal([]),
                                    "lifecycle": self._set_workload_stop_policy,
                                    "name": self.EImplemented.IMPLEMENTED,  # set in _set_workload_readiness_check
                                    "readinessProbe": self._set_workload_readiness_check,
                                    "reopenLogAction": {
                                        "handler": {
                                            "execAction": {
                                                "command": self._assert_equal([]),
                                            },
                                            "httpGet": {
                                                "host": self._assert_equal(""),
                                                "httpHeaders": self._assert_equal([]),
                                                "path": self._assert_equal(""),
                                                "port": self._assert_equal(""),
                                                "uriScheme": self._assert_equal(""),
                                            },
                                            "tcpSocket": {
                                                "host": self._assert_equal(""),
                                                "port": self._assert_equal(""),
                                            },
                                            "type": self._assert_equal_one_of_enum(["NONE"]),
                                        },
                                    },
                                    "resourceRequest": {
                                        "limit": {
                                            self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                            }
                                        },
                                        "request": {
                                            self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                            }
                                        }
                                    },
                                    "restartPolicy": self._warn_if_not_equal(
                                        {  # ignore default
                                            "maxPeriodSeconds": 60,
                                            "minPeriodSeconds": 1,
                                            "periodBackoff": 2,
                                            "periodJitterSeconds": 20
                                        },
                                        'restartPolicy configuration is not implemented at YD, on failure container will be restart after 10 seconds'
                                    ),
                                    "securityPolicy": {
                                        "runAsUser": self._assert_equal_one_of_enum(["", "root"]),  # skip root - default at YD
                                    },
                                    "unistatEndpoints": self._set_unistat_endpoints,
                                },
                            },
                            "dockerImage": self._set_docker_image,
                            "hostProvidedDaemons": {
                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    "type": self._set_host_provided_daemon,
                                },
                            },
                            "id": self._assert_equal(""),
                            "initContainers": self._set_init_containers,
                            "instanceAccess": {
                                "skynetSsh": self.EImplemented.IMPLEMENTED,  # Deploy has ssh to every box by default
                            },
                            "qemuKvm": self._set_qemu_kvm,
                            "instancectl": self.EImplemented.IMPLEMENTED,
                            "layersConfig": {
                                "layer": self._set_rootfs,
                                "bind": {
                                    self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                        "mode": self._assert_equal_one_of_enum([]),
                                        "mountPath": self._assert_equal(""),
                                        "path": self._assert_equal(""),
                                    }
                                }
                            },
                            "networkProperties": {
                                "etcHosts": self._assert_equal_one_of_enum(["KEEP_ETC_HOSTS"]),
                                "resolvConf": self._set_resolv_conf,
                            },
                            "notifyAction": {
                                "handlers": {
                                    self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                        "execAction": {
                                            "command": self._assert_equal([]),
                                        },
                                        "httpGet": {
                                            "host": self._assert_equal(""),
                                            "httpHeaders": self._assert_equal([]),
                                            "path": self._assert_equal(""),
                                            "port": self._assert_equal(""),
                                            "uriScheme": self._assert_equal(""),
                                        },
                                        "tcpSocket": {
                                            "host": self._assert_equal(""),
                                            "port": self._assert_equal(""),
                                        },
                                        "type": self._assert_equal_one_of_enum(["NONE"]),
                                    }
                                },
                                "resourceRequest": {
                                    "limit": self._assert_equal([]),
                                    "request": self._assert_equal([]),
                                }
                            },
                            "osContainerSpec": self._set_os_container_spec,
                            "type": self._assert_equal_one_of_enum(["DOCKER_LAYERS", "SANDBOX_LAYERS"]),
                            "volume": {
                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    "templateVolume": {
                                        "template": self._assert_equal([])
                                    },
                                    "itsVolume": {
                                        "itsUrl": self._assert_equal("http://its.yandex-team.ru/v1"),
                                        "maxRetryPeriodSeconds": self._assert_equal(300),
                                        "periodSeconds": self._assert_equal(60)
                                    },
                                    "secretVolume": self._set_nanny_vault_secret_volume,
                                    "vaultSecretVolume": self._set_yav_vault_secret_volume,
                                    "version": self.EImplemented.IMPLEMENTED,
                                    "name": self.EImplemented.IMPLEMENTED,
                                    "type": self.EImplemented.IMPLEMENTED,
                                }
                            },
                        },
                        "instances": {
                            "chosen_type": self.EImplemented.IMPLEMENTED,
                            "extended_gencfg_groups": {
                                "containers_settings": {
                                    "slot_porto_properties": self._assert_equal_one_of_enum(["NONE", "ALL_EXCEPT_GUARANTEES", "ALL"]),
                                },
                                "gencfg_volumes_settings": {
                                    "use_volumes": self._assert_equal_one_of_enum([True, False]),
                                },
                                "groups": self._set_extended_gencfg_groups,
                                "instance_properties_settings": {
                                    "tags": self._assert_equal_one_of_enum(["ALL_STATIC", "TOPOLOGY_DYNAMIC"]),
                                },
                                "network_settings": {
                                    "hbf_nat": self.EImplemented.IMPLEMENTED,  # all YD instances are isolated
                                    "use_mtn": self.EImplemented.IMPLEMENTED,  # all YD instances are isolated
                                },
                                "tags": self.EImplemented.IMPLEMENTED,  # ignore
                                "sysctl_settings": {
                                    "params": {
                                        self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                        },
                                    },
                                },
                            },
                            "gencfg_groups": self._assert_equal([]),
                            "instance_list": self._assert_equal([]),
                            "iss_settings": self.EImplemented.IMPLEMENTED,  # pod is managed by pod_agent - skip iss settings
                            "yp_pod_ids": self._set_yp_pod_ids,
                            "yp_pods": self._set_yp_pods,
                        },
                        "resources": {
                            "l7_fast_balancer_config_files": self._assert_equal([]),
                            "sandbox_bsc_shard": {
                                "chosen_type": self._assert_equal_one_of_enum([]),
                                "containers_settings": self._assert_equal(""),
                                "local_path": self._assert_equal(""),
                                "registered_shard": self._assert_equal(""),
                                "resource_type": self._assert_equal(""),
                                "sandbox_shardmap": self._assert_equal(""),
                                "storage": self._assert_equal(""),
                                "task_id": self._assert_equal(""),
                                "task_type": self._assert_equal(""),
                            },
                            "sandbox_files": self._set_static_resource_from_sandbox,
                            "services_balancer_config_files": self._assert_equal([]),
                            "static_files": self._set_static_resource_from_static_files,
                            "template_set_files": {
                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    "check_period": self._assert_equal(""),
                                    "extract_path": self._assert_equal(""),
                                    "is_dynamic": self._assert_equal(False),
                                    "layout": self._assert_equal(""),
                                    "local_path": self._assert_equal(""),
                                    "templates": self._assert_equal(""),
                                }
                            },
                            "url_files": {
                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                                    "check_period": self._assert_equal(""),
                                    "chksum": self._assert_equal(""),
                                    "extract_path": self._assert_equal(""),
                                    "is_dynamic": self._assert_equal(False),
                                    "local_path": self._assert_equal(""),
                                    "storage": self._assert_equal(""),
                                    "url": self._assert_equal(""),
                                }
                            }
                        },
                    },
                    "meta_info": self.EImplemented.IMPLEMENTED,
                    "parent_id": self.EImplemented.IMPLEMENTED,
                },
                "target_state": self.EImplemented.IMPLEMENTED,
            },
            mock_clients=mock_clients
        )

        self.nanny_token = nanny_token
        self.nanny_session = requests.Session()
        self.nanny_session.headers['Authorization'] = 'OAuth {}'.format(nanny_token)

        self.yp_token = yp_token

        self.release_rules = []
        self.instance = self._get_service_instance(service_id)

    def __is_instance_ctl_conf_file(self, filename):
        return filename == 'instancectl.conf' or filename == 'loop.conf'  # https://wiki.yandex-team.ru/jandekspoisk/sepe/instancectl/#config-file

    def __is_conf_local_file(self, filename):
        return filename == 'Conf.local'  # https://wiki.yandex-team.ru/jandekspoisk/sepe/instancectl/#opt-tags-example

    def __set_time_limit(self, time_limit, readiness_probe):
        time_limit.initial_delay_ms = readiness_probe.get("initialDelaySeconds", 0) * 1000
        time_limit.max_restart_period_ms = readiness_probe.get("maxPeriodSeconds", 0) * 1000
        time_limit.min_restart_period_ms = readiness_probe.get("minPeriodSeconds", 0) * 1000
        time_limit.restart_period_back_off = readiness_probe.get("periodBackoff", 0)
        time_limit.restart_period_scale_ms = 1

    def __convert_port_with_bsconfig(self, port):
        try:
            result = int(port)
        except ValueError:
            result = 80
            plus_prefix = '{BSCONFIG_IPORT_PLUS_'
            if plus_prefix in port:
                result += int(port[len(plus_prefix):-1])
            logging.warning('treating "%s" as %s ( based on https://wiki.yandex-team.ru/yp/instrukcija-nanny.yp-lite/#bsconfigiport )', port, result)
        return result

    def __wrap_command_with_bash(self, command):
        return "/bin/bash -c '{}'".format(command.replace("'", "'\\''"))

    def __instancectl_match_with_conf_local(self, match_string, resolve_context, path, errors):
        conf_local_content = ''
        for file in self.spec["runtime_attrs"]["content"]["resources"]["static_files"]:
            if self.__is_conf_local_file(file["local_path"]):
                conf_local_content = file["content"]

        result = {}
        for line in conf_local_content.splitlines():
            patterns = re.findall(match_string, line)
            if patterns:
                key, value = patterns[0].split('=', 1)
                result[key] = value

        try:
            result, bad_keys = template_resolver.resolve(result, resolve_context)
            for bad_key in bad_keys:
                errors.append(self.NotImplementedException('Substitute for var %s not implemented' % bad_key, path + [bad_key]))
        except Exception as e:
            errors.append(self.NotImplementedException('Error at template resolving: %s' % e, path + ['RESOLVE_ERROR']))

        return result

    def __check_not_yasmagent(self, command, path):
        # check yasmagent according to https://wiki.yandex-team.ru/golovan/dev/components/yasmagent/subagent/#howto
        yasm_binary_path = '/usr/local/yasmagent/python/bin/python'
        if yasm_binary_path in command:
            raise self.NotImplementedException('YASM_AGENT is not supported', path + ['YASM_AGENT'])

    def __get_release_rule(self, release_rule_id):
        if release_rule_id >= len(self.release_rules):
            self.release_rules += [autogen_pb2.TReleaseRule()] * (release_rule_id - len(self.release_rules) + 1)
        return self.release_rules[release_rule_id]

    #
    # STAGE
    #

    def _set_stage_project(self, category, path):
        annotation = self.stage.annotations.attributes.add()
        annotation.key = 'project'
        annotation.value = str('_'.join(filter(None, category.split('/'))))

    def _set_abc_id(self, abc_id, path):
        self.stage.meta.account_id = ('abc:service:' + str(abc_id)) if abc_id else "tmp"

    def _set_qemu_kvm(self, buf, path):
        if self.spec["runtime_attrs"]["content"]["instance_spec"]["type"] == "QEMU_KVM":
            raise self.NotImplementedException('QemuKvm migration is not supported', path)

    def _set_os_container_spec(self, buf, path):
        if self.spec["runtime_attrs"]["content"]["instance_spec"]["type"] == "OS_CONTAINER":
            raise self.NotImplementedException('OsContainerSpec migration is not supported', path)

    def _set_disk_quotas(self, buf, path):
        if self.spec["runtime_attrs"]["content"]["instances"]["chosen_type"] != "YP_POD_IDS" and \
           self.spec["runtime_attrs"]["content"]["instances"]["chosen_type"] != "EXTENDED_GENCFG_GROUPS":
            raise self.NotImplementedException('Disk quotas migration is not supported', path)

    def _set_yp_pod_ids(self, buf, path):
        if len(buf['pods']) == 0:
            return
        unit_id = self.spec['_id']
        pod_resources = self._get_pod_template_spec_spec(unit_id).resource_requests
        pod_disk_volumes = self._get_pod_template_spec_spec(unit_id).disk_volume_requests
        pod_ip6_addresses = self._get_pod_template_spec_spec(unit_id).ip6_address_requests
        pod_count_in_cluster = dict()

        for i in range(len(buf['pods'])):
            pod = buf['pods'][i]
            buf_resources = self._get_pod_resource_requests(pod_id=pod['pod_id'], address=pod['cluster'])
            buf_ip6_addresses = self._get_pod_ip6_address_requests(pod_id=pod['pod_id'], address=pod['cluster'])
            buf_disk_volumes = self._get_pod_disk_volume_requests(pod_id=pod['pod_id'], address=pod['cluster'])
            for it in buf_disk_volumes:
                it.pop('id')  # ignore for diff
            if i == 0:
                resources = buf_resources
                ip6_addresses = buf_ip6_addresses
                disk_volumes = buf_disk_volumes
            else:
                if resources != buf_resources:
                    logging.warn(
                        "Pod resources not equal\npod_ids:%s\n,resources:%s" %
                        (str([buf['pods'][0]['pod_id'], buf['pods'][i]['pod_id']]), str([resources, buf_resources])))
                    break
                if ip6_addresses != buf_ip6_addresses:
                    logging.warn(
                        "Pod ip6_addresses not equal\npod_ids:%s\n,ip6_addresses:%s" %
                        (str([buf['pods'][0]['pod_id'], buf['pods'][i]['pod_id']]), str([ip6_addresses, buf_ip6_addresses])))
                    break
                if disk_volumes != buf_disk_volumes:
                    logging.warn(
                        "Pod disk_volumes not equal\npod_ids:%s\n,disk_volumes:%s" %
                        (str([buf['pods'][0]['pod_id'], buf['pods'][i]['pod_id']]), str([disk_volumes, buf_disk_volumes])))
                    break

            pod_count_in_cluster[pod['cluster']] = pod_count_in_cluster.get(pod['cluster'], 0) + 1

        if len(disk_volumes) > 1:
            logging.warn("Multiple disk_volume_requests are not supported at YD yet, will sum up all disk qouta at one volume")
        big_disk_volume = None
        for disk_volume in disk_volumes:
            if big_disk_volume is None:
                big_disk_volume = {
                    "quota_policy": {
                        "capacity": 0,
                        "bandwidth_limit": 0,
                        "bandwidth_guarantee": 0
                    },
                    "storage_class": disk_volume['storage_class'],
                    "labels": {
                        "mount_path": disk_volume['labels']['mount_path'],
                        "volume_type": disk_volume['labels']['volume_type']
                    }
                }
                if disk_volume['labels']['mount_path'][1:]:
                    big_disk_volume_id = disk_volume['labels']['volume_type'] + '_' + disk_volume['labels']['mount_path'][1:]
                else:
                    big_disk_volume_id = disk_volume['labels']['volume_type']

            big_disk_volume['quota_policy']['capacity'] += int(disk_volume['quota_policy']['capacity'])
            big_disk_volume['quota_policy']['bandwidth_guarantee'] += int(disk_volume['quota_policy'].get('bandwidth_guarantee', 0))
            big_disk_volume['quota_policy']['bandwidth_limit'] += int(disk_volume['quota_policy'].get('bandwidth_limit', 0))

            if disk_volume['labels']['volume_type'] != 'root_fs':
                volume = self._get_pod_agent_payload_spec(unit_id).volumes.add()
                if disk_volume['labels']['mount_path'][1:]:
                    volume.id = disk_volume['labels']['mount_path'][1:] + '_' + disk_volume['labels']['volume_type']
                else:
                    volume.id = disk_volume['labels']['volume_type']
                box = self._get_box(unit_id)
                mounted_volume = box.volumes.add()
                mounted_volume.volume_ref = volume.id
                mounted_volume.mount_point = disk_volume['labels']['mount_path']
                mounted_volume.mode = pod_agent_pb2.EVolumeMountMode_READ_WRITE

        if big_disk_volume is not None:
            pod_disk_volume = pod_disk_volumes.add()
            pod_disk_volume.id = big_disk_volume_id
            pod_disk_volume.quota_policy.capacity = int(big_disk_volume['quota_policy']['capacity'])
            if pod_disk_volume.quota_policy.capacity < (migrator_base.BOUND_FOR_DISK_TAX_IN_GB << 30):
                pod_disk_volume.quota_policy.capacity += (migrator_base.DISK_TAX_IN_GB << 30)
                logging.warning('Add {}GB to disk capacity for pod_agent infrastructure'.format(migrator_base.DISK_TAX_IN_GB))

            pod_disk_volume.quota_policy.bandwidth_guarantee = int(big_disk_volume['quota_policy']['bandwidth_guarantee'])
            pod_disk_volume.quota_policy.bandwidth_limit = int(big_disk_volume['quota_policy']['bandwidth_limit'])
            pod_disk_volume.storage_class = big_disk_volume['storage_class']
            label = pod_disk_volume.labels.attributes.add()
            label.key = 'used_by_infra'
            label.value = yson.dumps(True)

        pod_resources.vcpu_guarantee = int(resources['vcpu_guarantee'])
        pod_resources.vcpu_limit = int(resources['vcpu_limit'])
        pod_resources.memory_guarantee = int(resources['memory_guarantee'])
        pod_resources.memory_limit = int(resources['memory_limit'])

        network_defaults_set = False
        for ip6_address in ip6_addresses:
            pod_ip6_address = pod_ip6_addresses.add()
            pod_ip6_address.enable_dns = bool(ip6_address['enable_dns'])
            pod_ip6_address.vlan_id = ip6_address['vlan_id']
            pod_ip6_address.network_id = ip6_address['network_id']
            if not network_defaults_set:
                network_defaults = self.stage.spec.deploy_units[unit_id].network_defaults
                network_defaults.network_id = ip6_address['network_id']
                network_defaults_set = True

        clusters = self.stage.spec.deploy_units[unit_id].multi_cluster_replica_set.replica_set.clusters
        instances_count = 0
        for cluster_name, pod_count in pod_count_in_cluster.items():
            cluster_info = clusters.add()
            cluster_info.cluster = cluster_name.lower()
            cluster_info.spec.replica_count = pod_count
            constraint = cluster_info.spec.constraints.antiaffinity_constraints.add()
            constraint.key = 'node'
            constraint.max_pods = 1
            instances_count += pod_count

        self._set_deployment_strategy(instances_count)

    def _set_yp_pods(self, buf, path):
        if len(buf['allocations']) or len(buf['tags']):
            raise self.NotImplementedException('Yp pods migration is not supported', path)

    def _set_rootfs(self, layers, path):
        unit_id = self.spec["_id"]
        pod_agent_payload_spec = self._get_pod_agent_payload_spec(unit_id)
        box = self._get_box(unit_id)
        errors = []
        for i in range(len(layers)):
            layer = layers[i]

            if not (layer.get('fetchableMeta')):
                errors.append(self.NotImplementedException('Not found field layer/X/fetchableMeta', path + [i]))
                continue

            box.rootfs.layer_refs.append(layer['fetchableMeta']['sandboxResource']['resourceType'])

            resource = pod_agent_payload_spec.resources.layers.add()
            resource.checksum = 'EMPTY:'
            resource.id = layer['fetchableMeta']['sandboxResource']['resourceType']
            resource.url = layer['url'][0]

        if errors:
            raise self.NotImplementedExceptionList(errors)

    def _set_stage_meta_acl(self, owner, path):
        acl = self.stage.meta.acl.add()

        acl.action = access_control_pb2.EAccessControlAction.ACA_ALLOW

        permissions = [
            access_control_pb2.EAccessControlPermission.ACP_READ,
            access_control_pb2.EAccessControlPermission.ACA_WRITE,
            access_control_pb2.EAccessControlPermission.ACA_CREATE,
            access_control_pb2.EAccessControlPermission.ACA_SSH_ACCESS,
            access_control_pb2.EAccessControlPermission.ACA_ROOT_SSH_ACCESS,
            access_control_pb2.EAccessControlPermission.ACP_READ_SECRETS,
        ]
        for permission in permissions:
            acl.permissions.append(permission)

        for user in owner["logins"]:
            acl.subjects.append(user)
        for i in range(len(owner["groups"])):
            staff_id = owner["groups"][i]
            abc_group = self._staff_id_to_abc_group(staff_id)
            if abc_group:
                acl.subjects.append(abc_group)
            else:
                logging.warning("Could not convert staff group %s to abc group (YD supports only abc groups) at %s", staff_id, path + ["groups", i])

    def _set_instancectl_conf(self, instancectl_conf, path):
        DEFAULTS_SECTION = "defaults"

        cfg = configparser.RawConfigParser(
            strict=False,  # allow duplicates for 'status_script's
            allow_no_value=True,
        )
        cfg.read_string(unicode(instancectl_conf))

        errors = []

        default_vars = cfg._sections.get(DEFAULTS_SECTION, {})
        context = {
            "BSCONFIG_IDIR": "/",
            "BSCONFIG_IPORT": "80",
            "logs_dir": "/usr/local/www/logs",
        }

        for i in range(len(cfg._sections)):
            section, local_vars = cfg._sections.items()[i]
            if section == DEFAULTS_SECTION:
                continue
            context["section"] = section
            conf = default_vars.copy()
            conf.update(local_vars)

            try:
                conf, bad_keys = template_resolver.resolve(conf, context)
                for bad_key in bad_keys:
                    errors.append(self.NotImplementedException('Substitute for var %s not implemented' % bad_key, path + [bad_key]))
            except Exception as e:
                errors.append(self.NotImplementedException('Error at template resolving: %s' % e, path + [i, 'RESOLVE_ERROR']))

            unit_id = self.spec["_id"]
            workload = self._get_workload(unit_id, workload_id=section)
            workload.start.command_line = conf["binary"] + ' ' + conf["arguments"].strip()

            try:
                self.__check_not_yasmagent(workload.start.command_line, path + [i])
            except self.NotImplementedException as e:
                errors.append(e)

            for key, value in conf.iteritems():
                try:
                    if key == "binary" or key == "arguments":
                        pass
                    elif key == "prepare_script":
                        box = self._get_box(unit_id)
                        init = box.init.add()
                        init.command_line = self.__wrap_command_with_bash(value)
                    elif key == "install_script":
                        init = workload.init.add()
                        init.command_line = self.__wrap_command_with_bash(value)
                    elif key == "status_check_type":
                        self._assert_equal_one_of_enum(["script"])(value, path + [i, key])
                    elif key == "status_script":
                        workload.readiness_check.container.command_line = self.__wrap_command_with_bash(value)
                    elif key == "stop_script":
                        workload.stop_policy.max_tries = 1
                        workload.stop_policy.container.command_line = self.__wrap_command_with_bash(value)
                    elif key == "action_stop_timeout":
                        workload.stop_policy.container.time_limit.max_execution_time_ms = int(value) * 1000
                    elif key in ["terminate_timeout", "kill_timeout"]:
                        logging.warning("stop timeouts are not implemented at YD: workload will be killed (SIGKILL)"
                                        " immediately after 'stop' https://wiki.yandex-team.ru/deploy/docs/podstack/workload/probes/#stop at %s", path + [i, key])
                    elif key == "env_match":
                        envs = self.__instancectl_match_with_conf_local(value, context, path + [i, 'env_match'], errors)
                        for env_key, env_value in envs.iteritems():
                            env = workload.env.add()
                            env.name = env_key
                            env.value.literal_env.value = env_value
                    elif key == "opt_match":
                        opts = self.__instancectl_match_with_conf_local(value, context, path + [i, 'opt_match'], errors)
                        for opt_key, opt_value in opts.iteritems():
                            workload.start.command_line = workload.start.command_line + ' -V %s=%s' % (opt_key, opt_value)
                    elif key == "limit_core":
                        ulimit = workload.ulimit_soft.add()
                        ulimit.name = pod_agent_pb2.EContainerULimit_CORE
                        if value == "unlimited":
                            ulimit.value = 128 << 30  # 128G
                            logging.warning("limit_core=unlimited is not supported at YD, will treat as %s at %s", ulimit.value, path + [i])
                        else:
                            ulimit.value = int(value)
                    elif key == "always_coredump":
                        self._assert_equal_one_of_enum(["True", "true", "yes"])(value, path + [i, key])
                    elif key == "coredump_probability":
                        self._assert_equal("100")(value, path + [i, key])
                    elif key in ["backoff", "delay", "max_delay", "max_jitter", "successful_start_timeout"]:
                        logging.warning("restartPolicy configuration is not implemented at YD, on failure container will be restart after 10 seconds at %s", path + [i, key])
                    elif key == "rename_binary":
                        logging.warning("rename_binary is not implemented at YD at %s", path + [i, key])
                    elif key == "use_porto":
                        pass  # every YD workload is a porto container
                    elif key in ["config", "logs_dir", "name", "section", "vlanbb", "vlanfb"]:
                        pass  # some user-defined properties
                    elif key.startswith("eval_"):
                        raise self.NotImplementedException('instancectl: eval_* (%s) not implemented yet' % key, path + [i, "eval_*"])
                    else:
                        raise self.NotImplementedException('instancectl: %s not implemented' % key, path + [i, key])
                except self.NotImplementedException as e:
                    errors.append(e)

        if errors:
            raise self.NotImplementedExceptionList(errors)

    def _set_static_resource_from_sandbox(self, sandbox_resources, path):
        unit_id = self.spec["_id"]
        pod_agent_payload_spec = self._get_pod_agent_payload_spec(unit_id)
        box = self._get_box(unit_id)
        errors = []
        for i in range(len(sandbox_resources)):
            sandbox_file = sandbox_resources[i]
            if sandbox_file['is_dynamic']:
                errors.append(self.NotImplementedException('Sandbox files with not false is_dynamic is not supported', path + [i, 'is_dynamic']))
                continue

            try:
                sandbox_resource = self._get_sandbox_resources_from_task_by_type(sandbox_file['task_id'], sandbox_file['resource_type'])
            except Exception as e:
                logging.warning("Sandbox task %s: %s at %s", sandbox_file['task_id'], e, path + [i])
                continue

            if sandbox_file['task_type'] == 'BUILD_JUGGLER_CHECKS_BUNDLE':
                box_juggler_id = unit_id
                resource = self.stage.spec.deploy_units[unit_id].box_juggler_configs[box_juggler_id].archived_checks.add()
                resource.checksum.value = sandbox_resource['md5'] if sandbox_resource.get('md5') else ''
                resource.checksum.type = downloadable_resources_pb2.MD5 if 'md5' in sandbox_resource else downloadable_resources_pb2.EMPTY
                resource.url = sandbox_resource['http']['proxy']
                continue

            if sandbox_file.get('extract_path'):
                errors.append(self.NotImplementedException('Sandbox files with not empty extract_path is not supported', path + [i, 'extract_path']))
                continue

            if self.__is_instance_ctl_conf_file(sandbox_file['local_path']):
                try:
                    instancectl_conf = self._get_url_content(sandbox_resource['http']['proxy'])
                    self._set_instancectl_conf(instancectl_conf, path + [i, 'local_path', 'instancectl'])
                except self.NotImplementedException as e:
                    errors.append(e)
                except self.NotImplementedExceptionList as el:
                    errors.extend(el.errors)
                continue

            resource = pod_agent_payload_spec.resources.static_resources.add()
            resource.id = sandbox_file['resource_type']
            resource.verification.checksum = ('MD5:' + sandbox_resource['md5']) if sandbox_resource.get('md5') else 'EMPTY:'
            resource.url = sandbox_resource['skynet_id']
            mounted = box.static_resources.add()
            mounted.resource_ref = resource.id
            mounted.mount_point = '/' + resource.id  # directory
            logging.warning('Resource %s will be placed in /%s. Please, reconfigure your application.' % (str(sandbox_resource['id']), resource.id))

        if errors:
            raise self.NotImplementedExceptionList(errors)

    def _set_workload_readiness_check(self, readiness_probe, path):
        unit_id = self.spec["_id"]
        workload_id = self.spec["runtime_attrs"]["content"]["instance_spec"]["containers"][path[-2]]["name"]
        workload = self._get_workload(unit_id, workload_id)
        sel_i = None
        for i in range(len(readiness_probe.get("handlers", []))):
            self._assert_equal_one_of_enum(['HTTP_GET', 'TCP_SOCKET', 'EXEC'])(readiness_probe["handlers"][i]["type"], path + ["handlers", i, "type"])
            if sel_i is None:
                sel_i = i
            else:
                logging.warning('YD supports only single readiness_check for workload, got %s and %s, will use first one' % (readiness_probe["handlers"][sel_i], readiness_probe["handlers"][i]))

        if sel_i is not None:
            if readiness_probe["handlers"][sel_i]["type"] == 'HTTP_GET':
                port = readiness_probe["handlers"][sel_i]["httpGet"]["port"]

                logging.warning('In YD response is successful if status code >= 200 and <300, in Nanny if < 400')

                workload.readiness_check.http_get.port = self.__convert_port_with_bsconfig(port)

                workload.readiness_check.http_get.path = readiness_probe["handlers"][sel_i]["httpGet"]["path"]
                workload.readiness_check.http_get.any = True
                self.__set_time_limit(workload.readiness_check.http_get.time_limit, readiness_probe)

            elif readiness_probe["handlers"][sel_i]["type"] == 'TCP_SOCKET':
                port = readiness_probe["handlers"][sel_i]["tcpSocket"]["port"]

                workload.readiness_check.tcp_check.port = self.__convert_port_with_bsconfig(port)

                self.__set_time_limit(workload.readiness_check.tcp_check.time_limit, readiness_probe)

            elif readiness_probe["handlers"][sel_i]["type"] == 'EXEC':
                workload.readiness_check.container.command_line = cast_command_to_str(readiness_probe["handlers"][sel_i]["execAction"]["command"])
                self.__set_time_limit(workload.readiness_check.container.time_limit, readiness_probe)

    def _set_workload_stop_policy(self, lifecycle, path):
        pre_stop = lifecycle.get("preStop")
        if not pre_stop:
            return
        self._assert_equal_one_of_enum(['NONE', 'HTTP_GET', 'EXEC'])(pre_stop["type"], path + ["preStop", "type"])
        if pre_stop["type"] == "NONE":
            return

        unit_id = self.spec["_id"]
        workload_id = self.spec["runtime_attrs"]["content"]["instance_spec"]["containers"][path[-2]]["name"]
        workload = self._get_workload(unit_id, workload_id)
        workload.stop_policy.max_tries = 1
        if pre_stop["type"] == "HTTP_GET":
            workload.stop_policy.http_get.port = self.__convert_port_with_bsconfig(pre_stop['httpGet'].get('port', ''))
            workload.stop_policy.http_get.path = pre_stop['httpGet']['path']
            workload.stop_policy.http_get.any = True
        elif pre_stop["type"] == "EXEC":
            workload.stop_policy.container.command_line = cast_command_to_str(pre_stop["execAction"]["command"])

    def _set_workload_start_command_line(self, commands, path):
        unit_id = self.spec["_id"]
        workload_id = self.spec["runtime_attrs"]["content"]["instance_spec"]["containers"][path[-2]]["name"]
        workload = self._get_workload(unit_id, workload_id)
        workload.start.command_line = cast_command_to_str(commands)
        self.__check_not_yasmagent(workload.start.command_line, path)

    def _set_juggler_settings(self, juggler_host_name, path):
        unit_id = self.spec["_id"]
        if self.spec["info_attrs"]["content"]["monitoring_settings"]["juggler_settings"]["is_enabled"]:
            juggler_subagent = self.stage.spec.deploy_units[unit_id].multi_cluster_replica_set.replica_set.pod_template_spec.spec.host_infra.monitoring.juggler_subagents.add()
            juggler_subagent.path = juggler_host_name

    def _set_deployment_strategy(self, instances_count):
        unit_id = self.spec["_id"]

        operating_degrade_level = 0.1  # default
        stop_degrade_level = 0.1  # default
        for recipe in self.spec["info_attrs"]["content"]["recipes"]["content"]:
            if recipe["name"] == "_activate_only_service_configuration.yaml":
                for context in recipe["context"]:
                    if context["key"] == "operating_degrade_level":
                        operating_degrade_level = float(context["value"])

        if operating_degrade_level != stop_degrade_level:
            logging.warning('YD does not support stop_degrade_level - it behaves as if operating_degrade_level (%s) is equal to stop_degrade_level (%s).', operating_degrade_level, stop_degrade_level)

        self.stage.spec.deploy_units[unit_id].multi_cluster_replica_set.replica_set.deployment_strategy.max_unavailable = max(
            int(instances_count * operating_degrade_level),
            1
        )

    def _set_static_resource_from_static_files(self, static_files, path):
        resource = pod_agent_pb2.TResource()
        resource.id = 'STATIC_FILES'
        resource.verification.checksum = 'EMPTY:'

        errors = []
        for i in range(len(static_files)):
            static_file = static_files[i]
            if static_file['is_dynamic']:
                errors.append(self.NotImplementedException('Dynamic static files is not supported', path + [i, 'is_dynamic']))
                continue
            if self.__is_instance_ctl_conf_file(static_file['local_path']):
                try:
                    self._set_instancectl_conf(static_file['content'], path + [i, 'local_path', 'instancectl'])
                except self.NotImplementedException as e:
                    errors.append(e)
                except self.NotImplementedExceptionList as el:
                    errors.extend(el.errors)
                continue
            if self.__is_conf_local_file(static_file['local_path']):
                continue  # needed only for instancectl migration
            file = resource.files.files.add()
            file.file_name = static_file['local_path']
            file.raw_data = static_file['content']

        if len(resource.files.files):
            unit_id = self.spec["_id"]
            pod_agent_payload_spec = self._get_pod_agent_payload_spec(unit_id)
            pod_agent_payload_spec.resources.static_resources.add().CopyFrom(resource)
            box = self._get_box(unit_id)
            mounted = box.static_resources.add()
            mounted.resource_ref = resource.id
            mounted.mount_point = '/static_files'
            logging.warning('Your static_files will be at %s. Please, reconfigure your application.', mounted.mount_point)

        if errors:
            raise self.NotImplementedExceptionList(errors)

    def _set_extended_gencfg_groups(self, groups, path):
        if not groups:
            return
        unit_id = self.spec["_id"]
        pod_resources = self._get_pod_template_spec_spec(unit_id).resource_requests
        pod_disk_volumes = self._get_pod_template_spec_spec(unit_id).disk_volume_requests
        pod_count_in_cluster = dict()

        big_disk_volume = None

        network_default = None
        for i in range(len(groups)):
            request, cluster = self._get_yp_lite_gencfg_group(groups[i]["name"], unit_id)
            if i == 0:
                resources = request
            else:
                if resources != request:
                    logging.warn(
                        "Gencfg group resources not equal\ngencfg_group:%s\n,resources:%s" %
                        (str([groups[0]["name"], groups[i]["name"]]), str([resources, request])))
            pod_count_in_cluster[cluster] = pod_count_in_cluster.get(cluster, 0) + request['replicas']

            if request["networkMacro"]:
                if not network_default:
                    network_default = request["networkMacro"]
            else:
                logging.warning("NetworkMacro is not found for %s gencfg group migration (probably, you are not using MTN) at %s", groups[i]["name"], path + [i])

            logging.warn("Multiple disk_volume_requests are not supported at YD yet, will sum up all disk qouta at one volume")
            for disk_volume in request['persistentVolumes']:
                if big_disk_volume is None:
                    big_disk_volume = {
                        "quota_policy": {
                            "capacity": 0,
                            "bandwidth_limit": 0,
                            "bandwidth_guarantee": 0
                        },
                        "storage_class": disk_volume['storageClass']
                    }
                big_disk_volume['quota_policy']['capacity'] += (int(disk_volume['diskQuotaMegabytes']) << 20)
                big_disk_volume['quota_policy']['bandwidth_guarantee'] += (int(disk_volume['bandwidthGuaranteeMegabytesPerSec']) << 20)
                big_disk_volume['quota_policy']['bandwidth_limit'] += (int(disk_volume['bandwidthLimitMegabytesPerSec']) << 20)

                volume = self._get_pod_agent_payload_spec(unit_id).volumes.add()
                volume.id = disk_volume['mountPoint'][1:]
                box = self._get_box(unit_id)
                mounted_volume = box.volumes.add()
                mounted_volume.volume_ref = volume.id
                mounted_volume.mount_point = disk_volume['mountPoint']
                mounted_volume.mode = pod_agent_pb2.EVolumeMountMode_READ_WRITE

        if big_disk_volume is not None:
            pod_disk_volume = pod_disk_volumes.add()
            pod_disk_volume.id = 'root_fs_' + unit_id
            pod_disk_volume.quota_policy.capacity = int(big_disk_volume['quota_policy']['capacity'])
            if pod_disk_volume.quota_policy.capacity < (migrator_base.BOUND_FOR_DISK_TAX_IN_GB << 30):
                pod_disk_volume.quota_policy.capacity += (migrator_base.DISK_TAX_IN_GB << 30)
                logging.warning('Add {}GB to disk capacity for pod_agent infrastructure'.format(migrator_base.DISK_TAX_IN_GB))

            pod_disk_volume.quota_policy.bandwidth_guarantee = int(big_disk_volume['quota_policy']['bandwidth_guarantee'])
            pod_disk_volume.quota_policy.bandwidth_limit = int(big_disk_volume['quota_policy']['bandwidth_limit'])
            pod_disk_volume.storage_class = big_disk_volume['storage_class']
            label = pod_disk_volume.labels.attributes.add()
            label.key = 'used_by_infra'
            label.value = yson.dumps(True)

        if not network_default:
            network_default = '_SEARCHSAND_'
            logging.warning("Could not found any network macro, will use %s", network_default)
        self.stage.spec.deploy_units[unit_id].network_defaults.network_id = network_default

        pod_resources.vcpu_guarantee = int(resources['vcpuGuarantee'])
        pod_resources.vcpu_limit = int(resources['vcpuLimit'])
        pod_resources.memory_guarantee = (int(resources['memoryGuaranteeMegabytes']) << 20)
        pod_resources.memory_limit = (int(resources['memoryGuaranteeMegabytes']) << 20)

        clusters = self.stage.spec.deploy_units[unit_id].multi_cluster_replica_set.replica_set.clusters
        instances_count = 0
        for cluster_name, pod_count in pod_count_in_cluster.items():
            cluster_info = clusters.add()
            cluster_info.cluster = cluster_name.lower()
            cluster_info.spec.replica_count = pod_count
            constraint = cluster_info.spec.constraints.antiaffinity_constraints.add()
            constraint.key = 'node'
            constraint.max_pods = 1
            instances_count += pod_count

        self._set_deployment_strategy(instances_count)

    def _set_env(self, nanny_envs, path):
        errors = []

        unit_id = self.spec["_id"]
        workload_id = self.spec["runtime_attrs"]["content"]["instance_spec"]["containers"][path[-2]]["name"]

        nanny_secret_names = set()
        for i in range(len(nanny_envs)):
            if nanny_envs[i]["valueFrom"]["type"] == 'SECRET_ENV':
                nanny_secret_names.add(nanny_envs[i]['name'])

        secret_from_nanny_id = None

        if self.nanny_token and len(nanny_secret_names) > 0:
            secrets = []

            environ = None
            try:
                environ = self._run_remote_call_on_instance(self.instance, 'cat /proc/$(cat pids/{})/environ'.format(workload_id))
            except Exception as e:
                logging.warning("Secrets from nanny vault skipped, because an error occurred while getting the secret values {}".format(str(e)))

            if environ is not None:
                for item in environ.split('\0'):
                    if item:
                        key, value = item.split('=', 1)
                        if key in nanny_secret_names:
                            secrets.append({
                                'key': key,
                                'value': value,
                            })

                if len(secrets):
                    vault_client = VaultClient()
                    secret_from_nanny_id = 'nanny-{}-workload-{}'.format(self._get_stage_id(''), workload_id)
                    secret_uuid = vault_client.create_secret(secret_from_nanny_id)
                    version_uuid = vault_client.create_secret_version(
                        secret_uuid,
                        secrets
                    )

                    logging.warning('Secret ACL (for users/abc services) werent migrated - secret is only available for you and stage instances.')
                    logging.warning('Please, update yav ACL manually at https://yav.yandex-team.ru/secret/%s', secret_uuid)

        workload = self._get_workload(unit_id, workload_id)
        for i in range(len(nanny_envs)):
            nanny_env = nanny_envs[i]
            if nanny_env["valueFrom"]["type"] == "VAULT_SECRET_ENV":
                if self.nanny_token:
                    secret_id = "%s:%s" % (nanny_env["valueFrom"]["vaultSecretEnv"]["vaultSecret"]["secretId"],
                                           nanny_env["valueFrom"]["vaultSecretEnv"]["vaultSecret"]["secretVer"])
                    client = VaultClient()
                    try:
                        self._create_vault_token(
                            client=client,
                            secret_id=nanny_env["valueFrom"]["vaultSecretEnv"]["vaultSecret"]["secretId"],
                            secret_version=nanny_env["valueFrom"]["vaultSecretEnv"]["vaultSecret"]["secretVer"],
                            object_id=self.spec["_id"],
                            unit_id=unit_id,
                            deploy_secret_id=secret_id
                        )
                    except Exception as e:
                        logging.warning('Could not create delegation token for YAV secret %s: %s', secret_id, str(e))
                    else:
                        env = workload.env.add()
                        env.name = nanny_env["name"]
                        env.value.secret_env.id = nanny_env["valueFrom"]["vaultSecretEnv"]["field"]
                        env.value.secret_env.alias = secret_id

            elif nanny_env["valueFrom"]["type"] == "LITERAL_ENV":
                env = workload.env.add()
                env.name = nanny_env["name"]
                env.value.literal_env.value = nanny_env["valueFrom"]["literalEnv"]["value"]

            elif nanny_env["valueFrom"]["type"] == "SECRET_ENV":
                if self.nanny_token and secret_from_nanny_id:
                    secret_id = secret_from_nanny_id
                    client = VaultClient()
                    try:
                        self._create_vault_token(
                            client=client,
                            secret_id=secret_uuid,
                            secret_version=version_uuid,
                            object_id=self.spec["_id"],
                            unit_id=unit_id,
                            deploy_secret_id=secret_id
                        )
                    except Exception as e:
                        logging.warning('Could not create delegation token for YAV secret %s: %s', secret_id, str(e))
                    else:
                        env = workload.env.add()
                        env.name = nanny_env["name"]
                        env.value.secret_env.id = env.name
                        env.value.secret_env.alias = secret_id

        if errors:
            raise self.NotImplementedExceptionList(errors)

    def _set_init_containers(self, init_containers, path):
        for init_container in init_containers:
            unit_id = self.spec["_id"]
            box = self._get_box(unit_id)
            init = box.init.add()
            init.command_line = cast_command_to_str(init_container["command"])

    def _set_yav_vault_secret_volume(self, vault_secret, path):
        unit_id = self.spec["_id"]

        if self.spec["runtime_attrs"]["content"]["instance_spec"]["volume"][path[-2]]["type"] != "VAULT_SECRET":
            return

        volume_name = self.spec["runtime_attrs"]["content"]["instance_spec"]["volume"][path[-2]]["name"]

        self._fill_vault_volume(
            secret_id=vault_secret["vaultSecret"]["secretId"],
            secret_version=vault_secret["vaultSecret"]["secretVer"],
            resource_id='VAULT_SECRET_%d' % path[-2],
            volume_name=volume_name,
            unit_id=unit_id
        )

    def _set_nanny_vault_secret_volume(self, nanny_vault_secret, path):
        unit_id = self.spec["_id"]

        if self.spec["runtime_attrs"]["content"]["instance_spec"]["volume"][path[-2]]["type"] != "SECRET":
            return

        volume_name = self.spec["runtime_attrs"]["content"]["instance_spec"]["volume"][path[-2]]["name"]

        secret_uuid = version_uuid = None

        if self.nanny_token:
            try:
                secret_keys = self._run_remote_call_on_instance(self.instance, 'find {} -maxdepth 1 -type f'.format(volume_name)).split()
            except Exception as e:
                logging.warning("Volume {}: Secrets skipped, because an error occurred while getting the secret values list {}".format(volume_name, str(e)))
            else:
                secrets = []
                for secret_key in secret_keys:
                    try:
                        secrets.append({
                            'key': secret_key.split('/')[1],  # secret_key = <volume_name>/<key>
                            'value': self._run_remote_call_on_instance(self.instance, 'cat {}'.format(secret_key))
                        })
                    except Exception as e:
                        logging.warning("Volume {}: Secret {} is skipped, because an error occurred while getting the secret value {}".format(volume_name, secret_key, str(e)))

                if len(secrets) > 0:
                    vault_client = VaultClient()
                    secret_uuid = vault_client.create_secret('nanny-{}-volume-{}'.format(self._get_stage_id(''), volume_name))

                    version_uuid = vault_client.create_secret_version(
                        secret_uuid,
                        secrets
                    )

                    logging.warning('Secret ACL (for users/abc services) werent migrated - secret is only available for you and stage instances.')
                    logging.warning('Please, update yav ACL manually at https://yav.yandex-team.ru/secret/%s', secret_uuid)

                    self._fill_vault_volume(
                        secret_id=secret_uuid,
                        secret_version=version_uuid,
                        resource_id='VAULT_SECRET_%d' % path[-2],
                        volume_name=volume_name,
                        unit_id=unit_id
                    )
                elif len(secret_keys) > 0:
                    logging.warning("Volume {} skipped".format(volume_name))

    def _fill_vault_volume(self, secret_id, secret_version, resource_id, volume_name, unit_id):
        pod_agent_payload_spec = self._get_pod_agent_payload_spec(unit_id)

        resource = pod_agent_payload_spec.resources.static_resources.add()
        resource.id = resource_id
        resource.verification.checksum = 'EMPTY:'

        if self.nanny_token:
            client = VaultClient()
            version = client.get_version(secret_version)

            self._create_vault_token(
                client=client,
                secret_id=secret_id,
                secret_version=secret_version,
                object_id=self.spec["_id"],
                unit_id=unit_id,
                deploy_secret_id="%s:%s" % (secret_id, secret_version)
            )

            for key in version["value"].iterkeys():
                file = resource.files.files.add()
                file.file_name = key
                file.secret_data.id = key
                file.secret_data.alias = "%s:%s" % (secret_id, secret_version)

        box = self._get_box(unit_id)
        mounted = box.static_resources.add()
        mounted.resource_ref = resource.id
        mounted.mount_point = "/%s" % volume_name

    def _set_host_provided_daemon(self, daemon, path):
        self._assert_equal_one_of_enum(["HOST_SKYNET", "YASM_AGENT"])(daemon, path)
        if daemon == "HOST_SKYNET":
            unit_id = self.spec["_id"]
            box = self._get_box(unit_id)
            box.bind_skynet = True
            logging.warning('Assuming that instance layers contain symlinks needed for bind_skynet ( https://wiki.yandex-team.ru/deploy/docs/podstack/box/#bindskynet )')
        elif daemon == "YASM_AGENT":
            pass  # not supported yet, usage checks are made at containers/command and intancectl migration

    def _set_aux_daemon(self, daemon, path):
        self._assert_equal_one_of_enum(["JUGGLER_AGENT", "LOGROTATE"])(daemon["type"], path + ["type"])
        if daemon["type"] == "JUGGLER_AGENT":
            pass  # daemon is being added at self._set_juggler_settings
        elif daemon["type"] == "LOGROTATE":
            pass  # will be covered by reopenLogAction implementation

    def _set_resolv_conf(self, resolv_conf, path):
        self._assert_equal_one_of_enum(["DEFAULT_RESOLV_CONF", "USE_NAT64", "KEEP_RESOLV_CONF"])(resolv_conf, path)
        unit_id = self.spec["_id"]
        box = self._get_box(unit_id)
        if resolv_conf == "USE_NAT64":
            box.resolv_conf = pod_agent_pb2.EResolvConf_NAT64
        elif resolv_conf == "KEEP_RESOLV_CONF":
            box.resolv_conf = pod_agent_pb2.EResolvConf_KEEP

    def _set_unistat_endpoints(self, unistat_endpoints, path):
        unit_id = self.spec["_id"]
        host_infra = self._get_pod_template_spec_spec(unit_id).host_infra

        for unistat_endpoint in unistat_endpoints:
            unistat = host_infra.monitoring.unistats.add()
            unistat.port = self.__convert_port_with_bsconfig(unistat_endpoint["port"])
            unistat.path = unistat_endpoint.get("path", '/')
            unistat.workload_id = self.spec["runtime_attrs"]["content"]["instance_spec"]["containers"][path[-2]]["name"]
            unistat.output_format = host_infra_pb2.TMonitoringUnistatEndpoint.EOutputFormat.OF_YASM_JSON

    def _set_recipe_context(self, context, path):
        # operating_degrade_level/stop_degrade_level implemented at self._set_deployment_strategy
        self._assert_equal_one_of_enum([
            "force",  # skip force
            "operating_degrade_level",
            "prepare_operating_degrade_level",
            "stop_degrade_level",
        ])(context['key'], path + ['key'])
        if context['key'] == "prepare_operating_degrade_level":
            if float(context['value']) != 1:  # skip default
                logging.warning('prepare_operating_degrade_level = %s will be skipped - YD does not support prepare stage yet', context['value'])

    def _set_deploy_monitoring(self, deploy_monitoring, path):
        self._assert_equal(False)(deploy_monitoring["is_enabled"], path + ["is_enabled"])

    def _set_docker_image(self, docker_image, path):
        if self.spec["runtime_attrs"]["content"]["instance_spec"]["type"] != 'DOCKER_LAYERS':
            return
        unit_id = self.spec["_id"]
        image = self.stage.spec.deploy_units[unit_id].images_for_boxes[unit_id]
        image.registry_host = docker_image["registry"]
        image.name = docker_image["name"].split(':')[0]
        image.tag = docker_image["name"].split(':')[1]

    def _set_coredump_processor_option(self, processor_option):
        def f(option, path):
            containers = self.spec["runtime_attrs"]["content"]["instance_spec"]["containers"]
            if processor_option == 'cleanup_ttl_seconds':
                idx = path[-6]
                if containers[idx]["coredumpPolicy"]["coredumpProcessor"]["cleanupPolicy"]["type"] != "TTL":
                    return
            else:
                idx = path[-4]
            if containers[idx]["coredumpPolicy"]["type"] != "COREDUMP":
                return
            unit_id = self.spec["_id"]
            workload_id = self.spec["runtime_attrs"]["content"]["instance_spec"]["containers"][idx]["name"]
            coredump_processor = self.stage.spec.deploy_units[unit_id].coredump_config[workload_id].coredump_processor
            setattr(coredump_processor, processor_option, option)
        return f

    def _set_coredump_aggregator_url(self, url, path):
        containers = self.spec["runtime_attrs"]["content"]["instance_spec"]["containers"]
        if containers[path[-6]]["coredumpPolicy"]["coredumpProcessor"]["aggregator"]["type"] != "SAAS_AGGREGATOR":
            return
        unit_id = self.spec["_id"]
        workload_id = self.spec["runtime_attrs"]["content"]["instance_spec"]["containers"][path[-6]]["name"]
        self.stage.spec.deploy_units[unit_id].coredump_config[workload_id].coredump_processor.aggregator.enabled = True
        if url != "http://cores.n.yandex-team.ru/corecomes" and url != "https://coredumps.n.yandex-team.ru/submit_core":  # skip defaults
            self.stage.spec.deploy_units[unit_id].coredump_config[workload_id].coredump_processor.aggregator.url = url

    #
    # Release rule
    #

    def _set_release_rule_desc(self, desc, path):
        if not self.spec["info_attrs"]["content"]["tickets_integration"]["service_release_tickets_enabled"]:
            return
        release_rule = self.__get_release_rule(path[-2])
        release_rule.spec.description = desc

    def _set_release_rule_auto_commit_policy(self, auto_commit_settings, path):
        if not self.spec["info_attrs"]["content"]["tickets_integration"]["service_release_tickets_enabled"]:
            return
        if auto_commit_settings.get('enabled'):
            scheduling_policy = self.spec["info_attrs"]["content"]["scheduling_policy"]["type"]
            self._assert_equal_one_of_enum(["NONE", "MAINTAIN_ACTIVE_TRUNK", "SEQUENTIAL_SNAPSHOT_ACTIVATION"])
            (scheduling_policy, ["info_attrs", "contsent", "scheduling_policy", "type"])

            release_rule = self.__get_release_rule(path[-2])
            if scheduling_policy == "MAINTAIN_ACTIVE_TRUNK":
                release_rule.spec.auto_commit_policy.type = release_rule_pb2.TAutoCommitPolicy.EType.MAINTAIN_ACTIVE_TRUNK
            elif scheduling_policy == "SEQUENTIAL_SNAPSHOT_ACTIVATION":
                release_rule.spec.auto_commit_policy.type = release_rule_pb2.TAutoCommitPolicy.EType.SEQUENTIAL_COMMIT

    def _set_release_rule_sandbox_task_type(self, sandbox_task_type, path):
        if not self.spec["info_attrs"]["content"]["tickets_integration"]["service_release_tickets_enabled"]:
            return
        logging.warning(
            "Please, check that your tickets_integration sandbox_task_type %s is compatible with YD release integration: "
            "https://wiki.yandex-team.ru/deploy/docs/release-integration/#sozdanierelizov",
            sandbox_task_type
        )
        release_rule = self.__get_release_rule(path[-2])
        release_rule.spec.sandbox.task_type = sandbox_task_type
        release_rule.meta.id = self._get_stage_id(self.spec["_id"]) + "_" + str(path[-2])
        release_rule.meta.stage_id = self._get_stage_id(self.spec["_id"])

    def _set_release_rule_sandbox_resource_type(self, sandbox_resource_type, path):
        if not self.spec["info_attrs"]["content"]["tickets_integration"]["service_release_tickets_enabled"]:
            return
        release_rule = self.__get_release_rule(path[-2])
        for resource in self.spec["runtime_attrs"]["content"]["resources"]["sandbox_files"]:
            if resource['resource_type'] == sandbox_resource_type:
                if resource['resource_type'] in release_rule.spec.patches:
                    raise self.NotImplementedException('release rule patches conflict: %s' % resource['resource_type'], path)
                patch = release_rule.spec.patches[resource['resource_type']]
                patch.sandbox.sandbox_resource_type = sandbox_resource_type
                patch.sandbox.static.deploy_unit_id = self.spec["_id"]
                patch.sandbox.static.static_resource_ref = resource['resource_type']
        for layer in self.spec["runtime_attrs"]["content"]["instance_spec"].get("layersConfig", {}).get("layer", []):
            layer_resource_type = layer['fetchableMeta']['sandboxResource']['resourceType']
            if layer_resource_type == sandbox_resource_type:
                if layer_resource_type in release_rule.spec.patches:
                    raise self.NotImplementedException('release rule patches conflict: %s' % layer_resource_type, path)
                patch = release_rule.spec.patches[layer_resource_type]
                patch.sandbox.sandbox_resource_type = sandbox_resource_type
                patch.sandbox.static.deploy_unit_id = self.spec["_id"]
                patch.sandbox.static.layer_ref = layer_resource_type

    def _set_release_rule_sandbox_release_types(self, expression, path):
        if not self.spec["info_attrs"]["content"]["tickets_integration"]["service_release_tickets_enabled"]:
            return

        prefix = "sandbox_release.release_type in ("
        if not expression.startswith(prefix):
            raise self.NotImplementedException('expression "%s" is not supported' % expression, path)

        release_types = expression[len(prefix):-1].split(',')
        release_rule = self.__get_release_rule(path[-3])
        for rt in release_types:
            rt = str(rt).translate(None, '" ')
            if rt:
                self._assert_equal_one_of(["unstable", "testing", "prestable", "stable"])(rt, path)
                release_rule.spec.sandbox.release_types.append(rt)

    def _get_service_instance(self, service_id):
        if self.nanny_token:
            url = 'http://nanny.yandex-team.ru/v2/services/{}/current_state/instances'.format(service_id)
            response = self.nanny_session.get(url)
            json = response.json()
            return json['result'][0]['container_hostname']
        return None

    #
    # STAFF API
    #

    def _staff_id_to_abc_group(self, staff_id):
        if self.nanny_token is None:
            return 'staff:%s' % staff_id
        else:
            url = ('https://staff-api.yandex-team.ru/v3/groups?'
                   '_one=1&'
                   '_fields=service.id&'
                   'id={}'.format(staff_id))
            headers = {
                'Authorization': 'OAuth {}'.format(self.nanny_token),
            }
            resp = requests.get(url, headers=headers, timeout=5)
            resp.raise_for_status()
            data = resp.json()
            if data['service']['id']:
                return 'abc:service:%s' % str(data['service']['id'])
            else:
                return None

    #
    # YP API
    #

    def _get_pod_resource_requests(self, pod_id, address):
        if self.yp_token is None:
            return {
                "vcpu_limit": 1,
                "vcpu_guarantee": 2,
                "memory_limit": 3,
                "memory_guarantee": 4
            }
        yp_client = yp.client.YpClient(
            address=address,
            config={
                'token': self.yp_token
            }
        )
        return yp_client.get_object(object_identity=pod_id, object_type="pod", selectors=["/spec/resource_requests"])[0]

    def _get_pod_disk_volume_requests(self, pod_id, address):
        if self.yp_token is None:
            return [{
                "quota_policy": {
                    "capacity": 1,
                    "bandwidth_limit": 0,
                    "bandwidth_guarantee": 0
                },
                "id": "disk_id",
                "storage_class": "hdd",
                "labels": {
                    "mount_path": "/mpath",
                    "volume_type": "root_fs"
                }
            }]
        yp_client = yp.client.YpClient(
            address=address,
            config={
                'token': self.yp_token
            }
        )
        return yp_client.get_object(object_identity=pod_id, object_type="pod", selectors=["/spec/disk_volume_requests"])[0]

    def _get_pod_ip6_address_requests(self, pod_id, address):
        if self.yp_token is None:
            return [{
                "enable_dns": True,
                "network_id": "_NETWORK_",
                "vlan_id": "fastbone"
            }]
        yp_client = yp.client.YpClient(
            address=address,
            config={
                'token': self.yp_token
            }
        )
        return yp_client.get_object(object_identity=pod_id, object_type="pod", selectors=["/spec/ip6_address_requests"])[0]

    def _get_yp_lite_gencfg_group(self, group_name, service_id):
        if self.nanny_token:
            response = self.nanny_session.post(
                'http://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/ComputeResourceAllocation/',
                json={
                    'gencfg_group': group_name,
                    'service_id': service_id,
                },
                timeout=42
            )
            json = response.json()
            return json['request'], json["cluster"].lower()
        else:
            return {
                "memoryGuaranteeMegabytes": 32768,
                "replicas": 3,
                "vcpuLimit": 7500,
                "vcpuGuarantee": 7500,
                "persistentVolumes": [
                    {
                        "diskQuotaMegabytes": 773120,
                        "storageClass": "hdd",
                        "bandwidthLimitMegabytesPerSec": 0,
                        "diskQuotaGigabytes": 0,
                        "mountPoint": "\/qemu-persistent",
                        "bandwidthGuaranteeMegabytesPerSec": 0
                    }
                ],
                "enableInternet": False,
                "networkMacro": "_GENCFG_MSK_MYT_MAPS_ADV_DEV_",
            }, "myt"

    #
    # MIGRATE
    #

    def migrate(self):
        self._migrate_spec()
        return self.stage, self.release_rules
