import migrator_base

from infra.awacs.proto import api_pb2, model_pb2, modules_pb2

from yp_proto.yp.client.api.proto import downloadable_resources_pb2
from yp_proto.yp.client.api.proto import host_infra_pb2
from yp_proto.yp.client.api.proto import pod_agent_pb2
from yp_proto.yp.client.api.proto import stage_pb2

import yt.yson as yson

from library.python.vault_client.instances import Production as VaultClient

import hashlib
import json
import logging
import re


class QloudMigrator(migrator_base.MigratorBase):
    def __init__(
        self,
        spec,
        stage_id,
        raise_on_error,
        yav_secret_alias=None,
        awacs_category='',
        awacs_namespace_id='',
        awacs_alerting_recipient=None,
        mock_clients=False,
    ):
        super(QloudMigrator, self).__init__(
            migrator_name="qloud",
            spec=spec,
            stage_id=stage_id,
            raise_on_error=raise_on_error,
            implemented_map={
                "abcId": self._set_abc_id,
                "domains": {
                    self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                        "CA": self._set_domain_certificate_ca,
                        "certificateIssuer": self.EImplemented.IMPLEMENTED,
                        "certificateMd5": self.EImplemented.IMPLEMENTED,
                        "certificates": self.EImplemented.IMPLEMENTED,  # duplicates fields above
                        "certificateSecretId": self.EImplemented.IMPLEMENTED,  # should be implemented, active only when certificateSource==SECRET
                        "certificateSerialNumber": self.EImplemented.IMPLEMENTED,
                        "certificateSha1": self.EImplemented.IMPLEMENTED,
                        "certificateSha256": self.EImplemented.IMPLEMENTED,
                        "certificateSource": self._warn_if_not_equal_one_of(
                            ["CERTIFICATOR", "NONE"],
                            "All certificates should be requested via Certificator. Will try to order it via Certificator - check awacs.yaml"
                        ),
                        "certificateValidFor": self.EImplemented.IMPLEMENTED,
                        "certificateValidNotAfter": self.EImplemented.IMPLEMENTED,
                        "certificateValidNotBefore": self.EImplemented.IMPLEMENTED,
                        "clientCertificateRequired": self._assert_equal(False),
                        "domainName": self._set_domain_id_and_host,
                        "domainStatus": self.EImplemented.IMPLEMENTED,
                        "errorPage": self._warn_if_not_equal(False, "custom error page is not supported"),
                        "errorPageUrl": self.EImplemented.IMPLEMENTED,
                        "httpAllowed": self._set_domain_https_only,
                        "httpOnly": self._set_domain_http_only,
                        "issueCertificateTicket": self.EImplemented.IMPLEMENTED,
                        "l7path": self._warn_if_not_equal("", "Qloud CDN is deprecated. Please, migrate to S3 https://wiki.yandex-team.ru/users/cheetah/migracija-proektov-v-s3/#kakpereexat"),
                        "moveTargets": self.EImplemented.IMPLEMENTED,
                        "published": self.EImplemented.IMPLEMENTED,
                        "requestClientCertificate": self._assert_equal(False),
                        "requester": self.EImplemented.IMPLEMENTED,
                        "statuses": self.EImplemented.IMPLEMENTED,
                        "type": self.EImplemented.IMPLEMENTED,
                        "url": self.EImplemented.IMPLEMENTED,
                        "warning": self.EImplemented.IMPLEMENTED,
                    }
                },
                "dump": {
                    "comment": self.EImplemented.IMPLEMENTED,  # not needed
                    "components": {
                        self.EImplemented.IMPLEMENTED: self._set_secrets_and_tvm,  # aggregate secrets from all components
                        self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                            "activateRecipe": {
                                "doneThreshold": self.EImplemented.IMPLEMENTED,
                                "recipe": self._assert_equal('INTERNAL'),
                                "updateLimit": self.EImplemented.IMPLEMENTED,
                                "updatePeriod": self.EImplemented.IMPLEMENTED,
                                "updateWindow": self._set_replica_set_deployment_strategy,
                            },
                            "componentFeatures": {
                                self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: self._set_feature,
                            },
                            "componentName": self.EImplemented.IMPLEMENTED,
                            "componentType": self._assert_equal_one_of_enum(['standard', 'component-proxy', 'proxy']),
                            "embedJugglerClient": self.EImplemented.IMPLEMENTED,  # at _set_juggler_subagent
                            "environmentVariables": self._set_environment,
                            "instanceGroups": self._set_replica_set_clusters,
                            "jugglerBundleResources": self._set_juggler_subagent,
                            "l3Config": {
                                "l3UpdateWindow": self.EImplemented.SKIP,  # l3 config update is manual at YD/RTC
                                "trafficDisablingDelay": self.EImplemented.SKIP,  # l3 config update is manual at YD/RTC
                                "vss": self._warn_if_not_equal([], "L3 over component should be migrated manually. See https://wiki.yandex-team.ru/traffic/l3mgr/"),
                            },
                            "overlays": self._assert_equal([]),
                            "prepareRecipe": self._set_prepare_recipe,
                            "properties": {
                                "allocationFailThreshold": self.EImplemented.IMPLEMENTED,  # won't be supported
                                "allocationStrategy": self._set_allocation_strategy,
                                "allowedCpuNames": self._warn_if_not_equal('', 'allowedCpuNames ignored'),
                                "collocationParentComponent": self.EImplemented.IMPLEMENTED,  # at allocationStrategy
                                "componentEnvironment": self.EImplemented.IMPLEMENTED,  # always same as environmentVariables
                                "deployPolicy": self._warn_if_not_equal('InPlace', 'only InPlace deployment is supported at YD, stage will use InPlace'),
                                "diskSerial": self.EImplemented.IMPLEMENTED,  # ignore qloud persistency which is not working correctly
                                                                              # https://wiki.yandex-team.ru/qloud/doc/environment/component/disks/#storage
                                "diskSize": self._set_disk_volume_requests,
                                "dnsCache": self._assert_equal('true'),
                                "dnsNat64": self._set_dns_nat64,
                                "ediskSerial": self._assert_equal('0'),
                                "ediskSize": self._set_ephemeral_volume,
                                "failTimeout": self._warn_if_not_equal_one_of(
                                    ['60', '-1'],  # skip 60 which is default
                                    "YD will use readiness_check (instead of failTimeout/maxFails) for active balancing / service discovery"
                                    " https://wiki.yandex-team.ru/deploy/docs/podstack/workload/probes/#readiness"
                                ),
                                "fastboneRequired": self.EImplemented.IMPLEMENTED,  # YD requests fastbone for every pod
                                "hardwareSegment": self.EImplemented.IMPLEMENTED,  # not needed
                                "hash": self.EImplemented.IMPLEMENTED,  # not needed
                                "healthCheckFall": self.EImplemented.IMPLEMENTED,  # at useHealthCheck
                                "healthCheckHttpExpectedCode": self.EImplemented.IMPLEMENTED,  # at useHealthCheck
                                "healthCheckHttpUrl": self.EImplemented.IMPLEMENTED,  # at useHealthCheck
                                "healthCheckInterval": self.EImplemented.IMPLEMENTED,  # at useHealthCheck
                                "healthCheckRise": self.EImplemented.IMPLEMENTED,  # at useHealthCheck
                                "healthCheckTimeout": self.EImplemented.IMPLEMENTED,  # at useHealthCheck
                                "healthCheckType": self.EImplemented.IMPLEMENTED,  # at useHealthCheck
                                "host": self._set_header_host,
                                "httpCheckOn": self._assert_equal('true'),
                                "ioLimit": self._assert_equal('0'),
                                "isolationGroup": self._warn_if_not_equal('root', 'Will use root instead'),
                                "isolationUser": self._warn_if_not_equal('root', 'Will use root instead'),
                                "java":  self._warn_if_not_equal('false', 'java is not supported at YD, please add JVM layer manually at deploy.yaml'),
                                "listenBacklogSize": self._assert_equal('511'),
                                "maxFails": self._warn_if_not_equal_one_of(
                                    ['3', '0', '-1'],  # skip 3 which is default
                                    "YD will use readiness_check (instead of failTimeout/maxFails) for active balancing / service discovery"
                                    " https://wiki.yandex-team.ru/deploy/docs/podstack/workload/probes/#readiness"
                                ),
                                "maxInstancesPerHost": self.EImplemented.IMPLEMENTED,  # at _set_replica_set_clusters
                                "maxWorkerConnections": self.EImplemented.IMPLEMENTED,  # ignored
                                "minPrepared": self.EImplemented.IMPLEMENTED,  # ignored
                                "multiAccept": self._assert_equal('false'),
                                "network": self._set_network,
                                "notResolveServerName": self._assert_equal_one_of_enum(["false"]),
                                "preAuthenticate": self._assert_equal('false'),
                                "profiles": self.EImplemented.IMPLEMENTED,  # ignored
                                "proxyRedirect": self._assert_equal(''),
                                "qloudCoreDumpDirectory": self._assert_equal('/coredumps_qloud'),
                                "qloudCoreDumpFileSizeGb": self._assert_equal('0'),
                                "qloudInitPolicy": self.EImplemented.IMPLEMENTED,  # not needed
                                "qloudInitVersion": self.EImplemented.IMPLEMENTED,  # not needed
                                "qloudMaxCoreDumpedStopsRespawnDelay": self._assert_equal('0s'),
                                "qloudMaxCoreDumpsOnDisk": self._assert_equal('0'),
                                "repository": self._set_images_for_boxes,
                                "servers": self._set_endpoint_set_instances,
                                "size": self._set_resource_requests,
                                "sslStapling": self._assert_equal('false'),
                                "statusChecksCorrected": self._assert_equal('true'),
                                "stderr": self._set_workload_transmit_logs,
                                "stdout": self._set_workload_transmit_logs,
                                "storage": self.EImplemented.IMPLEMENTED,  # allocation 'storage' is ignored
                                "tmpfsDiskSerial": self._assert_equal('0'),
                                "tmpfsDiskSize": self._set_tmpfs_init_script,
                                "unistat": self._set_unistat,
                                "unistatPath": self.EImplemented.IMPLEMENTED,  # at _set_unistat
                                "unistatPort": self.EImplemented.IMPLEMENTED,  # at _set_unistat
                                "units": self._assert_equal('0'),
                                "upstreamPort": self._set_endpoint_set_port,
                                "useDockerUserGroup": self.EImplemented.IMPLEMENTED,  # YD uses docker user/group if its found https://wiki.yandex-team.ru/deploy/docs/stage/ispolzovanie-docker-obraza/
                                "usedVolumes": self.EImplemented.IMPLEMENTED,  # allocation 'storage' is ignored
                                "useHealthCheck": self.EImplemented.IMPLEMENTED,  # at _set_readiness_probe
                                "useHttps": self._assert_equal('false'),
                                "useTorrents": self.EImplemented.IMPLEMENTED,  # not needed (stage_ctl will use rbtorrents for docker layers)
                            },
                            "sandboxResources": self._set_static_resources,
                            "secrets": self.EImplemented.IMPLEMENTED,  # at _set_secrets_and_tvm
                            "statusHookChecks": self._set_readiness_probe,
                            "stopAction": self._set_stop_policy,
                            "tvmConfig": self.EImplemented.IMPLEMENTED,  # at _set_secrets_and_tvm
                            "upstreamComponents": self.EImplemented.IMPLEMENTED,  # at _set_upstream_endpoint_set
                        },
                    },
                    "engine": self._warn_if_not_equal('platform', 'only "platform" engine is supported for migration'),
                    "objectId": self._set_stage_id,
                    "routeSettings": {
                        self.EImplemented.IMPLEMENTED_LIST_FOR_EACH: {
                            "componentName": self._set_upstream_endpoint_set,
                            "defaultErrorPageUrl": self.EImplemented.IMPLEMENTED,
                            "defaultNoUriPart": self.EImplemented.IMPLEMENTED,
                            "defaultProxyBuffers": self.EImplemented.IMPLEMENTED,
                            "defaultProxyBufferSize": self.EImplemented.IMPLEMENTED,
                            "defaultProxyConnectTimeout": self.EImplemented.IMPLEMENTED,
                            "defaultProxyNextUpstream": self.EImplemented.IMPLEMENTED,
                            "defaultProxyNextUpstreamTimeout": self.EImplemented.IMPLEMENTED,
                            "defaultProxyNextUpstreamTries": self.EImplemented.IMPLEMENTED,
                            "defaultProxyPolicy": self.EImplemented.IMPLEMENTED,
                            "defaultProxyReadTimeout": self.EImplemented.IMPLEMENTED,
                            "defaultProxyWriteTimeout": self.EImplemented.IMPLEMENTED,
                            "defaultWebAuthCert": self.EImplemented.IMPLEMENTED,
                            "defaultWebAuthCookie": self.EImplemented.IMPLEMENTED,
                            "defaultWebAuthIdmRole": self.EImplemented.IMPLEMENTED,
                            "defaultWebAuthToken": self.EImplemented.IMPLEMENTED,
                            "errorPage": self._warn_if_not_equal_one_of(["INHERITED", "OFF"], "custom error page is not supported"),
                            "errorPageUrl": self.EImplemented.IMPLEMENTED,
                            "geo": self._set_upstream_outer_balancing_option_locality,
                            "location": self._set_upstream_matcher_and_rewrite_src,
                            "keepAliveTimeout": self._assert_equal_one_of_enum(["-1s"]),
                            "molly": self._assert_equal(False),
                            "noUriPart": self._assert_equal(False),
                            "preAuthenticate": self._assert_equal(False),
                            "proxyBuffering": self._set_proxy_buffering,
                            "proxyBuffers": self._assert_equal("-1"),
                            "proxyBufferSize": self._assert_equal("-1"),
                            "proxyConnectTimeout": self._set_upstream_inner_balancing_option("connect_timeout", str),
                            "proxyNextUpstream": self._set_upstream_proxy_next_upstream,
                            "proxyNextUpstreamTimeout": self._set_upstream_inner_balancing_option("backend_timeout", lambda x: str(x) + 's'),
                            "proxyNextUpstreamTries": self._set_upstream_inner_balancing_option("attempts", int),
                            "proxyPolicy": self._warn_if_not_equal("round_robin", "only round_robin is supported, will use round_robin"),
                            "proxyReadTimeout": self._set_upstream_inner_balancing_option("backend_read_timeout", lambda x: str(x) + 's', skip_default=["5", "-1"]),
                            "proxyRequestBuffering": self._assert_equal(False),
                            "proxyWriteTimeout": self._set_upstream_inner_balancing_option("backend_write_timeout", lambda x: str(x) + 's', skip_default=["5", "-1"]),
                            "resolver": {
                                "resolverTimeout": self._assert_equal_one_of_enum(["-1s"]),
                                "resolverValid": self._assert_equal_one_of_enum(["-1s"]),
                            },
                            "upstreamPath": self._set_upstream_rewrite_dst,
                            "wallarm": self._assert_equal(False),
                            "webAuthCert": self._assert_equal_one_of_enum(["DISABLED"]),
                            "webAuthCheckIdm": self._assert_equal(False),
                            "webAuthCookies": self._assert_equal_one_of_enum(["DISABLED"]),
                            "webAuthMode": self._assert_equal_one_of_enum(["OFF"]),
                            "webAuthToken": self._assert_equal_one_of_enum(["DISABLED"]),
                        }
                    },
                    "userEnvironment": self.EImplemented.IMPLEMENTED,  # always same as userEnvironmentMap
                    "userEnvironmentMap": self.EImplemented.IMPLEMENTED,  # at _set_environment
                },
                "stable": self.EImplemented.IMPLEMENTED,  # contains instances list needed for secret migration
            },
            mock_clients=mock_clients
        )

        self.create_namespace_request = api_pb2.CreateNamespaceRequest()
        self.create_namespace_request.meta.id = awacs_namespace_id
        self.create_namespace_request.meta.auth.type = self.create_namespace_request.meta.auth.STAFF
        self.create_namespace_request.meta.category = awacs_category
        order = self.create_namespace_request.order
        order.flow_type = order.YP_LITE
        order.yp_lite_allocation_request.nanny_service_id_slug = 'required-not-used'
        order.yp_lite_allocation_request.network_macro = '_SEARCHSAND_'
        order.yp_lite_allocation_request.locations.append('SAS')
        order.yp_lite_allocation_request.locations.append('MAN')
        order.yp_lite_allocation_request.locations.append('VLA')
        order.yp_lite_allocation_request.preset.type = order.yp_lite_allocation_request.preset.MICRO
        order.yp_lite_allocation_request.preset.instances_count = 1
        order.alerting_simple_settings.notify_staff_group_id = awacs_alerting_recipient
        logging.info("L7 instances: will use %s and MICRO awacs preset at SAS/MAN/VLA by default." % order.yp_lite_allocation_request.network_macro)
        logging.info("Change config at awacs.yaml if needed or later at awacs UI.")

        self.awacs_l7_macro = modules_pb2.L7Macro()
        self.awacs_l7_macro.version = "0.1.0"

        headers = {
            'X-Forwared-Host': 'host',
            #  'Host': '',
            #  'Upgrade': '',
            #  'Connection': '',
            #  'X-Balancer-Host': '',
            'X-Forwarded-For': 'realip',
            'X-Forwarded-Proto': 'scheme',
            'X-Real-IP': 'realip',
            #  'X-Original-URL': '',
            #  'Proxy': '',
            'X-Req-Id': 'reqid',
            'X-Request-Id': 'reqid'
        }

        for target, func in headers.items():
            header = self.awacs_l7_macro.headers.add()
            header.create.target = target
            header.create.func = func

        self.awacs_l7_macro.announce_check_reply.url_re = "/ping"
        self.awacs_l7_macro.health_check_reply.SetInParent()
        self.awacs_l7_macro.include_domains.SetInParent()

        self.awacs_domains = []
        self.awacs_upstreams = []
        self.awacs_backends = {}
        self.awacs_endpoint_sets = {}

        self.yav_secret_alias = yav_secret_alias

    def __map_qloud_to_yp_cluster(self, location, unit_id):
        location_map = {
            'IVA': 'iva',
            'MAN': 'man',
            'MOSCOW': 'iva',
            'MYT': 'myt',
            'SAS': 'sas',
            'VLA': 'vla',
        }
        if location == 'ALL':
            hashed = int(hashlib.md5(unit_id).hexdigest(), 16)
            variants = sorted(list(set(location_map.values())))
            i = hashed % len(variants)
            return variants[i]
        else:
            for prefix, loc in location_map.iteritems():
                if location.startswith(prefix):
                    return loc
            return None

    def __get_domain(self, domain_id):
        if domain_id >= len(self.awacs_domains):
            self.awacs_domains += [model_pb2.DomainOrder.Content()] * (domain_id - len(self.awacs_domains) + 1)
        return self.awacs_domains[domain_id]

    def __get_upstream(self, upstream_id):
        if upstream_id >= len(self.awacs_upstreams):
            for i in range(upstream_id - len(self.awacs_upstreams) + 1):
                self.awacs_upstreams.append(modules_pb2.L7UpstreamMacro())
        return self.awacs_upstreams[upstream_id]

    def __get_backend(self, backend_id):
        return self.awacs_backends.setdefault(backend_id, model_pb2.BackendSelector())

    def __get_endpoint_set(self, endpoint_set_id):
        return self.awacs_endpoint_sets.setdefault(endpoint_set_id, model_pb2.EndpointSetSpec())

    def __add_disk_volume_request(self, unit_id, unit_i, disk_id, disk_size):
        if disk_size < 0:
            raise self.NotImplementedException("disk_size == %s < 0 (inherited disk), can't create disk_volume_request" % disk_size, ['components', unit_i])
        if disk_size < migrator_base.BOUND_FOR_DISK_TAX_IN_GB:
            disk_size += migrator_base.DISK_TAX_IN_GB
            logging.warning('adding %sGB to "%s" disk capacity for pod_agent infrastructure', migrator_base.DISK_TAX_IN_GB, unit_id)

        disk = self.stage.spec.deploy_units[unit_id].multi_cluster_replica_set.replica_set.pod_template_spec.spec.disk_volume_requests.add()
        disk.id = disk_id
        disk.quota_policy.capacity = disk_size << 30
        disk.storage_class = 'hdd'
        size = self.spec["dump"]["components"][unit_i]["properties"].get("size")
        if not size:
            raise self.NotImplementedException("field is empty, can't create disk_volume_request", ['components', unit_i, 'properties', 'size'])
        memory, cpu, network, io = size.split(';')
        io = int(int(io) / 100.0 * (100 << 20))  # assuming hdd bandwidth is 100MB/s

        disk.quota_policy.bandwidth_guarantee = io
        disk.quota_policy.bandwidth_limit = io
        return disk

    def __get_instance(self, component_name):
        try:
            return self.spec["stable"]["components"][component_name]["runningInstances"][0]["host"]
        except Exception as e:
            raise Exception('cannot found running instance for %s component at env spec: %s' % (component_name, e))

    def __dump_file_from_component(self, component_name, filename):
        return self._run_remote_call_on_instance(self.__get_instance(component_name), 'cat %s' % filename)

    def __dump_env_from_component_qloud_init(self, component_name):
        env_output = self._run_remote_call_on_instance(self.__get_instance(component_name), 'cat /proc/$(pidof qloud-init -o 1)/environ')
        result = {}
        for it in env_output.split('\0'):
            if it:
                k, v = it.split('=', 1)
                result[k] = v
        return result

    #
    # DOMAIN
    #

    def _set_domain_certificate_ca(self, ca, path):
        domain = self.__get_domain(path[-2])
        domain.cert_order.content.ca_name = ca

    def _set_domain_id_and_host(self, domain_name, path):
        domain = self.__get_domain(path[-2])
        domain.fqdns.append(domain_name)
        domain.cert_order.content.abc_service_id = self.spec.get("abcId", 0)
        domain.cert_order.content.common_name = domain_name

    def _set_domain_http_only(self, http_only, path):
        domain = self.__get_domain(path[-2])
        if http_only:
            self.awacs_l7_macro.http.SetInParent()
            domain.protocol = model_pb2.DomainSpec.Config.Protocol.HTTP_ONLY

    def _set_domain_https_only(self, http_allowed, path):
        domain = self.__get_domain(path[-2])
        if http_allowed:
            self.awacs_l7_macro.http.SetInParent()
            if domain.protocol != model_pb2.DomainSpec.Config.Protocol.HTTP_ONLY:
                domain.protocol = model_pb2.DomainSpec.Config.Protocol.HTTP_AND_HTTPS
        else:
            self.awacs_l7_macro.https.SetInParent()
            domain.protocol = model_pb2.DomainSpec.Config.Protocol.HTTPS_ONLY

    #
    # UPSTREAM
    #

    def _set_upstream_endpoint_set(self, component_name, path):
        errors = []
        upstream = self.__get_upstream(path[-2])
        upstream.version = "0.0.1"
        upstream.by_dc_scheme.on_error.static.status = 504
        upstream.by_dc_scheme.dc_balancer.attempt_all_dcs = True
        upstream.by_dc_scheme.dc_balancer.weights_section_id = 'by_geo'
        upstream.by_dc_scheme.balancer.do_not_retry_http_responses = True
        upstream.by_dc_scheme.balancer.do_not_limit_reattempts = True
        upstream.by_dc_scheme.balancer.max_pessimized_endpoints_share = 0.2
        for i in range(len(self.spec["dump"]["components"])):
            component = self.spec["dump"]["components"][i]
            if component["componentName"] == component_name:
                if component["componentType"] == "standard":
                    es_id = self.stage.meta.id + '.' + component_name
                    for group in component["instanceGroups"]:
                        if not group['units']:
                            logging.info('Will not create awacs backend for instanceGroup %s with 0 units at %s', group['location'], path[:-1])
                            continue
                        cluster = self.__map_qloud_to_yp_cluster(group["location"], component_name)
                        backend_id = es_id + '.' + cluster
                        backend = self.__get_backend(es_id + '.' + cluster)
                        backend.type = backend.YP_ENDPOINT_SETS_SD
                        if not len(backend.yp_endpoint_sets):
                            es = backend.yp_endpoint_sets.add()
                            es.cluster = cluster
                            es.endpoint_set_id = es_id
                        dc = upstream.by_dc_scheme.dcs.add()
                        dc.name = cluster
                        dc.backend_ids.append(backend_id)
                elif component["componentType"] == "component-proxy":
                    weights = set()
                    for upstream_component in component['upstreamComponents']:
                        weights.add(upstream_component['weight'])
                        dc = upstream.by_dc_scheme.dcs.add()
                        dc.name = 'TODO_CHOOSE: sas/man/vla/iva/myt'
                        dc.backend_ids.append('TODO_ENTER_BACKEND_FOR_%s' % upstream_component['componentId'])
                    if len(weights) > 1:
                        logging.warning('To adjust weight, you need to have a namespace in the awacs,\n \
                                make there backends YP_ENDPOINT_SETS_SD, pointing to deploy endpoint_sets,\n \
                                in upstream to include them with different weights.\n \
                                https://wiki.yandex-team.ru/cplb/awacs/tutorial/L3-L7/\n \
                                https://wiki.yandex-team.ru/cplb/awacs/cookbook/#staticheskizadannyevesadljagruppinstansov')
                    logging.warning('Specify the endpoint_set_id/cluster for the proxy component at awacs.yaml.')
                elif component["componentType"] == "proxy":
                    backend_id = component_name
                    backend = self.__get_backend(backend_id)
                    backend.type = backend.MANUAL
                    dc = upstream.by_dc_scheme.dcs.add()
                    dc.name = 'TODO_CHOOSE: sas/man/vla/iva/myt'
                    dc.backend_ids.append(backend_id)
                    logging.warning('Specify the endpoint_set_id/cluster for the proxy component at awacs.yaml.')
                else:
                    errors.append(
                        self.NotImplementedException(
                            'Not supported components with type %s' % component["componentType"],
                            ["dump", "components", i, "componentType", component["componentType"]]
                        )
                    )
        if errors:
            raise self.NotImplementedExceptionList(errors)

    def _set_upstream_inner_balancing_option(self, upstream_option, convert_type_func, skip_default=["-1", "0"]):
        def f(option, path):
            if option in skip_default:
                return
            if not str(option).endswith('s'):
                option = convert_type_func(option)
            upstream = self.__get_upstream(path[-2])
            setattr(upstream.by_dc_scheme.balancer, upstream_option, option)
        return f

    def _set_upstream_outer_balancing_option_locality(self, geo, path):
        upstream = self.__get_upstream(path[-2])
        if geo:
            upstream.by_dc_scheme.dc_balancer.method = modules_pb2.L7UpstreamMacro.DcBalancerSettings.Method.LOCAL_THEN_BY_DC_WEIGHT
        else:
            upstream.by_dc_scheme.dc_balancer.method = modules_pb2.L7UpstreamMacro.DcBalancerSettings.Method.BY_DC_WEIGHT

    def _set_upstream_matcher_and_rewrite_src(self, location, path):
        upstream = self.__get_upstream(path[-2])
        upstream.id = location.strip('/').replace('/', '_')
        upstream.id = re.sub(r'(?<!^)(?=[A-Z])', '_', upstream.id).lower()  # CamelCaseName -> snake_case; awacs will restrict capital letters
        if not upstream.id:
            upstream.id = 'root'
        upstream.matcher.path_re = '(' + location + '.*)?'

    def _set_upstream_rewrite_dst(self, upstream_path, path):
        location = self.spec["dump"]["routeSettings"][path[-2]]["location"]
        if upstream_path != location:
            upstream = self.__get_upstream(path[-2])
            rewrite = upstream.rewrite.add()
            rewrite.target = rewrite.PATH
            rewrite.pattern.re = location
            rewrite.pattern.literal.value = True
            rewrite.replacement = upstream_path

    def _set_upstream_proxy_next_upstream(self, proxy_next_upstream, path):
        upstream = self.__get_upstream(path[-2])
        for it in proxy_next_upstream.split(' '):
            if it.startswith('http_'):
                code = it[5:]
                upstream.by_dc_scheme.balancer.retry_http_responses.codes.append(code)
            else:
                self._assert_equal_one_of_enum(["error", "timeout"])(it, path)

    def _set_endpoint_set_instances(self, servers, path):
        endpoint_set_id = self.spec["dump"]["components"][path[-3]]["componentName"]
        endpoint_set = self.__get_endpoint_set(endpoint_set_id)
        for host in servers.split(','):
            port = 80
            instance = endpoint_set.instances.add()
            for prefix in ['http://', 'https://']:
                if host.startswith(prefix):
                    host = host[len(prefix):]
            if not host.endswith(']') and ':' in host:
                host, port = host.rsplit(':', 1)
            instance.host = host
            instance.port = int(port)
            instance.weight = 1
            instance.ipv6_addr = host[1:-1] if host.startswith('[') else self._resolve_ip6(host)

    def _set_header_host(self, host, path):
        if not host:
            return
        for i in range(len(self.spec["dump"]["routeSettings"])):
            route = self.spec["dump"]["routeSettings"][i]
            if route["componentName"] == self.spec["dump"]["components"][path[-3]]["componentName"]:
                logging.info('map %s - %s - %s', i, route["componentName"], host)
                upstream = self.__get_upstream(i)
                header = upstream.headers.add()
                header.create.target = 'Host'
                header.create.value = host

    #
    # STAGE
    #

    def _set_abc_id(self, abc_id, path):
        self.stage.meta.account_id = 'abc:service:' + str(abc_id)
        self.create_namespace_request.meta.abc_service_id = abc_id

    def _set_allocation_strategy(self, allocationStrategy, path):
        if allocationStrategy == 'collocation':
            logging.warning('for collocation you have to manually compose your boxes to one replica_set at deploy.yaml')
        elif allocationStrategy != 'dynamic':
            logging.warning('allocation strategy "%s" is not supported at YD - only "dynamic"', allocationStrategy)

    def _set_prepare_recipe(self, recipe, path):
        default_recipe = {
            "recipe": "INTERNAL",
            "updateWindow": "100%",
            "doneThreshold": "90%",
            "updateLimit": "100%",
            "updatePeriod": "20s"
        }
        if recipe == default_recipe:
            return  # ignore default qloud recipe
        logging.warning('prepare recipe is not yet supported at YD - resources are being downloaded at deploy phase')

    def _set_feature(self, feature, path):
        if feature == 'OS_CONTAINER':
            logging.warning('OS_CONTAINER is not supported at YD, please update workload command manually at deploy.yaml')
        elif feature == 'UNBOUND_CPU':
            logging.warning('UNBOUND_CPU is not supported at YD, please update your resources manually at deploy.yaml')
        elif feature in ['SSH_USER_MANAGEMENT', 'COREDUMPS_ENABLED']:
            logging.warning('%s feature is ignored' % feature)
        else:
            self._assert_equal_one_of_enum([
                'CAP_NET_ADMIN',  # enabled by default
                'CAP_NET_RAW',  # enabled by default
                'ISS_RESOURCE_BIND_MOUNT',  # caused by resource isolation trouble at qloud - ignore it
                'PORTO_SOCKET',  # enabled by default
            ])(feature, path)

    def _set_unistat(self, unistat, path):
        if unistat == 'false':
            return
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]
        host_infra = self._get_pod_template_spec_spec(unit_id).host_infra

        unistatConf = host_infra.monitoring.unistats.add()
        unistatConf.port = int(self.spec["dump"]["components"][path[-3]]["properties"]["unistatPort"])
        unistatConf.path = self.spec["dump"]["components"][path[-3]]["properties"]["unistatPath"]
        unistatConf.workload_id = unit_id
        unistatConf.output_format = host_infra_pb2.TMonitoringUnistatEndpoint.EOutputFormat.OF_YASM_JSON

    def _set_secrets_and_tvm(self, components, path):
        # dump secrets from components
        dumped_secrets = {}
        tvm_name_to_id = {}
        for i in range(len(components)):
            if components[i]["componentType"] != "standard":
                continue  # migrate secrets only for "standard" components

            used_env_secret_by_target = {}
            used_file_secrets = []
            for j in range(len(components[i]["secrets"])):
                sec = components[i]["secrets"][j]
                if sec['used']:
                    if sec['target'].startswith('/'):
                        used_file_secrets.append(sec)
                    else:
                        used_env_secret_by_target[sec['target']] = sec
            if not self.yav_secret_alias:
                continue

            # file secrets
            for sec in used_file_secrets:
                k = sec['objectId']
                v = self.__dump_file_from_component(components[i]["componentName"], sec['target'])
                if dumped_secrets.get(k, v) != v:
                    raise self.NotImplementedException(
                        'secrets value %s is different at different components, cannot migrate secret' % k,
                        path + [i, "secrets"]
                    )
                dumped_secrets[k] = v

            # dump qloud_init env
            if len(used_env_secret_by_target.keys()) > 0:
                try:
                    qloud_init_env = self.__dump_env_from_component_qloud_init(components[i]["componentName"])
                except Exception as e:
                    raise self.NotImplementedException(
                        'cannot dump env from component: %s' % e,
                        path + [i, "secrets"]
                    )
            else:
                qloud_init_env = {}

            # env secrets
            diff = set(used_env_secret_by_target.keys()) - set(qloud_init_env.keys())
            if diff:
                raise self.NotImplementedException('some secrets werent found at instance: %s' % diff, path + [i, "secrets"])

            for target, sec in used_env_secret_by_target.iteritems():
                k = sec['objectId']
                v = qloud_init_env[target]
                if dumped_secrets.get(k, v) != v:
                    raise self.NotImplementedException(
                        'secrets value %s is different at different components, cannot migrate secret' % k,
                        path + [i, "secrets"]
                    )
                dumped_secrets[k] = v

            # tvm
            tvm_config_env = 'QLOUD_TVM_CONFIG'
            tvm_secrets = {}
            if tvm_config_env in qloud_init_env:
                data = json.loads(qloud_init_env[tvm_config_env])
                for name, client in data['clients'].iteritems():
                    if tvm_name_to_id.get(name, client['self_tvm_id']) != client['self_tvm_id']:
                        raise self.NotImplementedException(
                            'tvm client name %s points to different tvm_ids at different components, cannot migrate tvm secret' % name,
                            path + [i, "tvmConfig", "clients"]
                        )
                    tvm_name_to_id[name] = client['self_tvm_id']
                    k = 'TVM_' + str(client['self_tvm_id'])
                    v = client['secret']
                    tvm_secrets[k] = v

            for j in range(len(components[i].get("tvmConfig", {}).get("clients", []))):
                client = components[i]["tvmConfig"]["clients"][j]
                if client["name"] not in tvm_name_to_id:
                    raise self.NotImplementedException('tvm secret for %s werent found at instance' % client["name"], path + [i, "tvmConfig", "clients", j])

            for k, v in tvm_secrets.iteritems():
                if dumped_secrets.get(k, v) != v:
                    raise self.NotImplementedException(
                        'tvm secrets value %s is different at different components, cannot migrate secret' % k,
                        path + [i, "tvmConfig"]
                    )
                dumped_secrets[k] = v

        if not dumped_secrets:
            return

        # find or create secret at yav
        if not self.yav_secret_alias:
            logging.warning('please, specify --yav-secret-alias for secret/tvm migration')
            return

        client = VaultClient()
        secret_uuid = self._create_yav_secret(client, self.yav_secret_alias, path)

        # upload secrets to yav
        secret_version = client.create_secret_version(
            secret_uuid,
            dumped_secrets,
        )
        logging.warning('Secret ACL (for users/abc services) werent migrated - secret is only available for you and stage instances.')
        logging.warning('Please, update yav ACL manually at https://yav.yandex-team.ru/secret/%s', secret_uuid)

        # set secrets/tvm to spec
        for i in range(len(components)):
            if components[i]["componentType"] != "standard":
                continue  # migrate secrets only for "standard" components
            unit_id = components[i]["componentName"]

            self._create_vault_token(
                client=client,
                secret_id=secret_uuid,
                secret_version=secret_version,
                object_id=self.spec["dump"]["objectId"],
                unit_id=unit_id,  # pod_set_id
                deploy_secret_id='QLOUD_DUMP'
            )

            file_secrets_by_dir = {}
            workload = self._get_workload(unit_id)
            for sec in components[i]["secrets"]:
                if sec['used']:
                    if sec['target'].startswith('/'):
                        file_secrets_by_dir.setdefault('/'.join(sec['target'].split('/')[:-1]), []).append(sec)
                    else:
                        it = workload.env.add()
                        it.name = sec['target']
                        it.value.secret_env.alias = 'QLOUD_DUMP'
                        it.value.secret_env.id = sec['objectId']
            pod_agent_payload_spec = self._get_pod_agent_payload_spec(unit_id)
            for directory, secs in file_secrets_by_dir.iteritems():
                resource = pod_agent_payload_spec.resources.static_resources.add()
                resource.id = directory.replace('/', '.')
                resource.verification.checksum = 'EMPTY:'
                for sec in secs:
                    file = resource.files.files.add()
                    file.file_name = sec['target'].split('/')[-1]
                    file.secret_data.alias = 'QLOUD_DUMP'
                    file.secret_data.id = sec['objectId']
                box = self._get_box(unit_id)
                mounted = box.static_resources.add()
                mounted.resource_ref = resource.id
                mounted.mount_point = directory

            self._set_tvm_config(components[i].get("tvmConfig", {}), path + [i, "tvmConfig"], tvm_name_to_id)

    def _set_tvm_config(self, tvm_config, path, tvm_name_to_id):
        if not tvm_config:
            return
        logging.warning('TVM client port is 2 at Y.Deploy (not 1 as in Qloud). Please, update your application to use $DEPLOY_TVM_TOOL_URL env.')
        logging.warning('TVM client token is at $TVMTOOL_LOCAL_AUTHTOKEN at Y.Deploy (not at $QLOUD_TVM_TOKEN as in Qloud). Please, update your application.')
        logging.warning('There is no analog for $QLOUD_TVM_CONFIG at Y.Deploy. Please, update your application. (You can aliases like "self" and use it like "/tvm/tickets?src=self&dsts=100500".)')
        unit_id = self.spec["dump"]["components"][path[-2]]["componentName"]
        cfg = self.stage.spec.deploy_units[unit_id].tvm_config
        cfg.mode = stage_pb2.TTvmConfig.EMode.Value('ENABLED')
        cfg.blackbox_environment = {
            "PROD": "Prod",
            "TEST": "Test",
            "PROD_YA_TEAM": "ProdYaTeam",
            "TEST_YA_TEAM": "TestYaTeam",
            "STRESS": "Stress",
        }[tvm_config["blackBoxType"]]
        for cl in tvm_config["clients"]:
            client = cfg.clients.add()
            client.source.app_id = tvm_name_to_id[cl["name"]]
            client.source.alias = cl["name"]
            client.secret_selector.alias = 'QLOUD_DUMP'
            client.secret_selector.id = 'TVM_' + str(tvm_name_to_id[cl["name"]])
            for dest in cl["destinations"]:
                destination = client.destinations.add()
                destination.app_id = dest["tvmId"]
                destination.alias = dest["name"]

    def _set_workload_transmit_logs(self, stdstream, path):
        if stdstream == 'ignore':
            return
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]
        workload = self._get_workload(unit_id)
        if not workload.transmit_logs:
            logging.warning('logs trasmission for "%s" costs extra cpu/mem/disk resources: https://wiki.yandex-team.ru/deploy/docs/sidecars/logs/#resources', unit_id)
        workload.transmit_logs = True

    def _set_endpoint_set_port(self, upstream_port, path):
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]
        es = self.stage.spec.deploy_units[unit_id].endpoint_sets.add()
        es.port = int(upstream_port)

    def _set_environment(self, environment_variables, path):
        if self.spec["dump"]["components"][path[-2]]["componentType"] != 'standard':
            return
        unit_id = self.spec["dump"]["components"][path[-2]]["componentName"]
        workload = self._get_workload(unit_id)

        env = self.spec["dump"]["userEnvironmentMap"].copy()  # global env for every component
        env.update({'QLOUD_HTTP_PORT': '80'})  # https://wiki.yandex-team.ru/qloud/doc/environment/envvar/#port
        env.update(environment_variables)

        for k, v in env.iteritems():
            var = workload.env.add()
            var.name = k
            var.value.literal_env.value = v

    def _set_ephemeral_volume(self, edisk_size, path):
        if not int(edisk_size):
            return
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]
        logging.warn("multiple disk_volume_requests for /ephemeral are not supported at YD yet, will sum up all disk qouta at one volume")

        volume = self._get_pod_agent_payload_spec(unit_id).volumes.add()
        volume.id = 'ephemeral'
        box = self._get_box(unit_id)
        mounted_volume = box.volumes.add()
        mounted_volume.volume_ref = volume.id
        mounted_volume.mount_point = '/ephemeral'
        mounted_volume.mode = pod_agent_pb2.EVolumeMountMode_READ_WRITE

    def _set_dns_nat64(self, dns_nat64, path):
        if dns_nat64 != 'true':
            return
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]
        box = self._get_box(unit_id)
        box.resolv_conf = pod_agent_pb2.EResolvConf_NAT64

    def _set_tmpfs_init_script(self, tmpfs_disk_size, path):
        if not int(tmpfs_disk_size):
            return
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]
        box = self._get_box(unit_id)
        init = box.init.add()
        init.command_line = '''bash -c \'\nif [ ! -d /tmpfs ]; then\n    mkdir /tmpfs\n    portoctl vcreate /tmpfs backend=tmpfs space_limit=%sG\nfi\n\'''' % tmpfs_disk_size

    def _set_static_resources(self, sandbox_resources, path):
        seen_dirs = set()
        errors = []
        for i in range(len(sandbox_resources)):
            try:
                unit_id = self.spec["dump"]["components"][path[-2]]["componentName"]
                pod_agent_payload_spec = self._get_pod_agent_payload_spec(unit_id)
                box = self._get_box(unit_id)

                res = sandbox_resources[i]
                if not res['symlink']:
                    raise self.NotImplementedException('Resource with empty symlink is not supported', path + [i, 'symlink'])
                self._assert_equal(False)(res['dynamic'], path + [i, 'dynamic'])
                self._warn_if_not_equal(
                    False,
                    'Resource extraction is not supported at YD, please, add extraction command to boxes/%s/init for resource %s at deploy.yaml' % (unit_id, res)
                )(res['extract'], path + [i, 'extract'])

                resource = pod_agent_payload_spec.resources.static_resources.add()
                resource.id = '.'.join(res['symlink'].split('/')[1:])

                directory = '/' + '/'.join(res['symlink'].split('/')[1:-1])  # symlink is always absolute
                if directory == '/':
                    directory = '/' + resource.id
                    logging.warning('Resource mounting to / is not supported, will put resource %s to directory %s' % (resource.id, directory))

                sandbox_resource = self._get_sandbox_resource(res['id'])

                resource.meta.sandbox_resource.resource_id = str(res['id'])
                if sandbox_resource.get('type'):
                    resource.meta.sandbox_resource.resource_type = str(sandbox_resource['type'])
                if sandbox_resource.get('task', {}).get('id'):
                    resource.meta.sandbox_resource.task_id = str(sandbox_resource['task']['id'])

                if sandbox_resource.get('skynet_id'):
                    resource.url = sandbox_resource['skynet_id']
                    resource.verification.checksum = 'EMPTY:'
                else:
                    resource.url = sandbox_resource['http']['proxy']
                    resource.verification.checksum = ('MD5:' + sandbox_resource['md5']) if sandbox_resource.get('md5') else 'EMPTY:'

                mounted = box.static_resources.add()
                mounted.resource_ref = resource.id
                mounted.mount_point = directory
                name = res['symlink'].split('/')[-1]
                if res['localName'] != name:
                    logging.warning("Resource renaming (%s -> %s) is not supported."
                                    " You can reconfigure your application or add 'mv' operation to boxes/%s/init at deploy.yaml" % (res['localName'], name, unit_id))
                if directory in seen_dirs:
                    logging.warning(
                        'Two static resources cannot be in one directory (mount_point) "%s"' % directory +
                        ', please set resources to different mount_points manually at boxes/%s/static_resources at deploy.yaml' % unit_id
                    )
                seen_dirs.add(directory)
            except self.NotImplementedException as e:
                errors.append(e)
        if errors:
            raise self.NotImplementedExceptionList(errors)

    def _set_readiness_probe(self, status_hook_checks, path):
        sel_i = None
        for i in range(len(status_hook_checks)):
            self._assert_equal_one_of_enum(['http', 'tcp'])(status_hook_checks[i]['type'], path + [i, 'type'])
            if status_hook_checks[i]['port'] == 1:  # skip qloud-init check
                continue
            if sel_i is None:
                sel_i = i
            else:
                logging.warning('YD supports only single readiness_check for workload, got %s and %s, will use first one' % (status_hook_checks[sel_i], status_hook_checks[i]))

        unit_id = self.spec["dump"]["components"][path[-2]]["componentName"]

        properties = self.spec["dump"]["components"][path[-2]]["properties"]
        if properties["useHealthCheck"] == "true":
            if sel_i is None:
                if self.spec["dump"]["components"][path[-2]]["componentType"] != "standard":
                    logging.warning("Will ignore (qloud) health check, component %s should use readiness_check for active balancing"
                                    " https://wiki.yandex-team.ru/deploy/docs/podstack/workload/probes/#readiness", unit_id)
                    return
                workload = self._get_workload(unit_id)
                logging.warning("Will use (qloud) health check as a (YD) readiness check at %s"
                                " https://wiki.yandex-team.ru/deploy/docs/podstack/workload/probes/#readiness", path)
                health_path = ["dump", "components", path[-2], "properties"]
                self._assert_equal_one_of_enum(['http'])(
                    properties["healthCheckType"],
                    health_path + ["healthCheckType"]
                )
                self._warn_if_not_equal_one_of(["http_2xx"], "YD supports only http_2xx as successful readiness check")(
                    properties["healthCheckHttpExpectedCode"],
                    health_path + ["healthCheckHttpExpectedCode"]
                )
                workload.readiness_check.http_get.port = int(properties["upstreamPort"])
                workload.readiness_check.http_get.path = properties["healthCheckHttpUrl"]
                workload.readiness_check.http_get.any = True
                workload.readiness_check.http_get.time_limit.min_restart_period_ms = int(properties["healthCheckInterval"])
                workload.readiness_check.http_get.time_limit.max_execution_time_ms = int(properties["healthCheckTimeout"])
                if properties["healthCheckFall"] != "1":
                    workload.readiness_check.failure_threshold = int(properties["healthCheckFall"])
                if properties["healthCheckRise"] != "1":
                    workload.readiness_check.success_threshold = int(properties["healthCheckRise"])
            else:
                logging.warning("Will ignore (qloud) health check, will use readiness_check %s for active balancing"
                                " https://wiki.yandex-team.ru/deploy/docs/podstack/workload/probes/#readiness", status_hook_checks[sel_i])

        if sel_i is None:
            return

        workload = self._get_workload(unit_id)
        if status_hook_checks[sel_i]['type'] == 'http':
            workload.readiness_check.http_get.port = status_hook_checks[sel_i]['port']
            workload.readiness_check.http_get.path = status_hook_checks[sel_i]['path']
            workload.readiness_check.http_get.any = True
            workload.readiness_check.http_get.time_limit.max_execution_time_ms = status_hook_checks[sel_i]['timeout']
        else:
            self._assert_equal("")(status_hook_checks[sel_i].get('path'), path + [sel_i, 'path'])
            workload.readiness_check.tcp_check.port = status_hook_checks[sel_i]['port']
            workload.readiness_check.tcp_check.time_limit.max_execution_time_ms = status_hook_checks[sel_i]['timeout']

    def _set_stop_policy(self, stop_action, path):
        self._assert_equal_one_of_enum(['signal', 'http'])(stop_action['type'], path + ['type'])
        unit_id = self.spec["dump"]["components"][path[-2]]["componentName"]
        workload = self._get_workload(unit_id)
        if stop_action['type'] == 'signal':
            logging.warning('Signal stop policy is not supported at YD, will use exec/pkill instead. Please, insert your regular expression for pkill at deploy.yaml.')
            workload.stop_policy.container.command_line = "bash -c 'pkill --signal %s -e <INSERT_REGEX_HERE>; exit 1;'" % stop_action['signal']
            workload.stop_policy.container.time_limit.min_restart_period_ms = stop_action['timeoutMs']
            workload.stop_policy.max_tries = 1
        elif stop_action['type'] == 'http':
            self._warn_if_not_equal(
                'GET',
                '%s http method is not supported at YD, will use GET instead' % stop_action['httpMethod']
            )(stop_action['httpMethod'], path + ['httpMethod'])
            workload.stop_policy.http_get.port = stop_action['httpPort']
            workload.stop_policy.http_get.path = stop_action['httpPath']
            workload.stop_policy.http_get.any = True
            workload.stop_policy.http_get.time_limit.max_execution_time_ms = stop_action['timeoutMs']
            workload.stop_policy.max_tries = 1

    def _set_resource_requests(self, size, path):
        memory, cpu, network, io = size.split(';')
        # network skipped
        # io implemented at _set_disk_volume_requests
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]
        resource_requests = self.stage.spec.deploy_units[unit_id].multi_cluster_replica_set.replica_set.pod_template_spec.spec.resource_requests
        resource_requests.vcpu_guarantee = int(float(cpu) * 1000)
        resource_requests.vcpu_limit = int(float(cpu) * 1000)
        resource_requests.memory_guarantee = int(memory) << 30
        resource_requests.memory_limit = int(memory) << 30

    def _set_images_for_boxes(self, repository, path):
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]

        image = self.stage.spec.deploy_units[unit_id].images_for_boxes[unit_id]
        image.registry_host = repository.split('/', 1)[0]
        image.name = repository.split('/', 1)[1].split(':')[0]
        image.tag = repository.split('/', 1)[1].split(':')[1]

        self._get_box(unit_id)  # add box

    def _set_disk_volume_requests(self, disk_size, path):
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]
        edisk_size = self.spec["dump"]["components"][path[-3]]["properties"]["ediskSize"]
        disk = self.__add_disk_volume_request(unit_id, path[-3], 'infra', int(disk_size) + int(edisk_size))
        label = disk.labels.attributes.add()
        label.key = 'used_by_infra'
        label.value = yson.dumps(True)

    def _set_network(self, network, path):
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]

        network_defaults = self.stage.spec.deploy_units[unit_id].network_defaults
        network_defaults.network_id = '_' + network + '_'

    def _set_juggler_subagent(self, juggler_bundle_resources, path):
        unit_id = self.spec["dump"]["components"][path[-2]]["componentName"]
        for it in juggler_bundle_resources:
            sandbox_resource = self._get_sandbox_resource(it['id'])
            box_juggler_configs = self.stage.spec.deploy_units[unit_id].box_juggler_configs[unit_id]
            check = box_juggler_configs.archived_checks.add()
            check.url = sandbox_resource['http']['proxy']
            check.checksum.type = downloadable_resources_pb2.MD5 if 'md5' in sandbox_resource else downloadable_resources_pb2.EMPTY
            check.checksum.value = sandbox_resource['md5'] if 'md5' in sandbox_resource else ''
            logging.warning('juggler subagent costs extra cpu/mem/disk resources. Please, read https://wiki.yandex-team.ru/deploy/docs/sidecars/jugglersubagent/#resources')

    def _set_replica_set_clusters(self, instance_groups, path):
        errors = []
        unit_id = self.spec["dump"]["components"][path[-2]]["componentName"]
        for i in range(len(instance_groups)):
            try:
                it = instance_groups[i]
                if not it['units']:
                    logging.info('Skipping instanceGroup %s with 0 units at %s', it['location'], path + [i])
                    continue
                cluster = self.stage.spec.deploy_units[unit_id].multi_cluster_replica_set.replica_set.clusters.add()
                cluster.cluster = self.__map_qloud_to_yp_cluster(it['location'], unit_id)
                if not cluster.cluster:
                    raise self.NotImplementedException("unknown location: %s" % it['location'], path + [i, 'location'])
                cluster.spec.replica_count = it['units']
                constraints = cluster.spec.constraints.antiaffinity_constraints.add()
                max_instances_per_host = int(self.spec["dump"]["components"][path[-2]]["properties"]["maxInstancesPerHost"])
                if max_instances_per_host != 999:
                    constraints.key = 'node'
                    constraints.max_pods = max_instances_per_host
                else:  # map default 999 to 1 per rack
                    constraints.key = 'rack'
                    constraints.max_pods = 1
                self._assert_equal(False)(it['backup'], path + [i, 'backup'])
            except self.NotImplementedException as e:
                errors.append(e)
        if errors:
            raise self.NotImplementedExceptionList(errors)

    def _set_replica_set_deployment_strategy(self, update_window, path):
        unit_id = self.spec["dump"]["components"][path[-3]]["componentName"]
        groups = self.spec["dump"]["components"][path[-3]]["instanceGroups"]

        total_pods = 0
        for it in groups:
            total_pods += it['units']

        max_unavailable = 0
        if update_window.endswith('%'):
            degrade_level = int(update_window[:-1]) / 100.0
            max_unavailable = max(1, int(total_pods * degrade_level))
        else:
            max_unavailable = int(update_window)

        self.stage.spec.deploy_units[unit_id].multi_cluster_replica_set.replica_set.deployment_strategy.max_unavailable = max_unavailable

    def _set_proxy_buffering(self, buffering, path):
        if buffering:
            upstream = self.__get_upstream(path[-2])
            upstream.by_dc_scheme.balancer.buffering = True

    def migrate(self):
        self._migrate_spec()

        for k, v in self.awacs_backends.iteritems():
            self.create_namespace_request.order.backends[k].CopyFrom(v)

        for domain in self.awacs_domains:
            domain.include_upstreams.type = modules_pb2.IncludeUpstreamsType.BY_ID
            for upstream in self.awacs_upstreams:
                domain.include_upstreams.ids.append(upstream.id)

        return self.stage, self.create_namespace_request, self.awacs_l7_macro, self.awacs_domains, self.awacs_upstreams, self.awacs_backends, self.awacs_endpoint_sets
