import logging

from infra.deploy_ci.approve_location.proto import approve_location_tasklet
from infra.deploy_ci.util import CommonInputMixin, retried_yp_call

from ci.tasklet.common.proto import service_pb2 as ci

import yt_yson_bindings
from yp import data_model
from yp.common import YpActionAlreadyPerformed
from yp_proto.yp.client.api.proto import object_service_pb2


log = logging.getLogger(__name__)


class ApproveLocationImpl(approve_location_tasklet.ApproveLocationBase, CommonInputMixin):
    def save_progress(
        self,
        message: str,
        yp_cluster: str = '',
        stage_id: str = '',
        deploy_unit_id: str = '',
        cluster: str = '',
        status=ci.TaskletProgress.Status.RUNNING,
    ):
        url = self.stage_status_link(
            yp_cluster=yp_cluster,
            stage_id=stage_id,
            deploy_unit_id=deploy_unit_id,
            cluster=cluster,
        )
        module = 'YANDEX_DEPLOY' if url else ''

        progress = ci.TaskletProgress()
        progress.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress.id = 'approve_location'
        progress.progress = 1.
        progress.text = message
        progress.module = module
        progress.url = url
        progress.status = status
        self.ctx.ci.UpdateProgress(progress)

    def find_location(self):
        matcher = self.input.location_match
        locations = list(filter(
            lambda loc: (
                (not matcher.yp_cluster or loc.yp_cluster == matcher.yp_cluster)
                and (not matcher.stage_id or loc.stage_id == matcher.stage_id)
                and (not matcher.deploy_unit or loc.deploy_unit == matcher.deploy_unit)
                and loc.cluster == matcher.cluster
            ),
            self.input.approval_locations,
        ))
        if len(locations) > 1:
            raise Exception(f"Multiple locations match filter: {matcher}")
        elif not locations:
            raise Exception(f"No matching approval location found for filter: {matcher}")

        return locations[0]

    def run(self):
        loc = None
        try:
            token = self.get_token()

            loc = self.find_location()
            log.info(
                "[yp %s] [%s] approving: deploy_unit=%r, cluster=%r revision=%r",
                loc.yp_cluster,
                loc.stage_id,
                loc.deploy_unit,
                loc.cluster,
                loc.revision,
            )

            self.init_yp_client(token, yp_address=loc.yp_cluster)

            action = data_model.TStageControl.TApproveAction()
            action.options.deploy_unit = loc.deploy_unit
            action.options.cluster = loc.cluster
            action.options.revision = loc.revision

            req = object_service_pb2.TReqUpdateObject()
            req.object_type = data_model.OT_STAGE
            req.object_id = loc.stage_id
            update = req.set_updates.add()
            update.path = '/control/approve'
            update.value_payload.yson = yt_yson_bindings.dumps_proto(action)

            try:
                retried_yp_call(
                    self.yp_stub.UpdateObject,
                    req,
                    ignore_exceptions=(YpActionAlreadyPerformed,),
                )
            except YpActionAlreadyPerformed as e:
                log.info("Approve failed with %r", str(e))
                self.output.state.success = True
                self.output.state.message = "Cluster is already approved"
            else:
                log.info("Approve succeeded")
                self.output.state.success = True
                self.output.state.message = "Success"

            self.save_progress(
                self.output.state.message,
                yp_cluster=loc.yp_cluster,
                deploy_unit_id=loc.deploy_unit,
                cluster=loc.cluster,
                status=ci.TaskletProgress.Status.SUCCESSFUL,
            )
        except Exception as e:
            self.output.state.success = False
            self.output.state.message = f"Approval failed: {e}"
            self.save_progress(
                str(e),
                yp_cluster=loc and loc.yp_cluster or '',
                stage_id=loc and loc.stage_id or '',
                deploy_unit_id=loc and loc.deploy_unit or '',
                cluster=loc and loc.cluster or '',
                status=ci.TaskletProgress.Status.FAILED,
            )
            raise
