import time
import hashlib
import logging
from urllib.parse import urljoin

import requests

from infra.deploy_ci.create_release.proto import create_release_tasklet
from infra.deploy_ci.util import CommonInputMixin, retried_yp_call

from ci.tasklet.common.proto import service_pb2 as ci, sandbox_pb2

from sandbox.common.rest import Client as SandboxClient
from sandbox.common import auth

import yt_yson_bindings
from yp import data_model
from yp_proto.yp.client.api.proto import object_service_pb2
from yp.common import YpNoSuchObjectError


log = logging.getLogger(__name__)


class CreateReleaseImpl(create_release_tasklet.CreateYaDeployReleaseBase, CommonInputMixin):
    steps: list
    release_kind: str
    affected_deploy_units: set[str]
    affected_dynamic_resources: set[str]
    yp_timestamp: int
    yp_tx_id: str
    sandbox_client: SandboxClient
    sandbox_resources: list[tuple[sandbox_pb2.SandboxResource, dict]]
    sandbox_task_info: dict
    commit_needed: bool
    ticket_approvals_needed: bool

    def save_progress(
        self,
        progress_ratio: float,
        message: str,
        step_id: str,
        status=ci.TaskletProgress.Status.RUNNING,
        deploy_ticket_id: str = '',
        release_id: str = '',
    ):
        url = self.stage_status_link(
            yp_cluster=self.input.config.yp_cluster or 'xdc',
            stage_id=self.input.config.stage_id,
            deploy_ticket_id=deploy_ticket_id,
            release_id=release_id,
        )
        module = 'YANDEX_DEPLOY' if url else ''

        progress = ci.TaskletProgress()
        progress.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress.id = step_id
        progress.progress = progress_ratio
        progress.text = message
        progress.module = module
        progress.url = url
        progress.status = status
        self.ctx.ci.UpdateProgress(progress)

    def collect_sandbox_infos(self):
        if self.sandbox_resources is None or self.sandbox_task_info is None:
            self.sandbox_resources, self.sandbox_task_info = collect_sandbox_infos(
                self.sandbox_client,
                self.input.sandbox_resources,
            )

    def create_transaction(self):
        self.yp_timestamp, self.yp_tx_id = create_transaction(self.yp_stub)

    def create_release(self):
        self.release_kind, self.output.release.object_id = create_release(
            self.yp_stub,
            self.sandbox_client,
            self.sandbox_resources,
            self.sandbox_task_info,
            self.input,
            self.yp_tx_id,
            self.yp_timestamp,
            self.yp_address,
        )

    def create_ticket(self):
        (self.affected_deploy_units,
         self.affected_dynamic_resources,
         self.output.deploy_ticket.object_id,
         self.commit_needed) = create_ticket(
             self.yp_stub,
             self.input,
             self.output.release.object_id,
             self.yp_tx_id,
             self.yp_timestamp,
             self.yp_address,
        )

    def check_approval_policy(self):
        self.ticket_approvals_needed = self.commit_needed and approval_required(
            self.yp_stub,
            self.yp_timestamp,
            self.input.config.stage_id,
        )

    def commit_ticket(self):
        if self.ticket_approvals_needed:
            log.info("stage has approval policy and ticket needs to be manually committed and approved")
            return

        if not self.commit_needed:
            log.info("ticket was committed in some previous attempt")
            return

        commit_ticket(self.yp_stub, self.input, self.output.deploy_ticket.object_id, self.yp_tx_id, self.yp_address)

        self.output.deploy_ticket.committed = True

    def commit_transaction(self):
        self.output.deploy_ticket.commit_timestamp = commit_transaction(self.yp_stub, self.yp_tx_id)

        self.save_progress(
            progress_ratio=1.,
            message='Release link',
            step_id='release',
            status=ci.TaskletProgress.Status.SUCCESSFUL,
            release_id=self.output.release.object_id,
        )
        self.save_progress(
            progress_ratio=1.,
            message='Ticket link',
            step_id='ticket',
            status=ci.TaskletProgress.Status.SUCCESSFUL,
            deploy_ticket_id=self.output.deploy_ticket.object_id,
        )

    def wait_for_ticket_approve_and_commit(self):
        if not self.ticket_approvals_needed:
            log.info("Stage has no approval policy, no wait for approvals needed")
            return

        timeout_minutes = self.input.config.timeout or 24 * 60
        timeout = timeout_minutes * 60
        approved = wait_for_approve(self.yp_stub, self.output.deploy_ticket.object_id, timeout)
        if not approved:
            raise Exception(f"Wait for approve failed (not finished in {timeout_minutes} minutes)")

        self.output.deploy_ticket.commit_timestamp = commit_ticket(self.yp_stub, self.input, self.output.deploy_ticket.object_id, "", self.yp_address)
        self.output.deploy_ticket.committed = True

    def get_locations(self):
        deploy_units, dynamic_resources, need_approvals = get_locations(
            self.yp_stub,
            self.input,
            self.affected_deploy_units,
            self.affected_dynamic_resources,
            self.output.deploy_ticket.commit_timestamp,
        )

        self.output.spec_revisions.yp_cluster = self.input.config.yp_cluster
        self.output.spec_revisions.stage_id = self.input.config.stage_id

        self.output.spec_revisions.deploy_units.update(deploy_units)
        self.output.spec_revisions.dynamic_resources.update(dynamic_resources)

        for du_id, yp_cluster, du_revision in need_approvals:
            al = self.output.approval_locations.add()
            al.yp_cluster = self.input.config.yp_cluster
            al.stage_id = self.input.config.stage_id
            al.deploy_unit = du_id
            al.cluster = yp_cluster
            al.revision = du_revision

    def wait_for_deploy(self):
        if len(self.output.approval_locations):
            log.info("Do not waiting for deploy to finish: some locations wait to be approved")
            return

        timeout_minutes = self.input.config.timeout or 24 * 60
        timeout = timeout_minutes * 60
        deployed = wait_for_deploy(self.yp_stub, self.output.deploy_ticket.object_id, timeout)
        if not deployed:
            raise Exception(f"Wait for deploy failed (not finished in {timeout_minutes} minutes)")

    def prepare_clients(self):
        token = self.get_token()

        self.init_yp_client(token)
        self.sandbox_client = SandboxClient(auth=auth.OAuth(token))

    def prepare_steps(self):
        self.steps = [
            (self.collect_sandbox_infos, "Collecting sandbox resource infos", True),
            (self.create_transaction, "Starting transaction", True),
            (self.create_release, "Creating release", True),
            (self.create_ticket, "Creating deploy ticket", True),
            (self.check_approval_policy, "Check if ticket approvals needed", True),
            (self.commit_ticket, "Committing deploy ticket", True),
            (self.commit_transaction, "Committing transaction", True),
            (self.wait_for_ticket_approve_and_commit, "Waiting for ticket to be approved and committing", False),
            (self.get_locations, "Getting affected locations", False),
            (self.wait_for_deploy, "Waiting for deploy to finish", False),
        ]

    def attempt(self):
        for idx, (step, message, can_restart_from_scratch) in enumerate(self.steps):
            log.info("[%2d] %s", idx, message)
            self.save_progress(progress_ratio=idx / len(self.steps), message=message, step_id='progress')
            try:
                step()
            except Exception as e:
                self.save_progress(
                    progress_ratio=1.,
                    message=f"Release failed: {e}",
                    step_id='progress',
                    status=ci.TaskletProgress.Status.FAILED,
                )
                self.output.state.success = False
                self.output.state.message = f"Step {idx} failed: {e}"
                if can_restart_from_scratch:
                    raise RetryTasklet(e)
                else:
                    raise

    def run(self):
        self.sandbox_resources = None
        self.sandbox_task_info = None
        self.prepare_clients()
        self.prepare_steps()

        attempts = 5
        for attempt in range(attempts):
            try:
                self.attempt()
            except RetryTasklet as e:
                log.error("Attempt failed")
                if attempt != attempts - 1:
                    sleep = 5.
                    log.info("Will retry after sleeping %.2f seconds", sleep)
                    time.sleep(sleep)
                else:
                    raise e.inner from None
            else:
                break

        self.save_progress(
            progress_ratio=1.,
            message="Done.",
            step_id='progress',
            status=ci.TaskletProgress.Status.SUCCESSFUL,
        )
        self.output.state.success = True
        self.output.state.message = "Success"


def _parse_attributes(attributes) -> dict[str, str]:
    if attributes is None:
        return {}
    assert isinstance(attributes, dict)
    return {str(k): str(v) for k, v in attributes.items()}


def _split_docker_image_name(image: str) -> tuple[str, str, str]:
    image, *tag_parts = image.rsplit(':', 1)
    if len(tag_parts) != 1:
        raise NotImplementedError("Docker images without exact tag are not supported yet")
    tag = tag_parts[0]

    registry, *image_parts = image.split('/', 1)
    if image_parts and '.' in registry:
        image = image_parts[0]
    else:
        registry = 'registry.yandex.net'

    return registry, image, tag


def _resolve_docker_info(registry: str, image: str, tag: str) -> dict:
    base_url = 'https://dockinfo.yandex-team.ru/api/docker/'
    rel_url = f'resolve?registryUrl={registry}/{image}&tag={tag}'

    with requests.Session() as s:
        s.mount(base_url, requests.adapters.HTTPAdapter(max_retries=5))
        r = s.get(urljoin(base_url, rel_url), timeout=30)
        r.raise_for_status()
        return r.json()


def _fill_release_docker(input, docker_spec: data_model.TDockerRelease):
    docker_spec.release_type = input.config.release_type or 'testing'
    docker_spec.release_author = input.context.flow_triggered_by

    seen_images = set()

    for patch in input.config.patches:
        registry, image, tag = _split_docker_image_name(patch.docker.image_name)

        seen_image = (registry, image, tag)
        if seen_image in seen_images:
            continue

        seen_images.add(seen_image)

        try:
            info = _resolve_docker_info(registry, image, tag)
            image_hash = info['hash']
        except Exception as e:
            raise Exception(f"Failed to find docker release for {registry}/{image}:{tag}: {e}")

        docker_image = docker_spec.images.add()
        docker_image.registry_host = registry
        docker_image.name = image
        docker_image.tag = tag
        docker_image.digest = image_hash


def collect_sandbox_infos(
    sandbox_client: SandboxClient,
    sandbox_resources: list[sandbox_pb2.SandboxResource]
) -> tuple[list[tuple[sandbox_pb2.SandboxResource, dict]], dict]:
    resources = [
        (resource, sandbox_client.resource[resource.id][:])
        for resource in sandbox_resources
    ]

    if resources:
        log.info("fetched %d resources infos", len(resources))
        task_info = sandbox_client.task[resources[0][1]['task']['id']][:]
        log.info("fetched task %s %r info", task_info['type'], task_info['id'])
    else:
        task_info = None

    return resources, task_info


def _fill_release_sandbox(
    sandbox_client: SandboxClient,
    resources: list[tuple[sandbox_pb2.SandboxResource, dict]],
    task_info: dict,
    input,
    sandbox_spec: data_model.TSandboxRelease,
):
    assert len(resources), "No sandbox resources given to make a release"

    sandbox_spec.title = task_info['type']
    sandbox_spec.description = task_info['description']
    sandbox_spec.task_type = task_info['type']
    sandbox_spec.task_id = str(task_info['id'])
    sandbox_spec.release_author = sandbox_client.user.current.read()["login"]
    sandbox_spec.task_author = task_info['author']
    sandbox_spec.release_type = input.config.release_type or 'testing'
    sandbox_spec.task_creation_time.FromJsonString(task_info['time']['created'])

    log.info("release author is %r", sandbox_spec.release_author)

    for resource, resource_info in resources:
        log.info("[resource %r] fetched info: %s", resource_info['id'], resource_info)
        if not resource_info['skynet_id']:
            log.info("[resource %r] skipped: no skynet_id available", resource_info['id'])
            continue

        sandbox_spec.task_id = str(resource_info['task']['id'])

        release_resource = sandbox_spec.resources.add()
        release_resource.resource_id = str(resource_info['id'])
        release_resource.type = str(resource_info['type'])
        release_resource.description = resource_info['description'] or ''
        release_resource.skynet_id = resource_info['skynet_id']
        release_resource.arch = resource_info['arch'] or ''
        release_resource.file_md5 = resource_info['md5'] or ''
        # release_resource.releasers.extend(resource_info['releasers'] or [])
        release_resource.filename = resource_info.get('file_name', resource_info.get('path')) or ''

        release_resource.attributes.update(_parse_attributes(resource_info.get('attributes')))
        release_resource.attributes.update(resource.attributes)


def set_ci_label(proto):
    label = proto.labels.attributes.add()
    label.key = 'deploy_engine'
    label.value = yt_yson_bindings.dumps("CI")


def make_id(input) -> str:
    flow_launch_id = hashlib.md5(input.context.job_instance_id.flow_launch_id.encode()).hexdigest()[:8]
    job_id = hashlib.md5(input.context.job_instance_id.job_id.encode()).hexdigest()[:8]

    return 'CI-{}-{}-{}-{}'.format(
        input.context.target_revision.hash[:8],
        flow_launch_id,
        job_id,
        input.context.job_instance_id.number,
    )


def make_build_name(context) -> str:
    if context.target_revision.pull_request_id:
        return (
            f"Pull request {context.target_revision.pull_request_id} "
            f"hash {context.target_revision.hash}"
        )
    else:
        return (
            f"Commit {context.target_revision.number} "
            f"hash {context.target_revision.hash}"
        )


def object_exists(yp_stub, ts: int, object_type: data_model.EObjectType, object_id: str) -> bool:
    req = object_service_pb2.TReqGetObject()
    req.timestamp = ts
    req.object_type = object_type
    req.object_id = object_id
    req.selector.paths.extend(['/meta/id'])

    try:
        retried_yp_call(yp_stub.GetObject, req)
    except YpNoSuchObjectError:
        return False
    else:
        return True


def approval_required(yp_stub, ts: int, stage_id: str) -> bool:
    req = object_service_pb2.TReqGetObject()
    req.timestamp = ts
    req.object_type = data_model.OT_APPROVAL_POLICY
    req.object_id = stage_id
    req.selector.paths.extend(['/spec'])
    req.format = object_service_pb2.PF_YSON

    try:
        result = retried_yp_call(yp_stub.GetObject, req).result.value_payloads
    except YpNoSuchObjectError:
        return False
    else:
        assert len(result) == 1, f"Unexpected answer from YP: 1 result expected, got {len(result)}"

        spec = yt_yson_bindings.loads_proto(
            result[0].yson,
            proto_class=data_model.TApprovalPolicySpec,
            skip_unknown_fields=True,
        )
        return spec.mode == data_model.TApprovalPolicySpec.REQUIRED and (
            spec.multiple_approval.approvals_count > 0
            or spec.mandatory_multiple_approval.approvals_count > 0
        )


def get_release_kind(patches: list) -> str:
    assert len(patches), "At least one patch must be present in tasklet `input.config.patches`"
    if all(
        patch.WhichOneof('payload') in ('sandbox', 'resource_bundle')
        for patch in patches
    ):
        return 'sandbox'
    elif all(
        patch.WhichOneof('payload') == 'docker'
        for patch in patches
    ):
        return 'docker'
    else:
        raise Exception("Mixed docker and sandbox releases ain't supported")


def create_transaction(yp_stub) -> tuple[int, str]:
    req = object_service_pb2.TReqStartTransaction()
    rsp = retried_yp_call(yp_stub.StartTransaction, req)
    log.info("started transaction at ts=%r, tx_id=%r", rsp.start_timestamp, rsp.transaction_id)
    return rsp.start_timestamp, rsp.transaction_id


def create_release(
    yp_stub,
    sandbox_client: SandboxClient,
    sandbox_resources: list[tuple[sandbox_pb2.SandboxResource, dict]],
    sandbox_task_info: dict,
    input,
    yp_tx_id: str,
    yp_timestamp: int,
    yp_address: str,
) -> tuple[str, str]:
    release_proto = data_model.TRelease()
    release_proto.meta.id = make_id(input)

    set_ci_label(release_proto)

    release_kind = get_release_kind(input.config.patches)
    release_spec = release_proto.spec

    if object_exists(yp_stub, yp_timestamp, data_model.OT_RELEASE, release_proto.meta.id):
        log.info("release %r is already created in YP %s", release_proto.meta.id, yp_address)
        return release_kind, release_proto.meta.id

    if release_kind == 'sandbox':
        _fill_release_sandbox(sandbox_client, sandbox_resources, sandbox_task_info, input, release_spec.sandbox)
    elif release_kind == 'docker':
        _fill_release_docker(input, release_spec.docker)
    else:
        raise NotImplementedError(f"Unsupported release kind: {release_kind!r}")

    release_spec.title = input.config.release_title or f"CI Build: {make_build_name(input.context)}"
    release_spec.description = input.config.release_description or (
        f'Autocheck for {make_build_name(input.context)}'
    )

    release_proto.status.processing.finished.status = data_model.CS_TRUE
    release_proto.status.processing.finished.last_transition_time.GetCurrentTime()
    release_proto.status.processing.finished.reason = 'CI'
    release_proto.status.processing.finished.message = 'Release processed by CI'

    log.info("prepared release proto: %s", release_proto)

    req = object_service_pb2.TReqCreateObject()
    req.object_type = data_model.OT_RELEASE
    req.transaction_id = yp_tx_id
    req.attributes_payload.yson = yt_yson_bindings.dumps_proto(release_proto)

    rsp = retried_yp_call(yp_stub.CreateObject, req)
    log.info("created release %r in YP %s", rsp.object_id, yp_address)

    return release_kind, rsp.object_id


def create_ticket(
    yp_stub,
    input,
    release_id: str,
    yp_tx_id: str,
    yp_timestamp: int,
    yp_address: str,
) -> tuple[set[str], set[str], str, bool]:
    affected_deploy_units = set()
    affected_dynamic_resources = set()

    ticket_proto = data_model.TDeployTicket()
    ticket_proto.meta.id = make_id(input)
    ticket_proto.meta.stage_id = input.config.stage_id
    ticket_proto.spec.source_type = ticket_proto.spec.RELEASE_INTEGRATION
    ticket_proto.spec.title = input.config.release_title or f'CI Build: {make_build_name(input.context)}'
    ticket_proto.spec.description = ''  # TODO take PR or commit description
    ticket_proto.spec.release_id = release_id
    for idx, patch in enumerate(input.config.patches):
        patch_kind = patch.WhichOneof('payload')
        if patch_kind == 'sandbox':
            sb = patch.sandbox
            sb_ref = sb.WhichOneof('resource_ref')
            if sb_ref == 'static':
                affected_deploy_units.add(sb.static.deploy_unit_id)
            elif sb_ref == 'dynamic':
                affected_dynamic_resources.add(sb.dynamic.dynamic_resource_id)
        elif patch_kind == 'docker':
            affected_deploy_units.add(patch.docker.docker_image_ref.deploy_unit_id)
        elif patch_kind == 'resource_bundle':
            if patch.resource_bundle.WhichOneof('resource_bundle_ref') == 'static':
                affected_deploy_units.add(patch.resource_bundle.static.deploy_unit_id)

        ticket_proto.spec.patches[f'patch{idx}'].CopyFrom(patch)
        ticket_proto.status.patches[f'patch{idx}'].SetInParent()
        if patch_kind == 'docker':
            _, image_name, _ = _split_docker_image_name(ticket_proto.spec.patches[f'patch{idx}'].docker.image_name)
            ticket_proto.spec.patches[f'patch{idx}'].docker.image_name = image_name

    log.info("affected deploy units %s", list(sorted(affected_deploy_units)))
    log.info("affected dynamic resources %s", list(sorted(affected_dynamic_resources)))

    set_ci_label(ticket_proto)

    req = object_service_pb2.TReqCreateObject()
    req.object_type = data_model.OT_DEPLOY_TICKET
    req.transaction_id = yp_tx_id
    req.attributes_payload.yson = yt_yson_bindings.dumps_proto(ticket_proto)

    if object_exists(yp_stub, yp_timestamp, data_model.OT_DEPLOY_TICKET, ticket_proto.meta.id):
        log.info("deploy ticket %r is already created in YP %s", ticket_proto.meta.id, yp_address)
        return affected_deploy_units, affected_dynamic_resources, ticket_proto.meta.id, False

    rsp = retried_yp_call(yp_stub.CreateObject, req)
    log.info("created deploy_ticket %r in YP %s", rsp.object_id, yp_address)

    return affected_deploy_units, affected_dynamic_resources, rsp.object_id, True


def commit_ticket(yp_stub, input, ticket_id: str, yp_tx_id: str, yp_address: str) -> int:
    action = data_model.TDeployTicketControl.TCommitAction()
    action.options.reason = 'CI_AUTOCOMMIT'
    action.options.message = f'CI autocommit: {input.config.release_title or make_build_name(input.context)}'
    action.options.patch_selector.type = data_model.DTPST_FULL

    req = object_service_pb2.TReqUpdateObject()
    req.object_type = data_model.OT_DEPLOY_TICKET
    req.object_id = ticket_id
    if yp_tx_id:
        req.transaction_id = yp_tx_id
    update = req.set_updates.add()
    update.path = '/control/commit'
    update.value_payload.yson = yt_yson_bindings.dumps_proto(action)

    result = retried_yp_call(yp_stub.UpdateObject, req)
    log.info(
        "committed deploy_ticket %r in YP %s",
        ticket_id,
        yp_address,
    )
    return result.commit_timestamp


def commit_transaction(yp_stub, yp_tx_id: str) -> int:
    req = object_service_pb2.TReqCommitTransaction()
    req.transaction_id = yp_tx_id
    rsp = retried_yp_call(yp_stub.CommitTransaction, req)
    log.info("committed transaction at %r", rsp.commit_timestamp)
    return rsp.commit_timestamp


def get_locations(
    yp_stub,
    input,
    affected_deploy_units: set[str],
    affected_dynamic_resources: set[str],
    yp_timestamp: int,
) -> tuple[dict[str, int], dict[str, int], list[tuple[str, str, int]]]:
    req = object_service_pb2.TReqGetObject()
    req.timestamp = yp_timestamp
    req.object_type = data_model.OT_STAGE
    req.object_id = input.config.stage_id
    req.selector.paths.extend(['/spec'])
    req.format = object_service_pb2.PF_YSON

    result = retried_yp_call(yp_stub.GetObject, req).result.value_payloads
    assert len(result) == 1, f"Unexpected answer from YP: 1 result expected, got {len(result)}"

    spec = yt_yson_bindings.loads_proto(
        result[0].yson,
        proto_class=data_model.TStageSpec,
        skip_unknown_fields=True,
    )

    deploy_units = {}
    dynamic_resources = {}
    need_approvals = []

    for du_id, du in spec.deploy_units.items():
        if du_id not in affected_deploy_units:
            continue

        deploy_units[du_id] = du.revision

        primitive = du.WhichOneof('pod_deploy_primitive')
        if primitive == 'replica_set':
            for location in du.deploy_settings.cluster_sequence:
                if location.need_approval:
                    need_approvals.append((du_id, location.yp_cluster, du.revision))
        elif primitive == 'multi_cluster_replica_set':
            # per-location approvals ain't supported for mcrs
            continue

    for dr_id, dr in spec.dynamic_resources.items():
        if dr_id not in affected_dynamic_resources:
            continue

        dynamic_resources[dr_id] = dr.dynamic_resource.revision

    return deploy_units, dynamic_resources, need_approvals


def _get_ticket_status(yp_stub, deploy_ticket_id: str) -> data_model.TDeployTicketStatus:
    req = object_service_pb2.TReqGetObject()
    req.object_type = data_model.OT_DEPLOY_TICKET
    req.object_id = deploy_ticket_id
    req.selector.paths.extend(['/status'])
    req.format = object_service_pb2.PF_YSON

    result = retried_yp_call(yp_stub.GetObject, req).result.value_payloads
    assert len(result) == 1, f"Unexpected answer from YP: 1 result expected, got {len(result)}"

    status = yt_yson_bindings.loads_proto(
        result[0].yson,
        proto_class=data_model.TDeployTicketStatus,
        skip_unknown_fields=True,
    )

    return status


def _check_ticket_fail(status: data_model.TDeployTicketStatus):
    for patch in status.patches.values():
        if patch.progress.cancelled.status == data_model.CS_TRUE:
            log.warning("Release was superseded")
            raise Exception("Release was superseded by some later deployment.")
        elif patch.progress.failed.status == data_model.CS_TRUE:
            log.warning("Release failed")
            raise Exception("Release failed.")


def wait_for_approve(yp_stub, deploy_ticket_id: str, timeout: int = 24 * 60 * 60) -> bool:
    approved = False
    deadline = time.time() + timeout
    while not approved and time.time() <= deadline:
        status = _get_ticket_status(yp_stub, deploy_ticket_id)
        approved = status.can_commit
        _check_ticket_fail(status)

        if not approved:
            time.sleep(max(0, min(5, deadline - time.time())))

    return approved


def wait_for_deploy(yp_stub, deploy_ticket_id: str, timeout: int = 24 * 60 * 60) -> bool:
    deployed = False
    deadline = time.time() + timeout
    while not deployed and time.time() <= deadline:
        status = _get_ticket_status(yp_stub, deploy_ticket_id)

        deployed = status.progress.closed.status == data_model.CS_TRUE

        if deployed:
            _check_ticket_fail(status)
        else:
            time.sleep(max(0, min(5, deadline - time.time())))

    return deployed


class RetryTasklet(Exception):
    def __init__(self, exc):
        self.inner = exc
