import time
import logging

from tasklet.services.yav.proto import yav_pb2 as yav

from yp.client import YpClient
from yp.common import YpDuplicateObjectIdError, YpNoSuchObjectError, YpNoSuchTransactionError
from yt.wrapper.retries import Retrier, YtError


log = logging.getLogger(__name__)
DEPLOY_URLS = {
    'xdc': 'https://deploy.yandex-team.ru/',
    'man-pre': 'https://man-pre.deploy.yandex-team.ru/',
    'sas-test': 'https://test.deploy.yandex-team.ru/',
}


class CommonInputMixin:
    yp_address: str
    yp_client: YpClient
    yp_stub: object  # object_service_pb2.ObjectService

    def get_token(self):
        secret_uid = self.input.context.secret_uid
        secret_key = 'deploy_ci.token'

        spec = yav.YavSecretSpec(uuid=secret_uid, key=secret_key)
        return self.ctx.yav.get_secret(spec, default_key=secret_key).secret

    @staticmethod
    def stage_status_link(
        yp_cluster: str = 'xdc',
        stage_id: str = '',
        deploy_unit_id: str = '',
        cluster: str = '',
        deploy_ticket_id: str = '',
        release_id: str = '',
    ) -> str:
        url = ''
        base_url = DEPLOY_URLS.get(yp_cluster)
        if base_url and release_id:
            url = base_url + f'releases/{release_id}'
        elif base_url and stage_id:
            url = base_url + f'stages/{stage_id}'
            if deploy_ticket_id:
                url += f'/deploy-tickets/{deploy_ticket_id}'
            elif deploy_unit_id:
                url += f'/status/{deploy_unit_id}'
                if cluster:
                    url += f'/{cluster}'
        return url

    def init_yp_client(self, token: str, yp_address: str = '') -> None:
        if not yp_address:
            yp_address = self.input.config.yp_cluster

        yp_address = yp_address or 'xdc'
        if '.' not in yp_address:
            yp_address = f'{yp_address}.yp.yandex.net:8090'

        self.yp_address = yp_address

        self.yp_client = YpClient(
            address=yp_address,
            config={
                'token': token,
            }
        )
        self.yp_stub = self.yp_client.create_grpc_object_stub()


def retried_yp_call(call, *args, ignore_exceptions=(), retry_count=6, backoff=20., **kwargs):
    class YpRetrier(Retrier):
        def __init__(self):
            retry_config = {
                'enable': True,
                'count': retry_count,
                'total_timeout': None,
                'backoff': {'policy': 'rounded_up_to_request_timeout'},
            }
            super().__init__(
                retry_config=retry_config,
                timeout=backoff * 1000.,
                exceptions=(YtError,),
                ignore_exceptions=ignore_exceptions + (YpDuplicateObjectIdError, YpNoSuchObjectError),
            )
            self.exception = None

        def action(self):
            return call(*args, **kwargs)

        def except_action(self, exception, attempt):
            log.exception("YP call %s failed with: %s", call, exception)
            if isinstance(exception, YpNoSuchTransactionError):
                if self.exception is None:
                    raise exception from None
                else:
                    raise self.exception from None
            else:
                self.exception = exception

        def backoff_action(self, attempt, backoff):
            log.info("will sleep %.2f seconds before next attempt", backoff)
            time.sleep(backoff)

    return YpRetrier().run()
