package exporter

import (
	"a.yandex-team.ru/infra/deploy_doctor/internal/abc"
	"a.yandex-team.ru/infra/deploy_doctor/internal/config"
	"a.yandex-team.ru/infra/deploy_doctor/internal/deploy"
	"a.yandex-team.ru/infra/deploy_doctor/internal/yp"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/yt/go/yterrors"
	"a.yandex-team.ru/yt/go/ytlock"
	"context"
	"errors"
	"fmt"
	"github.com/cenkalti/backoff/v4"
	"github.com/spf13/cobra"
	"net/http"
	"os"
	"sync"
	"time"
)

const (
	backoffTimeout = 10 * time.Second
	watchPeriod    = 60 * time.Second
	prefixStage    = "stages_stats_"
	prefixProblem  = "problems_stats_"
	qloudABC       = 741
)

type Engine struct {
	ctx             context.Context
	deployInspector *deploy.Inspector
	exitChan        chan struct{}
	shutDownFunc    context.CancelFunc
	cfg             *config.ExporterConfig
	yp              *yp.Client
	yt              yt.Client
	abcClient       *abc.Client
	log             *zap.Logger
	cache           sync.Map
	stageMetrics    []*unistat.Numeric
	problemMetrics  []*unistat.Numeric
}

func NewEngine(appCfg *config.ExporterConfig, l *zap.Logger) (*Engine, error) {
	ctx, shutdown := context.WithCancel(context.Background())
	return &Engine{
		ctx:          ctx,
		shutDownFunc: shutdown,
		exitChan:     make(chan struct{}),
		cfg:          appCfg,
		log:          l,
		deployInspector: &deploy.Inspector{
			Name:   "deploy",
			Config: appCfg.DeployChecks,
		},
	}, nil
}

func AbsoluteMaxAggregation() unistat.Aggregation {
	return unistat.StructuredAggregation{
		AggregationType: unistat.Absolute,
		Group:           unistat.Max,
		MetaGroup:       unistat.Max,
		Rollup:          unistat.Max,
	}
}

func (e *Engine) initMetrics() {
	e.stageMetrics = make([]*unistat.Numeric, deploy.LastCode+1)
	e.problemMetrics = make([]*unistat.Numeric, deploy.LastCode+1)
	for code := deploy.CodeStageInfo; code < deploy.LastCode; code++ {
		e.stageMetrics[code] = unistat.NewNumeric(prefixStage+code.String(), 1, AbsoluteMaxAggregation(), unistat.Last)
		e.problemMetrics[code] = unistat.NewNumeric(prefixProblem+code.String(), 1, AbsoluteMaxAggregation(), unistat.Last)
		unistat.Register(e.stageMetrics[code])
		unistat.Register(e.problemMetrics[code])
	}

	http.HandleFunc("/unistat", e.ServeUnistatHTTP)
	go func() {
		err := http.ListenAndServe(e.cfg.Unistat, nil)
		if err != nil {
			panic(fmt.Errorf("failed to start HTTP listener: %v", err))
		}
	}()

}

func (e *Engine) ServeUnistatHTTP(rw http.ResponseWriter, req *http.Request) {
	bytes, err := unistat.MarshalJSON()
	if err != nil {
		http.Error(rw, err.Error(), http.StatusInternalServerError)
		return
	}

	rw.Header().Set("Content-Type", "application/json; charset=utf-8")
	if _, err := rw.Write(bytes); err != nil {
		e.log.Errorf("write: %v", err)
	}
}

func (e *Engine) start() (err error) {
	e.yp, err = yp.NewYpClient(e.cfg.MainCluster)
	if err != nil {
		return fmt.Errorf("create YP client: %o", err)
	}

	e.yt, err = ythttp.NewClient(&yt.Config{
		Proxy: e.cfg.YT.Proxy,
		Token: os.Getenv("YT"),
	})

	if err != nil {
		return fmt.Errorf("create YT client: %o", err)
	}

	defer close(e.exitChan)

	e.log.Info("watcher started")

	lock := ytlock.NewLockOptions(e.yt, ypath.Path(e.cfg.YT.Path).JoinChild("lock"), ytlock.Options{
		CreateIfMissing: true,
		LockMode:        yt.LockExclusive,
	})

	e.abcClient = abc.NewClient(&abc.ClientConfig{
		IsEnabledTVM: e.cfg.IsTVMEnabled,
		TvmTicket:    e.cfg.TVMTicket,
	}, e.log)

	e.initMetrics()

	err = backoff.RetryNotify(
		func() error {
			tx, err := lock.Acquire(e.ctx)
			if err != nil {
				if e.ctx.Err() != nil {
					return backoff.Permanent(err)
				}

				return err
			}
			defer func() { _ = lock.Release(e.ctx) }()

			err = e.iteration(tx)
			if err != nil {
				if e.ctx.Err() != nil {
					return backoff.Permanent(err)
				}
				return err
			}

			return nil
		},
		backoff.WithContext(backoff.NewConstantBackOff(backoffTimeout), e.ctx),
		func(err error, sleep time.Duration) {
			if e.ctx.Err() != nil {
				return
			}

			if yterrors.ContainsErrorCode(err, yterrors.CodeConcurrentTransactionLockConflict) {
				e.log.Info("failed to acquire YT lock", log.String("error", err.Error()))
				return
			}

			e.log.Error("watch failed", log.Duration("sleep", sleep), log.Error(err))
		},
	)

	if err != nil && e.ctx.Err() == nil {
		return fmt.Errorf("failed to watch stages: %o", err)
	}
	e.log.Info("watcher stopped")
	return nil
}

func (e *Engine) sendToABC(response []deploy.Response, abcService int) error {
	return e.abcClient.SendProblems(abcService, response)
}

func (e *Engine) iteration(lostTx <-chan struct{}) error {
	e.log.Info("start main cycle")

	for {
		projects := e.yp.FetchAllProjects(e.ctx)
		stages := e.yp.FetchAllStages(e.ctx)
		stageMap := make(map[string]*ypapi.TStage)
		projectMap := make(map[string]*ypapi.TProject)
		abcToProblems := make(map[string][]deploy.Response)
		for _, stage := range stages {
			stageMap[stage.GetMeta().GetId()] = stage
		}
		for _, project := range projects {
			projectMap[project.GetMeta().GetId()] = project
		}

		problems := make([]deploy.Response, 0, len(stages))
		for _, stage := range stages {
			rev := stage.GetSpec().GetRevision()
			key := fmt.Sprintf("%s.%d", stage.GetMeta().GetId(), rev)
			oldProblems, found := e.cache.Load(key)
			if !found {
				data := e.deployInspector.Analyze(stage, projectMap[stage.GetMeta().GetProjectId()])
				problems = append(problems, data)
				e.cache.Store(key, data)
				e.log.Infof("new revision %d for stage %s", rev, stage.GetMeta().GetId())
			} else {
				problems = append(problems, oldProblems.(deploy.Response))
			}
		}

		total := 0
		countByCode := make(map[deploy.IssueCode]int)
		stagesByCode := make(map[deploy.IssueCode]map[string]bool)
		for code := deploy.CodeStageInfo; code < deploy.LastCode; code++ {
			stagesByCode[code] = make(map[string]bool)
		}

		for _, report := range problems {
			total += len(report.Issues)
			stageID := report.ObjectID
			for _, problem := range report.Issues {
				countByCode[problem.IssueCode]++
				stagesByCode[problem.IssueCode][stageID] = true
			}
			stage := stageMap[stageID]
			abcService := stage.GetMeta().GetAccountId()
			if abcService == "" {
				abcService = projectMap[stage.GetMeta().GetProjectId()].GetMeta().GetAccountId()
			}
			if abcService == "" || abcService == "tmp" {
				continue
			}
			abcToProblems[abcService] = append(abcToProblems[abcService], report)
		}

		qloudFullABC := fmt.Sprintf("abc:service:%d", qloudABC)
		err := e.sendToABC(abcToProblems[qloudFullABC], qloudABC)

		if err != nil {
			panic(err)
		} else {
			e.log.Info("successfully exported for qloud")
		}

		for code := deploy.CodeStageInfo; code < deploy.LastCode; code++ {
			e.log.Infof("%s: %d problems in %d stages", code.String(), countByCode[code], len(stagesByCode[code]))

			e.stageMetrics[code].Update(float64(len(stagesByCode[code])))
			e.problemMetrics[code].Update(float64(countByCode[code]))
		}

		if e.cfg.SyncWithABC {
			for abcFullID := range abcToProblems {
				err := e.sendToABC(abcToProblems[abcFullID], qloudABC)
				if err != nil {
					e.log.Errorf("error during service export: %v", err)
				}
			}
		}

		e.log.Infof("count stages: %d, count projects: %d", len(stages), len(projects))
		select {
		case <-e.ctx.Done():
			e.log.Info("watcher context canceled")
			return nil
		case <-lostTx:
			return errors.New("lost YT lock")
		case <-time.After(watchPeriod):

		}
	}
}

func (e *Engine) stop() {

}

func StartExport() *cobra.Command {
	return &cobra.Command{
		Use:   "export",
		Short: "export problems to ABC",
		Run: func(cmd *cobra.Command, args []string) {
			cfg, err := config.ReadExporterConfig(args[0])
			if err != nil {
				panic(err)
			}

			e, err := NewEngine(cfg, zap.Must(zap.NewProductionDeployConfig()))
			if err != nil {
				panic(err)
			}

			err = e.start()
			if err != nil {
				panic(err)
			}
		},
	}
}
