// Package server Infra doctor API.
//
//     Schemes: https
//     Host: deploy-doctor.in.yandex-team.ru
//     Version: 0.1.0
//
//     Consumes:
//     - application/json
//
//     Produces:
//     - application/json
//     - text/plain
//
// swagger:meta
package server

import (
	"a.yandex-team.ru/infra/deploy_doctor/internal/config"
	"a.yandex-team.ru/infra/deploy_doctor/internal/deploy"
	"a.yandex-team.ru/infra/deploy_doctor/internal/yp"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"context"
	"encoding/json"
	"fmt"
	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"
	"github.com/rs/cors"
	"github.com/spf13/cobra"
	"net/http"
	"strconv"
	"strings"
	"sync"
)

var ypClient *yp.Client
var bbClient *httpbb.Client
var deployInspector *deploy.Inspector
var cache sync.Map
var logger *zap.Logger
var whiteListedOrigins = []string{"https://localhost.yandex-team.ru", "https://*.deploy.yandex-team.ru", "https://yd.yandex-team.ru", "http://localhost"}
var mainDomain = "yd.yandex-team.ru"

const YpStage = "yp.stage"
const YpProject = "yp.project"

func Init(config *config.ServerConfig) {
	logger = zap.Must(zap.NewProductionDeployConfig())

	yc, err := yp.NewYpClient(config.MainCluster)
	if err != nil {
		panic(err)
	}

	ypClient = yc

	if config.AuthEnabled {
		tvmClient, err := tvmtool.NewDeployClient()
		if err != nil {
			panic(err)
		}

		bb, err := httpbb.NewIntranet(
			httpbb.WithLogger(logger),
			httpbb.WithTVM(tvmClient),
		)
		if err != nil {
			panic(err)
		}
		bbClient = bb
	}

	deployInspector = &deploy.Inspector{
		Name:   "deploy",
		Config: config.DeployChecks,
	}
}

func cookieChecker(SessionID string) {
	response, err := bbClient.SessionID(context.TODO(), blackbox.SessionIDRequest{
		SessionID: SessionID,
		Host:      mainDomain,
		Attributes: []blackbox.UserAttribute{
			blackbox.UserAttributeAccountNormalizedLogin,
		},
	})

	if err != nil {
		if blackbox.IsUnauthorized(err) {
			fmt.Println(fmt.Errorf("just unauthorized"))
		}
		panic(err)
	}

	fmt.Printf("user id: %d\n", response.User.ID)

	// Use the requested user attributes
	if login, ok := response.User.Attributes[blackbox.UserAttributeAccountNormalizedLogin]; ok {
		fmt.Printf("user login: %s\n", login)
	} else {
		fmt.Printf("user login: %s\n", response.User.Login)
	}
}

func processStage(stage *ypapi.TStage, project *ypapi.TProject) *deploy.AggregatedResponse {
	rev := stage.GetSpec().GetRevision()
	key := fmt.Sprintf("%s.%d", stage.GetMeta().GetId(), rev)
	if _, found := cache.Load(key); !found {
		data := deployInspector.Analyze(stage, project)
		cache.Store(key, data)
	}
	data, _ := cache.Load(key)
	statistics := deployInspector.Aggregate(data.(deploy.Response))
	return &statistics
}

func AuthMiddleware(next http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {

		sessionIDCookie, err := r.Cookie("Session_id")
		if err != nil {
			logger.Error("No Session_id cookie found")
		} else {
			cookieChecker(sessionIDCookie.Value)
		}

		next.ServeHTTP(w, r)
	})
}

func startServer(cfg *config.ServerConfig) {
	stageHandler := func(w http.ResponseWriter, req *http.Request) {
		stageID := chi.URLParam(req, "stageID")
		stage, _ := ypClient.StageInfoFull(context.Background(), stageID)
		if stage == nil {
			w.WriteHeader(http.StatusNotFound)
			return
		}
		project, _ := ypClient.ProjectInfoFull(context.Background(), stage.GetMeta().GetProjectId())

		if project == nil {
			w.WriteHeader(http.StatusNotFound)
			return
		}
		rev := stage.GetSpec().GetRevision()
		key := fmt.Sprintf("%s.%d", stage, rev)
		if _, found := cache.Load(key); !found {
			data := deployInspector.Analyze(stage, project)
			cache.Store(key, data)
		}
		rsp, _ := cache.Load(key)

		w.Header().Set("Content-Type", "application/json")
		if err := json.NewEncoder(w).Encode(rsp); err != nil {
			w.WriteHeader(http.StatusInternalServerError)
		}
	}

	// swagger:operation GET /api/v1/info object info
	//
	// Получение списка проблем объекта
	//
	// ---
	// parameters:
	// - name: id
	//   in: query
	//   type: string
	//   description: Object ID
	//   required: true
	//   example: stagectl
	// - name: object_type
	//   in: query
	//   type: string
	//   description: YP object type
	//   required: true
	//   example: yp.stage
	// - name: uuid
	//   in: query
	//   type: string
	//   description: object uuid
	//   required: false
	//   example: 8e9646cc-cfee-4c22-8f0d-01064b8147b5
	// responses:
	//   200:
	//     description: object issues list
	//     schema:
	//       "$ref": "#/definitions/Response"
	newStageHandler := func(w http.ResponseWriter, req *http.Request) {
		w.Header().Set("Access-Control-Allow-Origin", "*")
		stageID := req.URL.Query().Get("id")
		stageUUID := req.URL.Query().Get("uuid")
		objectType := req.URL.Query().Get("object_type")
		if strings.ToLower(objectType) != YpStage {
			w.WriteHeader(http.StatusNotFound)
			return
		}

		stage, _ := ypClient.StageInfoFull(context.Background(), stageID)
		if stage == nil || (req.URL.Query().Has("uuid") && stage.GetMeta().GetUuid() != stageUUID) {
			w.WriteHeader(http.StatusNotFound)
			return
		}
		project, _ := ypClient.ProjectInfoFull(context.Background(), stage.GetMeta().GetProjectId())

		if project == nil {
			w.WriteHeader(http.StatusNotFound)
			return
		}
		rev := stage.GetSpec().GetRevision()
		key := fmt.Sprintf("%s.%d", stage, rev)
		if _, found := cache.Load(key); !found {
			data := deployInspector.Analyze(stage, project)
			cache.Store(key, data)
		}
		rsp, _ := cache.Load(key)
		w.Header().Set("Content-Type", "application/json")
		if err := json.NewEncoder(w).Encode(rsp); err != nil {
			w.WriteHeader(http.StatusInternalServerError)
		}
	}

	// swagger:operation GET /api/v1/stat object info
	//
	// Получение агрегированной статистики о проблемах в проекте
	//
	// ---
	// parameters:
	// - name: id
	//   in: query
	//   type: string
	//   description: Object ID
	//   required: true
	//   example: stagectl
	// - name: object_type
	//   in: query
	//   type: string
	//   description: YP object type
	//   required: true
	//   example: yp.project
	// - name: uuid
	//   in: query
	//   type: string
	//   description: object uuid
	//   required: false
	//   example: 8e9646cc-cfee-4c22-8f0d-01064b8147b5
	// responses:
	//   200:
	//     description: project statistics info
	//     schema:
	//       "$ref": "#/definitions/AggregatedStatisticsResponse"
	statsHandler := func(w http.ResponseWriter, req *http.Request) {
		w.Header().Set("Access-Control-Allow-Origin", "*")
		objectType := req.URL.Query().Get("object_type")
		objectID := req.URL.Query().Get("id")
		if strings.ToLower(objectType) == YpProject {
			projectID := objectID
			project, _ := ypClient.ProjectInfoFull(context.Background(), projectID)

			if project == nil {
				w.WriteHeader(http.StatusNotFound)
				return
			}

			stages, err := ypClient.FetchAllStagesByProject(context.Background(), projectID)
			if err != nil {
				w.WriteHeader(http.StatusInternalServerError)
				return
			}

			rsp := deploy.AggregatedStatisticsResponse{
				Statistics: nil,
				Relative:   make(map[string]map[string]*deploy.AggregatedResponse),
			}

			rsp.Relative[YpStage] = make(map[string]*deploy.AggregatedResponse)
			for _, stage := range stages {
				statistics := processStage(stage, project)
				rsp.Relative[YpStage][stage.GetMeta().GetId()] = statistics
			}

			w.Header().Set("Content-Type", "application/json")
			if err := json.NewEncoder(w).Encode(rsp); err != nil {
				w.WriteHeader(http.StatusInternalServerError)
			}
			return
		}

		if strings.ToLower(objectType) == YpStage {
			stageID := objectID
			stage, _ := ypClient.StageInfoFull(context.Background(), stageID)
			if stage == nil {
				w.WriteHeader(http.StatusNotFound)
				return
			}
			project, _ := ypClient.ProjectInfoFull(context.Background(), stage.GetMeta().GetProjectId())

			if project == nil {
				w.WriteHeader(http.StatusNotFound)
				return
			}
			statistics := processStage(stage, project)
			rsp := deploy.AggregatedStatisticsResponse{
				Statistics: statistics,
				Relative:   nil,
			}
			w.Header().Set("Content-Type", "application/json")
			if err := json.NewEncoder(w).Encode(rsp); err != nil {
				w.WriteHeader(http.StatusInternalServerError)
			}
			return
		}

		w.WriteHeader(http.StatusNotFound)
	}

	reportByCodeHandler := func(w http.ResponseWriter, req *http.Request) {
		w.Header().Set("Access-Control-Allow-Origin", "*")
		problemCodeStr := req.URL.Query().Get("code")
		problemCode, err := strconv.Atoi(problemCodeStr)
		if err != nil {
			panic(err)
		}

		ctx := context.Background()

		projects := ypClient.FetchAllProjects(ctx)
		stages := ypClient.FetchAllStages(ctx)
		stageMap := make(map[string]*ypapi.TStage)
		projectMap := make(map[string]*ypapi.TProject)

		for _, stage := range stages {
			stageMap[stage.GetMeta().GetId()] = stage
		}

		for _, project := range projects {
			projectMap[project.GetMeta().GetId()] = project
		}

		problems := make([]deploy.Response, 0, len(stages))
		for _, stage := range stages {
			rev := stage.GetSpec().GetRevision()
			key := fmt.Sprintf("%s.%d", stage.GetMeta().GetId(), rev)
			oldProblems, found := cache.Load(key)
			if !found {
				data := deployInspector.Analyze(stage, projectMap[stage.GetMeta().GetProjectId()])
				problems = append(problems, data)
				cache.Store(key, data)
			} else {
				problems = append(problems, oldProblems.(deploy.Response))
			}
		}

		rsp := deploy.Report{
			Stages: make([]deploy.ReportItem, 0),
		}

		for _, report := range problems {
			for _, problem := range report.Issues {
				if problem.IssueCode == deploy.IssueCode(problemCode) {
					prj := stageMap[report.ObjectID].GetMeta().GetProjectId()
					owner := projectMap[prj].GetMeta().GetOwnerId()
					rsp.Stages = append(rsp.Stages, deploy.ReportItem{
						Owner:       owner,
						FullDU:      report.ObjectID + "." + problem.DeployUnit,
						Description: problem.Description,
					})
				}
			}
		}

		w.Header().Set("Content-Type", "application/json")
		if err := json.NewEncoder(w).Encode(rsp); err != nil {
			w.WriteHeader(http.StatusInternalServerError)
		}
	}
	r := chi.NewRouter()
	r.Use(middleware.Logger)

	if cfg.AuthEnabled {
		r.Use(AuthMiddleware)
	}

	corsHandler := cors.New(cors.Options{
		AllowedOrigins:   whiteListedOrigins,
		AllowedMethods:   []string{"GET", "POST", "OPTIONS", "HEAD"},
		AllowedHeaders:   []string{"Content-Type", "X-Requested-With", "X-CSRF-Token"},
		AllowCredentials: true,
		MaxAge:           300,
	})
	r.Use(corsHandler.Handler)

	r.Get("/stage/{stageID}", stageHandler)
	r.Get("/api/v1/info", newStageHandler)
	r.Get("/api/v1/stat", statsHandler)
	r.Get("/report_by_code", reportByCodeHandler)

	err := http.ListenAndServe(cfg.Address, r)
	if err != nil {
		logger.Fatal("unable to start server", log.Error(err))
	}
}

func StartServer() *cobra.Command {
	return &cobra.Command{
		Use:   "serve",
		Short: "Start infra doctor server",
		Run: func(cmd *cobra.Command, args []string) {
			cfg, err := config.ReadServerConfig(args[0])
			if err != nil {
				panic(err)
			}
			Init(cfg)
			startServer(cfg)
		},
	}
}
