package abc

import (
	"a.yandex-team.ru/infra/deploy_doctor/internal/deploy"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	tvmconsts "a.yandex-team.ru/library/go/httputil/middleware/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"context"
	"fmt"
	"strconv"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"
)

const (
	defaultURL            = "https://abc-back.yandex-team.ru/api/v4"
	defaultRequestTimeout = time.Second * 10
	StageHealth           = "stage_health"
)

type ClientConfig struct {
	URL            string        `yaml:"url"`
	RequestTimeout time.Duration `yaml:"request_timeout"`
	IsEnabledTVM   bool          `yaml:"is_enabled_tvm"`
	TvmTicket      string        `yaml:"tvm_ticket"`
}

type Client struct {
	c         *resty.Client
	tvm       *tvmtool.Client
	tvmTicket string
	logger    *zap.Logger
}

func NewClient(config *ClientConfig, logger *zap.Logger) *Client {
	requestTimeout := config.RequestTimeout
	if requestTimeout == 0 {
		requestTimeout = defaultRequestTimeout
	}

	URL := config.URL
	if URL == "" {
		URL = defaultURL
	}

	client := &Client{
		logger:    logger,
		tvmTicket: config.TvmTicket,
		c: resty.New().
			SetTimeout(requestTimeout).
			SetBaseURL(URL).
			SetHeader("Content-Type", "application/json").
			SetHeader("Accept", "application/json"),
	}

	if config.IsEnabledTVM {
		tvmClient, err := tvmtool.NewDeployClient(
			tvmtool.WithLogger(logger),
		)
		if err != nil {
			logger.Fatal("failed create tvm client", log.Error(err))
			panic(err)
		}
		client.tvm = tvmClient
	}

	return client
}

type IssueContext struct {
	En string `json:"en"`
	Ru string `json:"ru"`
}

type Issue struct {
	Code       string       `json:"code"`
	Context    IssueContext `json:"context"`
	Percentage float64      `json:"percentage_of_completion"`
}

type Request struct {
	Service   int     `json:"service"`
	GroupCode string  `json:"group_code"`
	Issues    []Issue `json:"issues"`
}

func (c *Client) SendProblems(abcService int, data []deploy.Response) error {
	rsp := make([]map[string]bool, deploy.LastCode+1)
	for _, stageReport := range data {
		for _, i := range stageReport.Issues {
			if i.IssueCode == 0 {
				continue
			}
			if rsp[i.IssueCode] == nil {
				rsp[i.IssueCode] = make(map[string]bool)
			}
			rsp[i.IssueCode][stageReport.ObjectID] = true
		}
	}

	issues := make([]Issue, 0)

	for code := deploy.CodeStageInfo + 1; code < deploy.CodeProjectQuotaMismatch; code++ {
		if len(rsp[code]) == 0 {
			continue
		}
		var sb strings.Builder
		for stage := range rsp[code] {
			if sb.Len() > 0 {
				sb.WriteString(", ")
			}
			sb.WriteString(stage)
		}

		issues = append(issues, Issue{
			Code: strconv.Itoa(int(code)),
			Context: IssueContext{
				En: sb.String(),
			},
			Percentage: 1,
		})
	}

	req := Request{
		Service:   abcService,
		GroupCode: StageHealth,
		Issues:    issues,
	}
	return c.SendProblemsBulk([]Request{req})
}

func (c *Client) getTVMTicket() string {
	if c.tvmTicket != "" {
		return c.tvmTicket
	}

	ticket, err := c.tvm.GetServiceTicketForAlias(context.Background(), "abc-proc")
	if err != nil {
		c.logger.Fatal("failed to get tvm ticket", log.String("dst", "abc-prod"), log.Error(err))
		panic(err)
	}

	return ticket
}

func (c *Client) SendProblemsBulk(data interface{}) error {
	//b, _ := json.Marshal(data)
	//fmt.Println(string(b))

	resp, err := c.c.R().SetBody(data).
		SetHeader(tvmconsts.XYaServiceTicket, c.getTVMTicket()).
		Put("/suspicion/issues/bulk/")
	if err != nil {
		return err
	}
	if resp.StatusCode() < 200 || resp.StatusCode() > 299 {
		return fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}
	return nil
}

//get test ticket
//curl "$DEPLOY_TVM_TOOL_URL/tvm/tickets?src=deploy-doctor&dsts=abc-prod" -H "Authorization: $TVMTOOL_LOCAL_AUTHTOKEN"
