package config

import (
	"context"
	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"
	"time"
)

func defaultDeployChecksConfig() ChecksConfig {
	return ChecksConfig{
		TargetPatcherRevision:         13,
		SafePatcherRevision:           11,
		TargetPodAgentRevision:        3006211268,
		TargetLogbrokerToolsRevision:  2739742779,
		TargetDruAgentRevision:        1985693042,
		TargetTvmClientRevision:       3017078191,
		SuccessfulDeployTimeThreshold: 24 * time.Hour,          // one day
		LastDeployTimeThreshold:       3 * 30 * 24 * time.Hour, // around 1/4 year
	}
}

type ChecksConfig struct {
	TargetPatcherRevision         uint32        `yaml:"target_patcher_revision"`
	SafePatcherRevision           uint32        `yaml:"safe_patcher_revision"`
	TargetPodAgentRevision        uint64        `yaml:"target_pod_agent_revision"`
	TargetLogbrokerToolsRevision  uint64        `yaml:"target_logbroker_tools_revision"`
	TargetDruAgentRevision        uint64        `yaml:"target_dru_agent_revision"`
	TargetTvmClientRevision       uint64        `yaml:"target_tvm_client_revision"`
	SuccessfulDeployTimeThreshold time.Duration `yaml:"successful_deploy_time_threshold"`
	LastDeployTimeThreshold       time.Duration `yaml:"last_deploy_time_threshold"`
}

type ServerConfig struct {
	MainCluster  string       `yaml:"main_cluster"`
	Address      string       `yaml:"address"`
	DeployChecks ChecksConfig `yaml:"deploy_checks"`
	AuthEnabled  bool         `yaml:"auth_enabled"`
}

type YTLockConfig struct {
	Proxy string `yaml:"proxy"`
	Token string `yaml:"token"`
	Path  string `yaml:"path"`
}

type ExporterConfig struct {
	MainCluster  string       `yaml:"main_cluster"`
	Unistat      string       `yaml:"unistat"`
	YT           YTLockConfig `yaml:"yt"`
	IsTVMEnabled bool         `yaml:"is_tvm_enabled"`
	TVMTicket    string       `yaml:"ticket"`
	SyncWithABC  bool         `yaml:"sync_with_abc"`
	DeployChecks ChecksConfig `yaml:"deploy_checks"`
}

func ReadExporterConfig(path string) (config *ExporterConfig, err error) {
	cfg := &ExporterConfig{
		DeployChecks: defaultDeployChecksConfig(),
	}

	loader := confita.NewLoader(env.NewBackend(), file.NewBackend(path))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, err
	}

	return cfg, nil
}

func ReadServerConfig(path string) (*ServerConfig, error) {
	cfg := &ServerConfig{
		AuthEnabled:  false,
		MainCluster:  "xdc",
		Address:      ":80",
		DeployChecks: defaultDeployChecksConfig(),
	}

	loader := confita.NewLoader(env.NewBackend(), file.NewBackend(path))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, err
	}

	return cfg, nil
}
