package deploy

import (
	"a.yandex-team.ru/infra/deploy_doctor/internal/config"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"fmt"
	"sort"
	"strconv"
	"strings"
	"time"
)

type Level string
type IssueClass string
type ActionCode string

const (
	Info Level = "info" // Info = 0
	Warn Level = "warn" // Warn = 1
	Crit Level = "crit" // Crit = 2
)

const baseDocURL = "https://deploy.yandex-team.ru/docs/reference/stage-health#"

var unknownResourceIDS = [...]int{3112769822, 725288970, 2035482041, 3184188154, 3246581353, 3167074025, 2737752116, 3262509008, 3044735180, 2316855241, 2697481489, 2767415909, 1523190760, 75537503911111111, 2880290326, 2793984439, 2685770147, 3180848874, 2767732512, 3042798409, 2901111800, 2767559451}

func isUnknownResource(val int) bool {
	for _, x := range unknownResourceIDS {
		if val == x {
			return true
		}
	}
	return false
}

var levelsPriority = map[Level]uint32{
	Crit: 0,
	Warn: 1,
	Info: 2,
}

const (
	Help           ActionCode = "help"
	UpdateSidecars ActionCode = "update-sidecars"
	UpdateRuntime  ActionCode = "update-runtime-version"
)

const (
	Security          IssueClass = "security"
	OldInfraResources IssueClass = "old_infra_resources"
	AccessIssue       IssueClass = "access_issues"
	BestPractice      IssueClass = "best_practice"
	Other             IssueClass = "other"
)

type IssueCode int

const (
	CodeStageInfo IssueCode = iota
	CodeOldRuntime
	CodeOldSidecar
	CodeNoRednessProbe
	CodeNoAlerts
	CodeOldSecrets
	CodeNoAnonLimit
	CodeDangerousBudget
	CodeNoOwner // implement me
	CodeNoDeploys
	CodeNotDeployed
	CodeDangerousNetwork
	CodeCommonIType
	CodeResourceProtocolProblem
	CodeSchedulingHints
	CodeSoxIssues
	CodeXrayIssues
	CodeAccessIssues
	CodeLowResourceUtilization
	CodeUnknownResource
	CodeNoNetworkGuarantees
	CodeProjectQuotaMismatch
	LastCode
)

func (s IssueCode) String() string {
	switch s {
	case CodeStageInfo:
		return "code_stage_info"
	case CodeOldRuntime:
		return "code_old_runtime"
	case CodeOldSidecar:
		return "code_old_sidecar"
	case CodeNoRednessProbe:
		return "code_no_redness_probe"
	case CodeNoAlerts:
		return "code_no_alerts"
	case CodeOldSecrets:
		return "code_old_secrets"
	case CodeNoAnonLimit:
		return "code_no_anon_limit"
	case CodeDangerousBudget:
		return "code_dangerous_budget"
	case CodeNoOwner:
		return "code_no_owner"
	case CodeNoDeploys:
		return "code_no_deploys"
	case CodeNotDeployed:
		return "code_not_deployed"
	case CodeDangerousNetwork:
		return "code_dangerous_network"
	case CodeCommonIType:
		return "code_common_itype"
	case CodeResourceProtocolProblem:
		return "code_resource_protocol_problem"
	case CodeSchedulingHints:
		return "code_scheduling_hints"
	case CodeSoxIssues:
		return "code_sox_issues"
	case CodeXrayIssues:
		return "code_xray_issues"
	case CodeAccessIssues:
		return "code_access_issues"
	case CodeLowResourceUtilization:
		return "code_low_resource_utilization"
	case CodeUnknownResource:
		return "code_unknown_resource"
	case CodeNoNetworkGuarantees:
		return "code_no_network_guarantees"
	case CodeProjectQuotaMismatch:
		return "code_project_quota_mismatch"
	}
	return "unknown"
}

func (s IssueCode) BuildAction() Action {
	if s == CodeOldRuntime {
		return Action{
			Code: UpdateRuntime,
		}
	}
	if s == CodeOldSidecar {
		return Action{
			Code: UpdateSidecars,
		}
	}
	return Action{
		Code:    Help,
		Message: baseDocURL + s.String(),
	}
}

var BadNetworks = []string{"_SEARCHSAND_"}

type Inspector struct {
	Name   string
	Config config.ChecksConfig
}

func getPodSpec(du *ypapi.TDeployUnitSpec) *ypapi.TPodSpec {
	if du.GetReplicaSet() != nil {
		return du.GetReplicaSet().GetReplicaSetTemplate().GetPodTemplateSpec().GetSpec()
	}

	return du.GetMultiClusterReplicaSet().GetReplicaSet().GetPodTemplateSpec().GetSpec()
}

func getPodSpecLocator(du *ypapi.TDeployUnitSpec) string {
	if du.GetReplicaSet() != nil {
		return "replica_set/replica_set_template/pod_template_spec/spec"
	}
	return "multi_cluster_replica_set/replica_set/pod_template_spec/spec"
}

func contains(list []string, s string) bool {
	for _, el := range list {
		if el == s {
			return true
		}
	}
	return false
}

func locator(du string) string {
	return "/spec/deploy_units/" + du
}

func (i *Inspector) Analyze(stage *ypapi.TStage, project *ypapi.TProject) Response {
	problems := make([]Issue, 0)

	for duID, du := range stage.GetSpec().GetDeployUnits() {
		// Patchers and Sidecars checks
		var useDynamicResources = false
		for _, dr := range stage.GetSpec().GetDynamicResources() {
			if dr.GetDeployUnitRef() == duID {
				useDynamicResources = true
				break
			}
		}

		if du.GetPatchersRevision() < i.Config.TargetPatcherRevision {
			problemLevel := Warn
			if du.GetPatchersRevision() < i.Config.SafePatcherRevision {
				problemLevel = Crit
			}
			problems = append(problems, Issue{
				Level:      problemLevel,
				IssueCode:  CodeOldRuntime,
				Class:      OldInfraResources,
				Action:     CodeOldRuntime.BuildAction(),
				Locator:    locator(duID),
				DeployUnit: duID,
				Description: fmt.Sprintf("Runtime version is %d instead of %d recommended", du.GetPatchersRevision(),
					i.Config.TargetPatcherRevision),
			})
		}

		if du.GetPodAgentSandboxInfo().GetRevision() < i.Config.TargetPodAgentRevision {
			problems = append(problems, Issue{
				Level:      Warn,
				IssueCode:  CodeOldSidecar,
				Class:      OldInfraResources,
				Locator:    locator(duID),
				DeployUnit: duID,
				Action:     CodeOldSidecar.BuildAction(),
				Description: fmt.Sprintf("PodAgent version is %d instead of %d recommended",
					du.GetPodAgentSandboxInfo().GetRevision(), i.Config.TargetPodAgentRevision),
			})
		}

		if du.GetTvmConfig().GetMode() == ypapi.TTvmConfig_ENABLED &&
			du.GetTvmSandboxInfo().GetRevision() > 0 &&
			du.GetTvmSandboxInfo().GetRevision() < i.Config.TargetTvmClientRevision {
			problems = append(problems, Issue{
				Level:      Warn,
				IssueCode:  CodeOldSidecar,
				Class:      OldInfraResources,
				Locator:    locator(duID),
				DeployUnit: duID,
				Action:     CodeOldSidecar.BuildAction(),
				Description: fmt.Sprintf("TVM tools version is %d instead of %d recommended",
					du.GetTvmSandboxInfo().GetRevision(), i.Config.TargetTvmClientRevision),
			})
		}

		podSpec := getPodSpec(du)
		podSpecLocator := getPodSpecLocator(du)
		// TODO: consider other logs
		//podSpec.GetPodAgentPayload().GetSpec().GetWorkloads()[...any].GetTransmitLogs()
		//podSpec.GetPodAgentPayload().GetSpec().GetTransmitSystemLogsPolicy() == ypapi.ETransmitSystemLogs_ETransmitSystemLogsPolicy_ENABLED

		if du.GetLogbrokerConfig().GetSidecarBringupMode() == ypapi.TLogbrokerConfig_MANDATORY &&
			du.GetLogbrokerToolsSandboxInfo().GetRevision() > 0 &&
			du.GetLogbrokerToolsSandboxInfo().GetRevision() < i.Config.TargetLogbrokerToolsRevision {
			problems = append(problems, Issue{
				Level:      Warn,
				IssueCode:  CodeOldSidecar,
				Class:      OldInfraResources,
				Locator:    locator(duID),
				DeployUnit: duID,
				Action:     CodeOldSidecar.BuildAction(),
				Description: fmt.Sprintf("LogbrokerTools version is %d instead of %d recommended",
					du.GetLogbrokerToolsSandboxInfo().GetRevision(), i.Config.TargetLogbrokerToolsRevision),
			})
		}

		if useDynamicResources && du.GetDynamicResourceUpdaterSandboxInfo().GetRevision() < i.Config.TargetDruAgentRevision {
			problems = append(problems, Issue{
				Level:      Warn,
				IssueCode:  CodeOldSidecar,
				Class:      OldInfraResources,
				Locator:    locator(duID),
				DeployUnit: duID,
				Action:     CodeOldSidecar.BuildAction(),
				Description: fmt.Sprintf("DRU sidecar version is %d instead of %d recommended",
					du.GetDynamicResourceUpdaterSandboxInfo().GetRevision(), i.Config.TargetDruAgentRevision),
			})
		}

		// Alerting checks
		if stage.GetSpec().GetDeployUnitSettings()[duID].GetAlerting().GetState() != ypapi.TStageSpec_TDeployUnitSettings_TAlerting_ENABLED {
			problems = append(problems, Issue{
				Level:       Info,
				IssueCode:   CodeNoAlerts,
				Class:       BestPractice,
				Locator:     locator(duID),
				DeployUnit:  duID,
				Action:      CodeNoAlerts.BuildAction(),
				Description: "Alerting is not configured",
			})
		}

		// Check Readness limits
		for index, w := range podSpec.GetPodAgentPayload().GetSpec().GetWorkloads() {
			if w.GetReadinessCheck() == nil {
				problems = append(problems, Issue{
					Level:       Warn,
					IssueCode:   CodeNoRednessProbe,
					Class:       BestPractice,
					Locator:     fmt.Sprintf("%s/%s/pod_agent_payload/spec/workloads/%d", locator(duID), podSpecLocator, index),
					DeployUnit:  duID,
					Action:      CodeNoRednessProbe.BuildAction(),
					Description: "Readness probe is not configured",
				})
			}
		}

		// Check anon limits
		if du.GetPatchersRevision() < i.Config.SafePatcherRevision {
			for index, b := range podSpec.GetPodAgentPayload().GetSpec().GetBoxes() {
				if b.GetComputeResources().GetAnonymousMemoryLimit() == 0 ||
					b.GetComputeResources().GetAnonymousMemoryLimit() == b.GetComputeResources().GetMemoryLimit() {
					problems = append(problems, Issue{
						Level:       Info,
						IssueCode:   CodeNoAnonLimit,
						Class:       BestPractice,
						Locator:     fmt.Sprintf("%s/%s/pod_agent_payload/spec/boxes/%d", locator(duID), podSpecLocator, index),
						DeployUnit:  duID,
						Action:      CodeNoAnonLimit.BuildAction(),
						Description: "Box without anon memory limits",
					})
				}
			}
		}

		if contains(BadNetworks, du.GetNetworkDefaults().GetNetworkId()) {
			problems = append(problems, Issue{
				Level:       Crit,
				IssueCode:   CodeDangerousNetwork,
				Class:       Security,
				Locator:     locator(duID),
				DeployUnit:  duID,
				Action:      CodeDangerousNetwork.BuildAction(),
				Description: fmt.Sprintf("Network macros %s is not recommended", du.GetNetworkDefaults().GetNetworkId()),
			})
		}

		// Secrets check
		if len(podSpec.GetSecrets()) > 0 {
			problems = append(problems, Issue{
				Level:       Crit,
				IssueCode:   CodeOldSecrets,
				Class:       Security,
				Locator:     locator(duID),
				DeployUnit:  duID,
				Action:      CodeOldSecrets.BuildAction(),
				Description: "Old secrets usage",
			})
		}

		// Itype check
		itype := podSpec.GetHostInfra().GetMonitoring().GetLabels()["itype"]
		if itype == "" || itype == "deploy" {
			problems = append(problems, Issue{
				Level:       Warn,
				IssueCode:   CodeCommonIType,
				Class:       BestPractice,
				Locator:     locator(duID),
				DeployUnit:  duID,
				Action:      CodeCommonIType.BuildAction(),
				Description: "Common itype usage",
			})
		}

		// Resources check
		resources := podSpec.GetPodAgentPayload().GetSpec().GetResources()
		for idx, layer := range resources.GetLayers() {
			if layer.GetUrl() == "" {
				continue
			}
			if !strings.HasPrefix(layer.GetUrl(), "sbr:") && layer.GetMeta().GetSandboxResource().GetResourceId() == "" {
				problems = append(problems, Issue{
					Level:       Warn,
					IssueCode:   CodeResourceProtocolProblem,
					Class:       BestPractice,
					Locator:     fmt.Sprintf("%s/%s/pod_agent_payload/spec/resources/layers/%d", locator(duID), podSpecLocator, idx),
					DeployUnit:  duID,
					Action:      CodeResourceProtocolProblem.BuildAction(),
					Description: fmt.Sprintf("Preferably use sbr resource or specify meta for layer %s", layer.GetId()),
				})
			}

			if strings.HasPrefix(layer.GetUrl(), "sbr:") {
				rids := layer.GetUrl()[4:]
				if rid, err := strconv.Atoi(rids); err == nil {
					if !isUnknownResource(rid) {
						continue
					}
				}

				problems = append(problems, Issue{
					Level:       Crit,
					IssueCode:   CodeUnknownResource,
					Class:       BestPractice,
					Locator:     fmt.Sprintf("%s/%s/pod_agent_payload/spec/resources/layers/%d", locator(duID), podSpecLocator, idx),
					DeployUnit:  duID,
					Action:      CodeUnknownResource.BuildAction(),
					Description: fmt.Sprintf("Unknown resource %s for layer %s", layer.GetUrl(), layer.GetId()),
				})
			}
		}

		for idx, resource := range resources.GetStaticResources() {
			if resource.GetUrl() == "" {
				continue
			}
			if !strings.HasPrefix(resource.GetUrl(), "sbr:") && resource.GetMeta().GetSandboxResource().GetResourceId() == "" {
				problems = append(problems, Issue{
					Level:       Warn,
					IssueCode:   CodeResourceProtocolProblem,
					Class:       BestPractice,
					Locator:     fmt.Sprintf("%s/%s/pod_agent_payload/spec/resources/static_resources/%d", locator(duID), podSpecLocator, idx),
					DeployUnit:  duID,
					Action:      CodeResourceProtocolProblem.BuildAction(),
					Description: fmt.Sprintf("Preferably use sbr resource or specify meta for static resource %s", resource.GetId()),
				})
			}

			if strings.HasPrefix(resource.GetUrl(), "sbr:") {
				rids := resource.GetUrl()[4:]
				if rid, err := strconv.Atoi(rids); err == nil {
					if !isUnknownResource(rid) {
						continue
					}
				}
				problems = append(problems, Issue{
					Level:       Crit,
					IssueCode:   CodeUnknownResource,
					Class:       BestPractice,
					Locator:     fmt.Sprintf("%s/%s/pod_agent_payload/spec/resources/layers/%d", locator(duID), podSpecLocator, idx),
					DeployUnit:  duID,
					Action:      CodeUnknownResource.BuildAction(),
					Description: fmt.Sprintf("Unknown resource %s for static resource %s", resource.GetUrl(), resource.GetId()),
				})
			}
		}

		// Budget check
		if du.GetReplicaSet() != nil {
			for cluster, spec := range du.GetReplicaSet().GetPerClusterSettings() {
				podCount := spec.GetPodCount()
				if podCount == 0 {
					problems = append(problems, Issue{
						Level:       Crit,
						IssueCode:   CodeDangerousBudget,
						Class:       BestPractice,
						Locator:     fmt.Sprintf("%s/replica_set/per_cluster_settings/%s", locator(duID), cluster),
						DeployUnit:  duID,
						Action:      CodeDangerousBudget.BuildAction(),
						Description: "No pods in location",
					})
				} else {
					if spec.GetDeploymentStrategy().GetMaxUnavailable() == 0 || spec.GetDeploymentStrategy().GetMaxUnavailable() >= podCount {
						problems = append(problems, Issue{
							Level:       Info,
							IssueCode:   CodeDangerousBudget,
							Class:       BestPractice,
							Locator:     fmt.Sprintf("%s/replica_set/per_cluster_settings/%s", locator(duID), cluster),
							DeployUnit:  duID,
							Action:      CodeDangerousBudget.BuildAction(),
							Description: fmt.Sprintf("Disruption budget will lead to unavailable location (%s)", cluster),
						})
					}
				}
			}
		}

		// Check budget
		if du.GetMultiClusterReplicaSet() != nil {
			var podCount uint32
			for _, spec := range du.GetMultiClusterReplicaSet().GetReplicaSet().GetClusters() {
				podCount += spec.GetSpec().GetReplicaCount()
			}
			if podCount == 0 {
				problems = append(problems, Issue{
					Level:       Crit,
					IssueCode:   CodeDangerousBudget,
					Class:       BestPractice,
					Locator:     fmt.Sprintf("%s/multi_cluster_replica_set", locator(duID)),
					DeployUnit:  duID,
					Description: "No pods in mcrs",
				})
			}
			if du.GetMultiClusterReplicaSet().GetReplicaSet().GetDeploymentStrategy().GetMaxUnavailable() == 0 ||
				du.GetMultiClusterReplicaSet().GetReplicaSet().GetDeploymentStrategy().GetMaxUnavailable() >= podCount {
				problems = append(problems, Issue{
					Level:       Info,
					IssueCode:   CodeDangerousBudget,
					Class:       BestPractice,
					Locator:     fmt.Sprintf("%s/multi_cluster_replica_set", locator(duID)),
					DeployUnit:  duID,
					Action:      CodeDangerousBudget.BuildAction(),
					Description: "Disruption budget will lead to unavailable mcrs",
				})
			}
		}

		if len(podSpec.GetScheduling().GetHints()) > 0 {
			problems = append(problems, Issue{
				Level:       Warn,
				IssueCode:   CodeSchedulingHints,
				Class:       Other,
				Locator:     fmt.Sprintf("%s/%s/scheduling/hints", locator(duID), podSpecLocator),
				DeployUnit:  duID,
				Action:      CodeSchedulingHints.BuildAction(),
				Description: fmt.Sprintf("Used %d scheduling hint(s). These may lead to long allocation", len(podSpec.GetScheduling().GetHints())),
			})
		}

		if podSpec.GetResourceRequests().GetNetworkBandwidthGuarantee() == 0 {
			problems = append(problems, Issue{
				Level:       Warn,
				IssueCode:   CodeNoNetworkGuarantees,
				Class:       BestPractice,
				Locator:     fmt.Sprintf("%s/%s/resource_requests/network_bandwidth_guarantee", locator(duID), podSpecLocator),
				DeployUnit:  duID,
				Action:      CodeNoNetworkGuarantees.BuildAction(),
				Description: "Network bandwidth guarantee will be required soon",
			})
		}
	}

	for duID, du := range stage.GetStatus().GetDeployUnits() {

		if time.Since(time.UnixMilli(int64(du.GetTargetSpecTimestamp()))) > i.Config.LastDeployTimeThreshold {
			problems = append(problems, Issue{
				Level:       Warn,
				IssueCode:   CodeNoDeploys,
				Class:       BestPractice,
				Action:      CodeNoDeploys.BuildAction(),
				Locator:     locator(duID),
				DeployUnit:  duID,
				Description: fmt.Sprintf("Deploy unit %s has not been deployed for a long time", duID),
			})
		}

		if du.GetDeployUnitTimeline().GetStatus() == ypapi.TDeployUnitStatus_TDeployUnitTimeline_DEPLOYING &&
			du.GetDeployUnitTimeline().GetLatestDeployedRevision() > 0 &&
			time.Since(time.UnixMilli(int64(du.GetDeployUnitTimeline().GetStartTimestamp()))) > i.Config.SuccessfulDeployTimeThreshold {
			problems = append(problems, Issue{
				Level:       Crit,
				IssueCode:   CodeNotDeployed,
				Class:       Other,
				Action:      CodeNotDeployed.BuildAction(),
				Locator:     locator(duID),
				DeployUnit:  duID,
				Description: fmt.Sprintf("Deploy unit %s is not ready for a long time", duID),
			})
		}

	}

	if stage.GetMeta().GetAccountId() != project.GetMeta().GetAccountId() {
		problems = append(problems, Issue{
			Level:       Info,
			IssueCode:   CodeProjectQuotaMismatch,
			Class:       BestPractice,
			Action:      CodeProjectQuotaMismatch.BuildAction(),
			Locator:     "/meta/account_id",
			DeployUnit:  "",
			Description: "Stage account differ from project account",
		})
	}

	sort.Slice(problems, func(i, j int) bool {
		if levelsPriority[problems[i].Level] != levelsPriority[problems[j].Level] {
			return levelsPriority[problems[i].Level] < levelsPriority[problems[j].Level]
		}
		if problems[i].DeployUnit != problems[j].DeployUnit {
			return problems[i].DeployUnit < problems[j].DeployUnit
		}
		return problems[i].IssueCode < problems[j].IssueCode
	})

	return Response{
		ObjectID: stage.GetMeta().GetId(),
		Revision: stage.GetSpec().GetRevision(),
		Issues:   problems,
	}
}

func (i *Inspector) Aggregate(resp Response) AggregatedResponse {
	res := AggregatedResponse{
		CountByLevel: map[Level]uint32{},
	}
	for _, issue := range resp.Issues {
		res.CountByLevel[issue.Level]++
	}
	return res
}

type Action struct {
	Code    ActionCode `json:"code"`
	Message string     `json:"message"`
}

type Issue struct {
	// Issue crit level
	Level Level `json:"level"`
	// yp-path
	Locator     string     `json:"locator"`
	IssueCode   IssueCode  `json:"issue_code"`
	Class       IssueClass `json:"class"`
	Description string     `json:"description"`
	Action      Action     `json:"action"`
	DeployUnit  string
}

// Response returns data for stage issues
// swagger:model Response
type Response struct {
	// YP object id
	ObjectID string `json:"object_id"`
	// Object revision
	Revision uint32 `json:"revision"`
	// Issue list
	Issues []Issue `json:"issues"`
}

type AggregatedResponse struct {
	// count_by_level, map level в число
	CountByLevel map[Level]uint32 `json:"count_by_level"`
}

// AggregatedStatisticsResponse returns data for aggregated issues
// swagger:model AggregatedStatisticsResponse
type AggregatedStatisticsResponse struct {
	// statistics, map stage_id в статистику
	Statistics *AggregatedResponse                       `json:"statistics"`
	Relative   map[string]map[string]*AggregatedResponse `json:"relative"`
}

type ReportItem struct {
	Owner       string `json:"owner"`
	FullDU      string `json:"fullDU"`
	Description string `json:"description"`
}
type Report struct {
	Stages []ReportItem `json:"stages"`
}
