package yp

import (
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
	"context"
	"fmt"
)

const (
	DefaultGetObjectsBatchSize = 1000
	MetaSelector               = "/meta"
	SpecSelector               = "/spec"
	StatusSelector             = "/status"
)

var DefaultGetObjectsSelectors = []string{MetaSelector, SpecSelector, StatusSelector}
var LightGetObjectsSelectors = []string{MetaSelector, SpecSelector}
var DefaultSelectIDsSelectors = []string{MetaSelector}

type Client struct {
	*yp.Client
}

func NewYpClient(cluster string) (*Client, error) {
	c, err := yp.NewClient(cluster, yp.WithSystemAuthToken())
	if err != nil {
		return nil, fmt.Errorf("create YP-client: %w", err)
	}

	return &Client{
		Client: c,
	}, nil
}

func (y *Client) StageInfoFull(ctx context.Context, stageName string) (*ypapi.TStage, error) {
	rsp, err := y.GetStage(ctx, yp.GetStageRequest{
		Format:    yp.PayloadFormatProto,
		ID:        stageName,
		Selectors: DefaultGetObjectsSelectors,
	})
	if err != nil {
		return nil, fmt.Errorf("get stage: %w", err)
	}

	stage := &ypapi.TStage{
		Meta:   &ypapi.TStageMeta{},
		Spec:   &ypapi.TStageSpec{},
		Status: &ypapi.TStageStatus{},
	}

	if err := rsp.Fill(stage.Meta, stage.Spec, stage.Status); err != nil {
		return nil, fmt.Errorf("fill results: %w", err)
	}

	return stage, nil
}

func (y *Client) ProjectInfoFull(ctx context.Context, projectName string) (*ypapi.TProject, error) {
	rsp, err := y.GetProject(ctx, yp.GetProjectRequest{
		Format:    yp.PayloadFormatProto,
		ID:        projectName,
		Selectors: LightGetObjectsSelectors,
	})
	if err != nil {
		return nil, fmt.Errorf("get project: %w", err)
	}

	project := &ypapi.TProject{
		Meta: &ypapi.TProjectMeta{},
		Spec: &ypapi.TProjectSpec{},
	}

	if err := rsp.Fill(project.Meta, project.Spec); err != nil {
		return nil, fmt.Errorf("fill results: %w", err)
	}

	return project, nil
}

func (y *Client) ListStages(ctx context.Context) (int, error) {
	rsp, err := y.SelectStages(ctx, yp.SelectStagesRequest{
		Filter:            "",
		Format:            0,
		Selectors:         []string{"/meta/id"},
		ContinuationToken: "",
		Offset:            0,
		Limit:             0,
		Timestamp:         0,
		FetchRootObject:   false,
	})
	if err != nil {
		return 0, err
	}

	return rsp.Count(), nil
}

func (y *Client) getStageBatch(ctx context.Context, ct string) ([]*ypapi.TStage, string, error) {
	rv := make([]*ypapi.TStage, 0, DefaultGetObjectsBatchSize)
	req := yp.SelectStagesRequest{
		Format:    yp.PayloadFormatProto,
		Selectors: DefaultGetObjectsSelectors,
		Limit:     DefaultGetObjectsBatchSize,
	}
	if ct != "" {
		req.ContinuationToken = ct
	}
	rsp, err := y.SelectStages(ctx, req)
	if err != nil {
		return nil, ct, err
	}
	for rsp.Next() {
		t := &ypapi.TStage{
			Meta:   &ypapi.TStageMeta{},
			Spec:   &ypapi.TStageSpec{},
			Status: &ypapi.TStageStatus{},
		}
		err := rsp.Fill(t.Meta, t.Spec, t.Status)
		if err != nil {
			return nil, ct, err
		}
		rv = append(rv, t)
	}
	ct = rsp.ContinuationToken()
	return rv, ct, nil
}

func (y *Client) getProjectBatch(ctx context.Context, ct string) ([]*ypapi.TProject, string, error) {
	rv := make([]*ypapi.TProject, 0, DefaultGetObjectsBatchSize)
	req := yp.SelectProjectsRequest{
		Format:    yp.PayloadFormatProto,
		Selectors: DefaultGetObjectsSelectors,
		Limit:     DefaultGetObjectsBatchSize,
	}
	if ct != "" {
		req.ContinuationToken = ct
	}
	rsp, err := y.SelectProjects(ctx, req)
	if err != nil {
		return nil, ct, err
	}
	for rsp.Next() {
		t := &ypapi.TProject{
			Meta:   &ypapi.TProjectMeta{},
			Spec:   &ypapi.TProjectSpec{},
			Status: &ypapi.TProjectStatus{},
		}
		err := rsp.Fill(t.Meta, t.Spec, t.Status)
		if err != nil {
			return nil, ct, err
		}
		rv = append(rv, t)
	}
	ct = rsp.ContinuationToken()
	return rv, ct, nil
}

func (y *Client) FetchAllStages(ctx context.Context) []*ypapi.TStage {
	allStages := make([]*ypapi.TStage, 0, DefaultGetObjectsBatchSize)

	token := ""
	for {
		stages, newToken, err := y.getStageBatch(ctx, token)
		if err != nil {
			panic(fmt.Errorf("failed getting stages: %w", err))
		}
		token = newToken
		allStages = append(allStages, stages...)
		if len(stages) < DefaultGetObjectsBatchSize {
			break
		}
	}
	return allStages
}

func (y *Client) FetchAllStagesByProject(ctx context.Context, projectID string) ([]*ypapi.TStage, error) {
	rv := make([]*ypapi.TStage, 0, DefaultGetObjectsBatchSize)
	req := yp.SelectStagesRequest{
		Filter:    fmt.Sprintf("[/meta/project_id]='%s'", projectID),
		Format:    yp.PayloadFormatProto,
		Selectors: DefaultGetObjectsSelectors,
		Limit:     DefaultGetObjectsBatchSize,
	}

	rsp, err := y.SelectStages(ctx, req)
	if err != nil {
		return nil, err
	}
	for rsp.Next() {
		t := &ypapi.TStage{
			Meta:   &ypapi.TStageMeta{},
			Spec:   &ypapi.TStageSpec{},
			Status: &ypapi.TStageStatus{},
		}
		err := rsp.Fill(t.Meta, t.Spec, t.Status)
		if err != nil {
			return nil, err
		}
		rv = append(rv, t)
	}
	return rv, nil
}

func (y *Client) FetchAllProjects(ctx context.Context) []*ypapi.TProject {
	allProjects := make([]*ypapi.TProject, 0, DefaultGetObjectsBatchSize)

	token := ""
	for {
		projects, newToken, err := y.getProjectBatch(ctx, token)
		if err != nil {
			panic(fmt.Errorf("failed getting projects: %w", err))
		}
		token = newToken
		allProjects = append(allProjects, projects...)
		if len(projects) < DefaultGetObjectsBatchSize {
			break
		}
	}
	return allProjects
}
