"""
gevent-friendly thrift transport

Motivation: SWAT-2585

TFramedTransport, may create frames of arbitrary size.
On the client side it may take too much time to process single frame
so gevent may be hung during the process.
"""
from __future__ import unicode_literals

from cStringIO import StringIO

from thrift.transport.TTransport import TTransportBase, CReadableTransport


class TGeventIdleTransport(TTransportBase, CReadableTransport):
    """
    Switch context at least after each given :param idle_period: bytes read.
    It may not work in case of read(sz) where sz is greater than :param idle_period:.

    Idling on writing is not supported.
    """

    def __init__(self, trans, idle_byte_period, idle_method):
        """
        Will call :param idle_method: after reading at least :param idle_byte_period: bytes.

        :type idle_byte_period: int
        :type idle_method: collections.Callable
        """
        self.__trans = trans
        self.__idle_period = idle_byte_period
        self.__idle_method = idle_method
        self.__remainder = idle_byte_period
        self.__rbuf = StringIO('')

    def isOpen(self):
        return self.__trans.isOpen()

    def open(self):
        return self.__trans.open()

    def close(self):
        return self.__trans.close()

    def write(self, buf):
        self.__trans.write(buf)

    def flush(self):
        self.__trans.flush()

    def _refill_buffer(self, prefix, reqlen):
        # Prefix is used to support CReadableTransport interface
        if self.__remainder <= 0:
            self.__idle_method()
            self.__remainder = self.__idle_period
        suffix = self.__trans.read(max(reqlen, self.__remainder))
        self.__remainder -= len(suffix)
        self.__rbuf = StringIO(prefix + suffix)

    def read(self, sz):
        res = self.__rbuf.read(sz)
        remains = sz - len(res)
        if remains > 0:
            self._refill_buffer(res, remains)
            res = self.__rbuf.read(sz)
        return res

    # Implement the CReadableTransport interface.
    @property
    def cstringio_buf(self):
        return self.__rbuf

    def cstringio_refill(self, prefix, reqlen):
        # self.__rbuf will already be empty here because fastbinary doesn't
        # ask for a refill until the previous buffer is empty.  Therefore,
        # we can start reading immediately
        self._refill_buffer(prefix, reqlen)
        return self.__rbuf
