import time
import logging
from collections import defaultdict
from flask import Flask

from sepelib.core import config as global_config
from gevent import pywsgi, sleep, spawn_later

from infra.deploy_export_stats.src.libs.metrics import MetricsExt, ROOT_REGISTRY
from infra.deploy_export_stats.src.reporters import base, yp_allocated, qloud_allocated, iss_allocated
from infra.swatlib.webserver import WebServer, WSGIHandler

log = logging.getLogger('app')


class ReporterManager(object):
    @staticmethod
    def determine_first_idle_seconds(all_hosts, current_host, reporter_idle_seconds):
        if not all_hosts or current_host not in all_hosts:
            raise ValueError('Wrong value for all_hosts and current_host: {}, {}'.format(all_hosts, current_host))
        _all_hosts = sorted(all_hosts)
        return (reporter_idle_seconds / len(_all_hosts)) * (_all_hosts.index(current_host) + 1)

    def __init__(self, reporters, config, registry=None):  # type: (list[base.BaseReporter], dict, ...) -> None
        self._reporters = reporters or []
        self._registry = registry or ROOT_REGISTRY
        self._wake_up_every_seconds = config.get('wake_up_every_seconds', 30 * 60)
        self._last_reporter_run = defaultdict(lambda: 0)
        self._reporter_idle_seconds = config.get('reporter_idle_seconds', 24 * 60 * 60)
        self.first_run_idle_seconds = self.determine_first_idle_seconds(
            config.get('all_hosts'), config.get('current_host'), self._reporter_idle_seconds)

    def run_immediately(self):
        for reporter in self._reporters:
            try:
                reporter.run(int(time.time()), initial=True)
                log.info("Initial run {} success.".format(reporter.__class__.__name__))
            except Exception as e:
                log.exception(e)
                log.warning('Initial run {} failed'.format(reporter.__class__.__name__))

    def run_forever(self):
        while True:
            now = int(time.time())
            for reporter in self._reporters:
                if self._last_reporter_run[reporter] + self._reporter_idle_seconds < now:
                    try:
                        reporter.run(now)
                        log.warning('Success run {}'.format(reporter.__class__.__name__))
                    except Exception as e:
                        log.exception(e)
                        log.warning('Failed run {}'.format(reporter.__class__.__name__))

                    self._last_reporter_run[reporter] = now
            log.info("All Done... wake up after {} seconds".format(self._wake_up_every_seconds))
            sleep(self._wake_up_every_seconds)


def create_app(registry=None):
    app = Flask(__name__)

    @app.route('/ping')
    def ping():
        return 'pong'

    MetricsExt(app, registry=registry, performance_enabled=True, status_code_enabled=False)

    return app


class Application(object):
    name = 'exports-stats'

    def __init__(self, instance_id, yp_token=None, nanny_token=None, qloud_token=None):
        self.instance_id = instance_id
        app = create_app()
        http_config = global_config.get_value('web.http')
        listener = WebServer._create_server_socket(http_config.get('host'), http_config.get('port'),
                                                   backlog=http_config.get('backlog'))
        kwargs = {
            'application': app,
            'handler_class': WSGIHandler,
        }
        self.wsgi = pywsgi.WSGIServer(listener, **kwargs)
        self._reporters = []

        if qloud_token:
            self._reporters.append(qloud_allocated.QloudAllocatedReporter(
                token=qloud_token,
                config=global_config.get_value('qloud_allocated_reporter'),
                registry=ROOT_REGISTRY
            ))
        yp_reporter = None
        if yp_token:
            yp_reporter = yp_allocated.YpAllocatedReporter(
                    token=yp_token,
                    config=global_config.get_value('yp_allocated_reporter'),
                    registry=ROOT_REGISTRY,
                )
            self._reporters.append(yp_reporter)

        if nanny_token and yp_reporter:
            self._reporters.append(
                iss_allocated.IssAllocatedReporter(
                    token=nanny_token,
                    config=global_config.get_value('iss_allocated_reporter'),
                    registry=ROOT_REGISTRY,
                    yp_reporter=yp_reporter,
                ))

        self.reporter_manager = ReporterManager(self._reporters, global_config.get_value('reporters_manager'))

    def run(self):
        self.reporter_manager.run_immediately()

        log.info('Reporters Manager spawn later, after {} seconds'.format(self.reporter_manager.first_run_idle_seconds))
        spawn_later(self.reporter_manager.first_run_idle_seconds, self.reporter_manager.run_forever)
        self.wsgi.serve_forever()

    def stop(self):
        self.wsgi.stop()
