import ssl
import time
import logging
from itertools import chain
from collections import defaultdict

import ujson
import requests
from gevent.pool import Pool as GeventPool
from infra.swatlib.gevent import geventutil

from infra.deploy_export_stats.src.reporters.base import BaseReporter
from infra.deploy_export_stats.src.libs import metrics

log = logging.getLogger('qloud')

ssl._create_default_https_context = ssl._create_unverified_context


class QloudObjectsStats(object):
    def __init__(self):
        self.projects = 0
        self.applications = 0
        self.environments = 0

    def export(self, registry):
        registry.get_gauge('projects').set(self.projects)
        registry.get_gauge('applications').set(self.applications)
        registry.get_gauge('environments').set(self.environments)

    def __str__(self):
        return "projects:{self.projects};" \
               " applications:{self.applications};" \
               " environments:{self.environments};".format(self=self)


class QloudResourcesStats(object):
    def __init__(self):
        self.instances = 0
        self.cpu = 0.
        self.memory_gb = 0.
        self.ssd_gb = 0.
        self.hdd_gb = 0.

    def join(self, other):
        self.instances += other.instances
        self.cpu += other.cpu
        self.memory_gb += other.memory_gb
        self.ssd_gb += other.ssd_gb
        self.hdd_gb += other.hdd_gb

    @property
    def ssd_tb(self):
        return int(self.ssd_gb / 1024)

    @property
    def hdd_tb(self):
        return int(self.hdd_gb / 1024)

    def export(self, registry):
        registry.get_gauge('cpu-cores').set(int(self.cpu))
        registry.get_gauge('memory-gb').set(int(self.memory_gb))
        registry.get_gauge('pods-count').set(self.instances)

        registry.get_gauge('ssd-tb').set(self.ssd_tb)
        registry.get_gauge('hdd-tb').set(self.hdd_tb)

    def __str__(self):
        return "instances:{self.instances};" \
               " cpu:{self.cpu}c;" \
               " memory:{self.memory_gb}Gb;" \
               " hdd:{self.hdd_tb}Tb;" \
               " ssd:{self.ssd_tb}Tb;".format(self=self)


class QloudAllocatedReporter(BaseReporter):
    INTERNAL_URL = 'https://qloud.yandex-team.ru'
    EXTERNAL_URL = 'https://qloud-ext.yandex-team.ru'

    PROJECTS_RESOURCE_URI = '/api/v1/project/'
    PROJECT_TREE_URI = '/api/project-tree/'
    PROJECT_QUOTA_URL_TPL = '/api/v1/project/{project}/quota'
    ENVIRONMENT_INFO_URI_TPL = '/api/v1/environment/stable/{environment}'

    ALLOCATED_STATS_URL = '/api/stats/allocated'
    HOSTS_INFO_URL = '/api/v1/hosts/search'

    REQUESTS_POOL_SIZE = 2

    def __init__(self, token, config=None, registry=None):  # type: (str, dict, metrics.Registry) -> None
        self._token = token
        self._config = config or {}
        self._qloud_internal_url = self._config.get('internal_url', self.INTERNAL_URL)
        self._qloud_external_url = self._config.get('external_url', self.EXTERNAL_URL)

        self._requests_pool_size = self._config.get('requests_pool_size', self.REQUESTS_POOL_SIZE)
        self._registry = registry or metrics.ROOT_REGISTRY
        self._reporter_registry = self._registry.path('reporters', self.__class__.__name__)

    def update_resources_stats(self, qloud_url, project, stats_by_locations):
        # type: (str, dict, dict[str, QloudResourcesStats]) -> None
        session = requests.Session()
        session.headers['Authorization'] = 'OAuth {}'.format(self._token)
        quotas_url = qloud_url + self.PROJECT_QUOTA_URL_TPL.format(
            project=project['projectName'],
        )
        resp = session.get(quotas_url)

        project_quota_info = ujson.loads(resp.content).get('usedQuota')

        # u'SSD_GB', u'DISC_GB', u'MEMORY_GB', u'CPU'

        for service_name, resources in project_quota_info.items():
            for resource in resources:
                if resource['resource'] == 'CPU':
                    stats_by_locations[resource['location']].cpu += resource['quota']
                elif resource['resource'] == 'MEMORY_GB':
                    stats_by_locations[resource['location']].memory_gb += resource['quota']
                elif resource['resource'] == 'DISC_GB':
                    stats_by_locations[resource['location']].hdd_gb += resource['quota']
                elif resource['resource'] == 'SSD_GB':
                    stats_by_locations[resource['location']].ssd_gb += resource['quota']

    def get_stats_by_projects_qouta(self, qloud_url):
        # type: (str) -> dict[QloudResourcesStats]
        session = requests.Session()
        session.headers['Authorization'] = 'OAuth {}'.format(self._token)

        resources_stats_by_locations = defaultdict(QloudResourcesStats)
        pool = GeventPool(size=self._requests_pool_size)

        projects_url = qloud_url + self.PROJECTS_RESOURCE_URI
        resp = session.get(projects_url)
        resp.raise_for_status()

        projects = ujson.loads(resp.content)

        for project in geventutil.gevent_idle_iter(projects):
            pool.apply(self.update_resources_stats, kwds=dict(
                qloud_url=qloud_url,
                project=project,
                stats_by_locations=resources_stats_by_locations
            ))

        pool.join()

        return resources_stats_by_locations

    def get_stats_by_allocated(self, qloud_url):
        session = requests.Session()
        session.headers['Authorization'] = 'OAuth {}'.format(self._token)

        resp = session.get(qloud_url + self.ALLOCATED_STATS_URL)
        resp.raise_for_status()

        content = resp.content.splitlines()
        if not content:
            raise ValueError('No Allocated info returns from server')

        relative_index = dict(zip(content[0].strip(' \n').split('\t'), list(range(0, 16))))

        stat = QloudResourcesStats()

        for i in geventutil.gevent_idle_iter(range(1, len(content))):
            row = content[i].strip('\n').split('\t')

            stat.cpu += float(row[relative_index["cpu_limit"]])
            stat.memory_gb += float(row[relative_index["memory_limit"]]) / 1024 ** 3

        return stat

    def get_objects_stats(self, qloud_url):
        session = requests.Session()
        session.headers['Authorization'] = 'OAuth {}'.format(self._token)
        project_tree_url = qloud_url + self.PROJECT_TREE_URI

        objects_stats = QloudObjectsStats()

        resp = session.get(project_tree_url)
        resp.raise_for_status()

        project_tree = ujson.loads(resp.content).get('projects', [])

        for project_info in geventutil.gevent_idle_iter(project_tree):
            objects_stats.projects += 1
            objects_stats.applications += len(project_info['applications'])
            objects_stats.environments += len(list(chain(*[a['environments'] for a in project_info['applications']])))

        return objects_stats

    def get_stats_by_hosts(self, qloud_url):  # type: (str) -> dict[str, dict[str, QloudResourcesStats]]
        session = requests.Session()
        session.headers['Authorization'] = 'OAuth {}'.format(self._token)
        hosts_url = qloud_url + self.HOSTS_INFO_URL
        resp = session.get(qloud_url + self.HOSTS_INFO_URL)
        resp.raise_for_status()

        hosts = ujson.loads(resp.content)
        if not hosts:
            raise ValueError('{} return empty response'.format(hosts_url))

        resources_stats = defaultdict(lambda: defaultdict(QloudResourcesStats))

        for host in geventutil.gevent_idle_iter(hosts):
            location = host.get('dataCenter')
            if not location:
                log.warning('Host {}, has empty "dataCenter" property'.format(host.get('fqdn')))
                continue
            segment = host['segment']
            stats = resources_stats[location][segment]
            active_instances = [i['allocationId'].split('#')[0] for i in host['instances'] if i['status'] == 'active']
            mount_points_types = {d['allocationMountPoint'][0]: d['type']
                                  for d in host['disks'] if d['allocationMountPoint']}

            for slot in host['slots']:
                component = slot['componentId'].rsplit('.', 1)[0]
                if component not in active_instances:
                    continue
                stats.instances += 1
                stats.cpu += slot['cores']
                stats.memory_gb += slot['memoryBytes'] / 1024 ** 3
                disc_type = mount_points_types.get(slot.get('mount', ""))
                if disc_type == 'SSD':
                    stats.ssd_gb += slot['diskCapacityBytes'] / 1024 ** 3
                elif disc_type == 'HDD':
                    stats.hdd_gb += slot['diskCapacityBytes'] / 1024 ** 3

        global_all_stats = QloudResourcesStats()

        for location, stats_by_segments in resources_stats.items():
            all_stats = QloudResourcesStats()

            for segment, stats in stats_by_segments.items():
                global_all_stats.join(stats)
                all_stats.join(stats)

            stats_by_segments['all'] = all_stats

        resources_stats['GLOBAL']['all'] = global_all_stats

        return resources_stats

    def run(self, start_at, initial=False):
        qloud_urls_map = {
            'qloud-int': self._qloud_internal_url,
            'qloud-ext': self._qloud_external_url
        }
        self._reporter_registry.get_gauge('last-run-start').set(start_at)

        for deploy_engine, gloud_url in qloud_urls_map.items():
            allocation_stats = self.get_stats_by_allocated(gloud_url)

            log.info('------------------------------------------')
            log.info('---- {}-{}-{}: {}'.format('all', deploy_engine, 'all', allocation_stats))

            allocated_registry = self._registry.path('allocated', 'iss')

            with self._reporter_registry.get_histogram('fetch-hosts-info').timer():
                stats_by_locations_and_segments = self.get_stats_by_hosts(gloud_url)

            for location, stats_by_segment in stats_by_locations_and_segments.items():
                for segment, stats in stats_by_segment.items():
                    if segment == 'all':
                        stats.export(allocated_registry.path(location.lower(), deploy_engine, segment))
                    log.info('By Hosts {}-{}-{}: {}'.format(location.lower(), deploy_engine, segment, stats))

            objects_stats = self.get_objects_stats(gloud_url)
            objects_stats.export(allocated_registry.path('global', deploy_engine, 'all'))
            log.info('Objects {}-{}-{}: {}'.format('global', deploy_engine, 'all', objects_stats))

        self._reporter_registry.get_gauge('last-run-done').set(int(time.time()))


if __name__ == "__main__":
    import sys
    import os

    handler = logging.StreamHandler(sys.stdout)
    handler.setLevel(logging.DEBUG)

    log.setLevel(logging.DEBUG)
    log.addHandler(handler)

    qloud_token = os.environ['QLOUD_TOKEN']
    reporter = QloudAllocatedReporter(token=qloud_token)
    reporter.run(time.time())
