import os
import time
import random
import argparse

from ujson import dumps


WORDS = "Sed ut perspiciatis, unde omnis iste natus error sit voluptatem accusantium doloremque laudantium, totam rem aperiam eaque ipsa, quae ab illo inventore veritatis et quasi architecto beatae vitae dicta sunt, explicabo. Nemo enim ipsam voluptatem, quia voluptas sit, aspernatur aut odit aut fugit, sed quia consequuntur magni dolores eos, qui ratione voluptatem sequi nesciunt, neque porro quisquam est, qui dolorem ipsum, quia dolor sit, amet, consectetur, adipisci velit, sed quia non numquam eius modi tempora incidunt, ut labore et dolore magnam aliquam quaerat voluptatem. Ut enim ad minima veniam, quis nostrum exercitationem ullam corporis suscipit laboriosam, nisi ut aliquid ex ea commodi consequatur? Quis autem vel eum iure reprehenderit, qui in ea voluptate velit esse, quam nihil molestiae consequatur, vel illum, qui dolorem eum fugiat, quo voluptas nulla pariatur? At vero eos et accusamus et iusto odio dignissimos ducimus, qui blanditiis praesentium voluptatum deleniti atque corrupti, quos dolores et quas molestias excepturi sint, obcaecati cupiditate non provident, similique sunt in culpa, qui officia deserunt mollitia animi, id est laborum et dolorum fuga. Et harum quidem rerum facilis est et expedita distinctio. Nam libero tempore, cum soluta nobis est eligendi optio, cumque nihil impedit, quo minus id, quod maxime placeat, facere possimus, omnis voluptas assumenda est, omnis dolor repellendus. Temporibus autem quibusdam et aut officiis debitis aut rerum necessitatibus saepe eveniet, ut et voluptates repudiandae sint et molestiae non recusandae. Itaque earum rerum hic tenetur a sapiente delectus, ut aut reiciendis voluptatibus maiores alias consequatur aut perferendis doloribus asperiores repellat.".split(" ")  # noqa

LEVEL_INTS = [1000, 2000, 3000, 4000, 5000]
LEVELS = "ERROR WARNING INFO DEBUG".split(" ")
STATUSES = "OK FAIL DUNNO".split(" ")
THREADS = "main non-main".split(" ")
LOGGERS = "root logs logs.generator logs.generator.third logs.generator.another logs.something supportlibrary supportlibrary.mess other".split(" ")
COUNTER = 0


def make_sentence(min_length, max_length, char=' '):
    length = random.randint(min_length, max_length + 1)
    return char.join(random.choices(WORDS, k=length))


def make_message(validating_marker=None):
    global COUNTER
    msg = {
        "msg": make_sentence(3, 10),
        "stackTrace": "\n".join(make_sentence(3, 6) for _ in range(random.randint(2, 10))),
        "levelStr": random.choice(LEVELS),
        "level": random.choice(LEVEL_INTS),
        "loggerName": random.choice(LOGGERS),
        "threadName": random.choice(THREADS),
        "@fields": {
            "request_id": make_sentence(3, 3, '-'),
            "uri": make_sentence(3, 5, '/'),
            "status": {
                "code": random.randint(100, 600),
                "message": random.choice(STATUSES),
            },
        }
    }

    if validating_marker:
        COUNTER += 1
        msg['@fields']['validating'] = validating_marker
        msg['@fields']['counter'] = COUNTER

    return msg


def write_marker(path, marker, records, start_timestamp, end_timestamp):
    with open(path + '.new', 'w') as f:
        f.write(dumps({
            "marker": marker,
            "records": records,
            "start_timestamp": start_timestamp,
            "end_timestamp": end_timestamp,
        }))
        f.flush()
        os.fsync(f.fileno())
    os.rename(path + '.new', path)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--daily-gb', type=int, default=1)
    parser.add_argument('--max-batch', type=int, default=20)
    parser.add_argument('--validating', action='store_true', default=False)
    parser.add_argument('--marker-file', default='validating-marker.txt')
    parser.add_argument('--total-bytes', type=int, default=0)
    args = parser.parse_args()

    bps = args.daily_gb * 1024 * 1024 * 1024 / (24 * 60 * 60)

    validating_marker = f"{random.getrandbits(128):x}" if args.validating else None
    validating_records = 0
    start_timestamp = time.time()

    total_written = 0
    while True:
        started = time.time()
        written = 0
        for _ in range(random.randint(args.max_batch // 2, args.max_batch)):
            message = dumps(make_message(validating_marker=validating_marker))
            try:
                print(message)
            except BrokenPipeError:
                raise SystemExit()

            written += len(message)
            total_written += len(message)
            validating_records += 1

            if args.total_bytes and total_written > args.total_bytes:
                if validating_marker:
                    end_timestamp = time.time()
                    write_marker(args.marker_file, validating_marker, validating_records, start_timestamp, end_timestamp)
                return

        elapsed = time.time() - started

        to_sleep = written / bps - elapsed
        if to_sleep > 0:
            time.sleep(to_sleep)
