#include "unistat_manager_base.h"

#include <infra/libs/sensors/macros.h>

#include <infra/deploy_monitoring_controller/libs/unistat/unistat.h>

#include <util/generic/ymath.h>
#include <util/string/builder.h>

namespace NInfra::NDeployMonitoringController {

namespace {

const TVector<ui64> BUCKETS = {
    0
    , 5
    , 10
    , 20
    , 30
    , 40
    , 60
    , 80
    , 100
    , 120
    , 140
    , 160
    , 180
    , 200
    , 225
    , 250
    , 275
    , 300
    , 350
    , 400
    , 450
    , 500
    , 600
    , 700
    , 800
    , 900
    , 1000
    , 1200
    , 1400
    , 1600
    , 1800
    , 2000
    , 2400
    , 2800
    , 3200
    , 3600
    , 4000
    , 5000
    , 6000
    , 7000
    , 8000
    , 9000
    , 10000
    , 15000
    , 20000
    , 50000
    , 100000
};

struct TTimeInterval {
    TString Name_;
    TDuration Duration_;
};

const TVector<TTimeInterval> TIME_INTERVALS = {
    {"all", TDuration::Max()}
    , {"month", TDuration::Days(30)}
    , {"week", TDuration::Days(7)}
    , {"day", TDuration::Days(1)}
    , {"hour", TDuration::Hours(1)}
};

} // namespace

TString TUnistatManager::GetObjectId() const {
    return ToString(AtomicGet(Id_));
}

ui64 TUnistatManager::TObject::GetLag(const TInstant& begin, const TInstant& end) {
    return (end - begin).Seconds();
}

bool TUnistatManager::TObject::IsReady() const {
    return SpecRevision_ == StatusRevision_ && Ready_;
}

void TUnistatManager::GenerateYpUpdates(
    const ISingleClusterObjectManager::TDependentObjects& /* dependentObjects */
    , TVector<ISingleClusterObjectManager::TRequest>& /* requests */
    , TLogFramePtr frame
) const {
    const TString signalTagsAndPrefix = TStringBuilder()
        << "ctype=" << UnistatConfig_.GetCType()
        << ";geo=" << UnistatConfig_.GetGeo()
        << ";prj=" << UnistatConfig_.GetProject()
        << ";" << UnistatConfig_.GetSignalPrefix() << "."
    ;

    TVector<TVector<ui64>> readyLag(TIME_INTERVALS.size());
    TVector<TVector<ui64>> readyLagWithCorrectionFactor(TIME_INTERVALS.size());
    NLogEvent::TDeployObjectLag readyLagEv;
    readyLagEv.SetObjectType(ObjectType_);
    readyLagEv.SetLagType(NLogEvent::TDeployObjectLag_EDeployLagType_LT_READY);

    TVector<TVector<ui64>> notReadyLag(TIME_INTERVALS.size());
    TVector<TVector<ui64>> notReadyLagWithCorrectionFactor(TIME_INTERVALS.size());
    NLogEvent::TDeployObjectLag notReadyLagEv;
    notReadyLagEv.SetObjectType(ObjectType_);
    notReadyLagEv.SetLagType(NLogEvent::TDeployObjectLag_EDeployLagType_LT_NOT_READY);

    const TVector<IObjectStorage::TObject> storageInfo = ObjectStorage_->GetObjectList();
    TMap<TString, const IObjectStorage::TObject&> storageInfoMap;
    for (const auto& object : storageInfo) {
        storageInfoMap.insert({object.Id_, object});
    }
    TVector<IObjectStorage::TObject> newStorageInfo;

    static auto addLag = [](
        const TString& objectId
        , const TInstant& specTimestamp
        , ui64 currentLag
        , TVector<TVector<ui64>>& lagData
        , TVector<TVector<ui64>>& lagDataWithCorrectionFactor
        , NLogEvent::TDeployObjectLag& event
        , const double correctionFactor
        , const TInstant& currentTime
    ) {
        ui64 currentLagWithCorrectionFactor = round(currentLag * correctionFactor);

        NLogEvent::TDeployObjectLag::TDeployObject* objectEv = event.MutableObjects()->Add();
        objectEv->SetId(objectId);
        objectEv->SetLag(currentLag);
        objectEv->SetCorrectionFactor(correctionFactor);

        for (size_t i = 0; i < TIME_INTERVALS.size(); ++i) {
            if (currentTime - specTimestamp < TIME_INTERVALS[i].Duration_) {
                lagData[i].push_back(currentLag);
                lagDataWithCorrectionFactor[i].push_back(currentLagWithCorrectionFactor);
            }
        }
    };

    TInstant currentTime = TInstant::Now();
    for (const auto& object : Objects_) {
        auto ptr = storageInfoMap.FindPtr(object.Id_);

        TInstant currentRevisionSpecTimestamp = (ptr && object.SpecRevision_ == ptr->CurrentRevision_)
            // Sometimes spec timestamp can change without changing the revision
            // in this case the change is not applied to objects lower in the hierarchy and no processes should occur
            // so if the revision has not changed, we consider the timestamp from controller state to be correct
            ? ptr->CurrentRevisionSpecTimestamp_
            : object.SpecTimestamp_
        ;

        if (ptr && ptr->LastReadyRevision_ == object.SpecRevision_) {
            // If object was ready and revision not changed it stays ready
            newStorageInfo.emplace_back(
                ptr->Id_
                , ptr->LastReadyRevision_
                , ptr->LastReadyAchivedTimestamp_
                , object.SpecRevision_
                , currentRevisionSpecTimestamp
            );
            addLag(
                object.Id_
                , currentRevisionSpecTimestamp
                , TObject::GetLag(currentRevisionSpecTimestamp, ptr->LastReadyAchivedTimestamp_)
                , readyLag
                , readyLagWithCorrectionFactor
                , readyLagEv
                , object.LagCorrectionFactor_
                , currentTime
            );
        } else {
            if (object.IsReady()) {
                newStorageInfo.emplace_back(
                    object.Id_
                    , object.SpecRevision_
                    , object.ReadyAchivedTimestamp_
                    , object.SpecRevision_
                    , currentRevisionSpecTimestamp
                );
                addLag(
                    object.Id_
                    , currentRevisionSpecTimestamp
                    , TObject::GetLag(currentRevisionSpecTimestamp, object.ReadyAchivedTimestamp_)
                    , readyLag
                    , readyLagWithCorrectionFactor
                    , readyLagEv
                    , object.LagCorrectionFactor_
                    , currentTime
                );
            } else {
                newStorageInfo.emplace_back(
                    object.Id_
                    , ptr ? ptr->LastReadyRevision_ : 0
                    , ptr ? ptr->LastReadyAchivedTimestamp_ : TInstant::Zero()
                    , object.SpecRevision_
                    , currentRevisionSpecTimestamp
                );
                addLag(
                    object.Id_
                    , currentRevisionSpecTimestamp
                    , TObject::GetLag(currentRevisionSpecTimestamp, currentTime)
                    , notReadyLag
                    , notReadyLagWithCorrectionFactor
                    , notReadyLagEv
                    , object.LagCorrectionFactor_
                    , currentTime
                );
            }
        }
    }

    frame->LogEvent(
        ELogPriority::TLOG_DEBUG
        , readyLagEv
    );
    frame->LogEvent(
        ELogPriority::TLOG_DEBUG
        , notReadyLagEv
    );

    for (size_t i = 0; i < TIME_INTERVALS.size(); ++i) {
        const TString readySignal = TStringBuilder()
            << signalTagsAndPrefix
            << "ready." << TIME_INTERVALS[i].Name_ << "." << LagSignal_ << "_ahhh"
        ;
        const TString notReadySignal = TStringBuilder()
            << signalTagsAndPrefix
            << "not_ready." << TIME_INTERVALS[i].Name_ << "." << LagSignal_ << "_ahhh"
        ;

        const TString readySignalWithCorrectionFactor = TStringBuilder()
            << signalTagsAndPrefix
            << "ready.with_correction_factor." << TIME_INTERVALS[i].Name_ << "." << LagSignal_ << "_ahhh"
        ;
        const TString notReadySignalWithCorrectionFactor = TStringBuilder()
            << signalTagsAndPrefix
            << "not_ready.with_correction_factor." << TIME_INTERVALS[i].Name_ << "." << LagSignal_ << "_ahhh"
        ;

        if (!AmILeader()) {
            // Clear the signals to avoid duplicates
            THistogramUnistat::Instance().RemoveHistogram(readySignal);
            THistogramUnistat::Instance().RemoveHistogram(notReadySignal);

            THistogramUnistat::Instance().RemoveHistogram(readySignalWithCorrectionFactor);
            THistogramUnistat::Instance().RemoveHistogram(notReadySignalWithCorrectionFactor);
        } else {
            THistogramUnistat::Instance().AddOrUpdateHistogram(
                readySignal
                , readyLag[i]
                , BUCKETS
            );
            THistogramUnistat::Instance().AddOrUpdateHistogram(
                notReadySignal
                , notReadyLag[i]
                , BUCKETS
            );

            THistogramUnistat::Instance().AddOrUpdateHistogram(
                readySignalWithCorrectionFactor
                , readyLagWithCorrectionFactor[i]
                , BUCKETS
            );
            THistogramUnistat::Instance().AddOrUpdateHistogram(
                notReadySignalWithCorrectionFactor
                , notReadyLagWithCorrectionFactor[i]
                , BUCKETS
            );
        }
    }

    if (AmILeader()) {
        // Update storage
        ObjectStorage_->UpdateObjectList(newStorageInfo);
    }
}

TVector<NController::ISingleClusterObjectManager::TSelectArgument> TUnistatManagerFactoryBase::GetSelectArguments(const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */, NInfra::TLogFramePtr) const {
    return {
        GetSelectArgument()
    };
}

TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TUnistatManagerFactoryBase::TValidationError>> TUnistatManagerFactoryBase::GetSingleClusterObjectManagers(
    const TVector<NController::TSelectObjectsResultPtr>& selectorResults
    , TLogFramePtr /* frame */
) const {
    TVector<TUnistatManager::TObject> objects;
    objects.reserve(selectorResults[0]->Results.size()); // There will be at least as many objects

    for (const auto& selectorResultPtr : selectorResults[0]->Results) {
        for (const TUnistatManager::TObject& object : GetObjects(selectorResultPtr)) {
            objects.push_back(object);
        }
    }

    return TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TUnistatManagerFactoryBase::TValidationError>>(
        {
            NController::TSingleClusterObjectManagerPtr(new TUnistatManager(objects, ObjectType_, LagSignal_, UnistatConfig_, ObjectStorage_))
        }
    );
}

} // namespace NInfra::NDeployMonitoringController
