#pragma once

#include <infra/deploy_monitoring_controller/libs/config/config.pb.h>
#include <infra/deploy_monitoring_controller/libs/object_storage/yt_object_storage.h>

#include <infra/libs/controller/object_manager/object_manager.h>

namespace NInfra::NDeployMonitoringController {

class TUnistatManager : public NController::ISingleClusterObjectManager {
public:
    struct TObject {
        TObject(
            const TString& id
            , const ui64 specRevision
            , const TInstant& specTimestamp
            , const ui64 statusRevision
            , const bool ready
            , const TInstant& readyAchivedTimestamp
            , const double lagCorrectionFactor
        )
            : Id_(id)
            , SpecRevision_(specRevision)
            , SpecTimestamp_(specTimestamp)
            , StatusRevision_(statusRevision)
            , Ready_(ready)
            , ReadyAchivedTimestamp_(readyAchivedTimestamp)
            , LagCorrectionFactor_(lagCorrectionFactor)
        {}

        static ui64 GetLag(const TInstant& begin, const TInstant& end);

        bool IsReady() const;

        const TString Id_;

        const ui64 SpecRevision_;
        const TInstant SpecTimestamp_;

        const ui64 StatusRevision_;

        const bool Ready_;
        const TInstant ReadyAchivedTimestamp_;

        const double LagCorrectionFactor_;
    };

public:
    TUnistatManager(
        const TVector<TObject>& objects
        , const NLogEvent::TDeployObjectLag::EDeployObjectType objectType
        , const TString& lagSignal
        , const TUnistatConfig& unistatConfig
        , TObjectStoragePtr objectStorage
    )
        : Objects_(objects)
        , ObjectType_(objectType)
        , LagSignal_(lagSignal)
        , UnistatConfig_(unistatConfig)
        , ObjectStorage_(objectStorage)
    {
        AtomicIncrement(Id_);
    }

    virtual TString GetObjectId() const override final;

    virtual void GenerateYpUpdates(
        const ISingleClusterObjectManager::TDependentObjects& dependentObjects
        , TVector<ISingleClusterObjectManager::TRequest>& requests
        , TLogFramePtr frame
    ) const override final;

private:
    static inline TAtomic Id_ = 0;

    const TVector<TObject> Objects_;
    const NLogEvent::TDeployObjectLag::EDeployObjectType ObjectType_;
    const TString LagSignal_;
    const TUnistatConfig UnistatConfig_;
    TObjectStoragePtr ObjectStorage_;
};

class TUnistatManagerFactoryBase : public NController::ISingleClusterObjectManagersFactory {
public:
    TUnistatManagerFactoryBase(
        const TString& factoryName
        , const NLogEvent::TDeployObjectLag::EDeployObjectType objectType
        , const TString& lagSignal
        , const TUnistatConfig& unistatConfig
        , const TYtObjectStorageConfig& ytStorageConfig
        , NController::TShardPtr shard
    )
        : ISingleClusterObjectManagersFactory(factoryName, shard, true)
        , ObjectType_(objectType)
        , LagSignal_(lagSignal)
        , UnistatConfig_(unistatConfig)
        , ObjectStorage_(
            new TYtObjectStorage(
                ytStorageConfig
                , STORAGE_PREFIX + factoryName
            )
        )
    {}

    virtual TVector<NController::ISingleClusterObjectManager::TSelectArgument> GetSelectArguments(const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */ = {}, NInfra::TLogFramePtr = {}) const override final;

    virtual TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TValidationError>> GetSingleClusterObjectManagers(
        const TVector<NController::TSelectObjectsResultPtr>& selectorResults
        , TLogFramePtr frame
    ) const override final;

protected:
    virtual NController::ISingleClusterObjectManager::TSelectArgument GetSelectArgument(const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */ = {}, NInfra::TLogFramePtr = {}) const = 0;
    virtual TVector<TUnistatManager::TObject> GetObjects(const NController::TSelectorResultPtr& selectorResultPtr) const = 0;

private:
    static constexpr const char* STORAGE_PREFIX = "storage_";

    const NLogEvent::TDeployObjectLag::EDeployObjectType ObjectType_;
    const TString LagSignal_;
    const TUnistatConfig UnistatConfig_;

    TObjectStoragePtr ObjectStorage_;
};

} // namespace NInfra::NDeployMonitoringController
