#include "yt_object_storage.h"

#include <mapreduce/yt/util/wait_for_tablets_state.h>

#include <util/string/builder.h>

namespace NInfra::NDeployMonitoringController {

void TYtObjectStorage::EnsureTableMounted() {
    Client_->Create(
        TablePath_
        , NYT::NT_TABLE
        , NYT::TCreateOptions()
            .IgnoreExisting(true)
            .Recursive(true)
            .Attributes(
                NYT::TNode()
                    ("dynamic", true)
                    ("schema", SCHEMA.ToNode())
            )
    );

    Client_->MountTable(TablePath_);
    NYT::WaitForTabletsState(Client_, TablePath_, NYT::TS_MOUNTED);
}

TVector<TYtObjectStorage::TObject> TYtObjectStorage::GetObjectList() const {
    TVector<NYT::TNode> rows = Client_->SelectRows(TStringBuilder() << "* from [" << TablePath_ << "]");

    TVector<TObject> objects;
    for (const auto& row : rows) {
        objects.emplace_back(
            row.ChildAsString(ID_KEY)
            , row.ChildAsUint64(LAST_READY_REVISION_KEY)
            , TInstant::MicroSeconds(row.ChildAsUint64(LAST_READY_ACHIVED_TIMESTAMP_KEY))
            , row.ChildAsUint64(CURRENT_REVISION_KEY)
            , TInstant::MicroSeconds(row.ChildAsUint64(CURRENT_REVISION_SPEC_TIMESTAMP_KEY))
        );
    }

    return objects;

}

void TYtObjectStorage::UpdateObjectList(const TVector<TYtObjectStorage::TObject>& objects) {
    TVector<TYtObjectStorage::TObject> currentObjects = GetObjectList();

    TMap<TString, const TYtObjectStorage::TObject&> currentObjectsMap;
    for (const auto& object : currentObjects) {
        currentObjectsMap.insert({object.Id_, object});
    }

    TMap<TString, const TYtObjectStorage::TObject&> newObjectMap;
    for (const auto& object : objects) {
        newObjectMap.insert({object.Id_, object});
    }

    for (const auto& [objectId, objectInfo] : currentObjectsMap) {
        if (!newObjectMap.contains(objectId)) {
            RemoveObject(objectInfo);
        }
    }

    for (const auto& [objectId, objectInfo] : newObjectMap) {
        if (auto oldPtr = currentObjectsMap.FindPtr(objectId)) {
            if (*oldPtr != objectInfo) {
                // Override info
                InsertObject(objectInfo);
            }
        } else {
            // Insert new object
            InsertObject(objectInfo);
        }
    }
}

void TYtObjectStorage::InsertObject(const TObject& object) {
    Client_->InsertRows(TablePath_
        , {
            NYT::TNode()
                (ID_KEY, object.Id_)
                (LAST_READY_REVISION_KEY, object.LastReadyRevision_)
                (LAST_READY_ACHIVED_TIMESTAMP_KEY, object.LastReadyAchivedTimestamp_.MicroSeconds())
                (CURRENT_REVISION_KEY, object.CurrentRevision_)
                (CURRENT_REVISION_SPEC_TIMESTAMP_KEY, object.CurrentRevisionSpecTimestamp_.MicroSeconds())
        }
    );
}

void TYtObjectStorage::RemoveObject(const TObject& object) {
    Client_->DeleteRows(TablePath_
        , {
            NYT::TNode()
                (ID_KEY, object.Id_)
        }
    );
}

} // namespace NInfra::NDeployMonitoringController
