#include "unistat.h"

#include <util/generic/algorithm.h>
#include <util/generic/yexception.h>
#include <util/string/builder.h>

namespace NInfra::NDeployMonitoringController {

void THistogramUnistat::AddOrUpdateHistogram(
    const TString& signal
    , const TVector<ui64>& values
    , const TVector<ui64>& buckets
) {
    TGuard<TMutex> guard(Mutex_);

    Y_ENSURE(!buckets.empty(), "At least one bucket must be provided");
    Y_ENSURE(buckets.size() <= MAX_BUCKETS_SIZE, "Buckets size can't be bigger then " << MAX_BUCKETS_SIZE << ", actual size is " << buckets.size());
    Y_ENSURE(buckets[0] == 0, "First bucket must be zero, but actual value is " << buckets[0]);
    for (size_t i = 0; i < buckets.size() - 1; ++i) {
        Y_ENSURE(buckets[i] < buckets[i + 1]
            , "Buckets must be strictly sorted, but buckets[" << i << "] = " << buckets[i] << ", buckets[" << i + 1 << "] = " << buckets[i + 1]
        );
    }

    TVector<std::pair<ui64, ui64>>& histogram = Signals_[signal];

    histogram.resize(buckets.size());
    for (size_t i = 0; i < buckets.size(); ++i) {
        histogram[i] = {buckets[i], 0};
    }

    for (ui64 value : values) {
        // value >= 0, buckets[0] == 0
        size_t ptr = UpperBound(buckets.begin(), buckets.end(), value) - buckets.begin() - 1;
        ++histogram[ptr].second;
    }
}

void THistogramUnistat::RemoveHistogram(
    const TString& signal
) {
    TGuard<TMutex> guard(Mutex_);
    Signals_.erase(signal);
}

void THistogramUnistat::RemoveAllHistograms() {
    TGuard<TMutex> guard(Mutex_);
    Signals_.clear();
}

TString THistogramUnistat::DumpHistograms() const {
    TGuard<TMutex> guard(Mutex_);

    TStringBuilder ret;

    ret << "[";
    bool firstSignal = true;
    for (const auto& [signal, histogram] : Signals_) {
        if (firstSignal) {
            firstSignal = false;
        } else {
            ret << ",";
        }

        ret << "[\"" << signal << "\",[";
        bool firstInHistogram = true;
        for (const auto& [bucket, value] : histogram) {
            if (firstInHistogram) {
                firstInHistogram = false;
            } else {
                ret << ",";
            }
            ret << "[" << bucket << "," << value << "]";
        }
        ret << "]]";
    }
    ret << "]";

    return ret;
}

} // namespace NInfra::NDeployMonitoringController
